﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/core/ens_Task.h>
#include <nn/ens/detail/core/ens_HttpConnectionProvider.h>

namespace nn { namespace ens { namespace detail { namespace core {

/**
 * @brief   HTTP 通信を行うクラス
 *
 * @details
 *  HTTP 通信を行うタスクの共通処理を実装したクラスです。@n
 *  Main() 関数の実装はないため、本クラスを継承したクラスは Main() を実装する必要があります。
 */
class HttpTask : public Task
{
public:
    NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO(HttpTask, Task);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    HttpTask() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~HttpTask() NN_NOEXCEPT;

    /**
     * @brief   HTTP コネクションプロバイダーを設定します。
     *
     * @param[in]   pConnectionProvider HTTP コネクションプロバイダー
     *
     * @pre
     *  - pConnectionProvider != nullptr
     *
     * @details
     */
    void SetHttpConnectionProvider(HttpConnectionProvider* pConnectionProvider) NN_NOEXCEPT;

    /**
     * @brief   ネットワークサービスアカウントの ID トークンを設定します。
     *
     * @param[in]   pToken  ネットワークサービスアカウントの ID トークン
     *
     * @pre
     *  - pToken != nullptr
     *
     * @details
     */
    void SetNetworkServiceAccountIdToken(const char* pToken) NN_NOEXCEPT;

protected:
    /**
     * @brief   HTTP コネクションプロバイダーを取得します。
     *
     * @return  HTTP コネクションプロバイダー
     *
     * @details
     *  本クラスを継承するタスクは、始めに本関数で HTTP コネクションプロバイダーを取得します。@n
     *  プロバイダーから供給される HTTP コネクションオブジェクトに対して独自の設定を注入し、その後 Perform() を呼び出してください。
     */
    HttpConnectionProvider* GetHttpConnectionProvider() NN_NOEXCEPT;

    /**
     * @brief   ネットワークサービスアカウントの ID トークンを取得します。
     *
     * @return  ネットワークサービスアカウントの ID トークン
     *
     * @pre
     *  - 非同期コンテキストが登録されている
     *
     * @details
     *  本関数で取得できるのはアプリケーションから渡されたポインタの浅いコピーです。@n
     *  そのため、 IsAsyncContextRegistered() が false を返した場合、値にアクセスしてはいけません。
     */
    const char* GetNetworkServiceAccountIdToken() const NN_NOEXCEPT;

    /**
     * @brief   通信処理の共通オプションを設定します。
     *
     * @param[in]   pCurl   HTTP コネクション
     *
     * @return  処理結果
     *
     * @details
     */
    nn::Result SetHttpCommonOptions(CURL* pCurl) NN_NOEXCEPT;

    /**
     * @brief   通信処理を行います。
     *
     * @return  処理結果
     *
     * @details
     *  通信処理の完了後、HTTP コネクションからステータスコードを取得し、その値に従ってハンドリングを行います。
     *
     *  ステータスコードが 2xx の場合、成功が返ります。@n
     *  ステータスコードが 2xx 以外の場合、エラーレスポンスの解析を行い、該当するエラーが返ります。
     */
    nn::Result Perform() NN_NOEXCEPT;

    /**
     * @brief   送信バッファを取得します。
     *
     * @return  送信バッファ
     *
     * @details
     *  HTTP タスクのバッファは送受信の両方で使用します。@n
     *  そのため、 Perform() 呼び出し後はバッファを書き変えてはいけません。
     */
    void* GetRequestBuffer() NN_NOEXCEPT;

    /**
     * @brief   送信バッファサイズを取得します。
     *
     * @return  送信バッファサイズ
     *
     * @details
     */
    size_t GetRequestBufferSize() const NN_NOEXCEPT;

    /**
     * @brief   受信データを取得します。
     *
     * @return  受信データ
     *
     * @pre
     *  - Perform() 呼び出し後であること
     *
     * @details
     */
    void* GetResponse() NN_NOEXCEPT;

    /**
     * @brief   受信データサイズを取得します。
     *
     * @return  受信データサイズ
     *
     * @pre
     *  - Perform() 呼び出し後であること
     *
     * @details
     */
    size_t GetResponseSize() const NN_NOEXCEPT;

private:
    //
    size_t m_ReceivedSize;
    //
    HttpConnectionProvider* m_pConnectionProvider;
    //
    const char* m_pToken;

private:
    //
    static void DumpCommunicationPerformance(CURL* pCurl) NN_NOEXCEPT;
    //
    static int SockOptFunction(void* pParam, curl_socket_t sock, curlsocktype purpose) NN_NOEXCEPT;
    //
    static CURLcode SslCtxFunction(CURL* pCurl, void* pSsl, void* pParam) NN_NOEXCEPT;
    //
    static size_t HttpWriteFunction(char* pBuffer, size_t size, size_t count, void* pParam) NN_NOEXCEPT;
};

}}}}
