﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "capsrv_AlbumCacheData.h"

#include "capsrv_AlbumAccess.h"
#include "capsrv_AlbumControl.h"

namespace nn{ namespace capsrv{

//! @name アルバムキャッシュ操作
//! @{

    //! @brief アルバムのキャッシュを作り直します（テスト用）
    //! @param[in] storage キャッシュを作り直すストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が AlbumStorage の有効な値
    nn::Result RefreshAlbumCache(AlbumStorageType storage) NN_NOEXCEPT;

    //! @brief アルバムのキャッシュを取得します（テスト用）
    //! @param[in] storage キャッシュを取得するストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutValue != nullptr
    //! @pre storage が AlbumStorage の有効な値
    //! @pre contents が AlbumFileContents の有効な値
    nn::Result GetAlbumCache(AlbumCacheData* pOutValue, AlbumStorageType storage, AlbumFileContentsType contents) NN_NOEXCEPT;

//! @}

//! @name アルバムマウント操作
//! @{

    //! @brief 指定したストレージのアルバムを強制的にアンマウントします（テスト用）
    //! @param[in] storage アンマウントするアルバムのあるストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が AlbumStorage の有効な値
    nn::Result ForceAlbumUnmounted(AlbumStorageType storage) NN_NOEXCEPT;

    //! @brief 指定したストレージのアルバムのマウント状態をリセットします（テスト用）
    //! @param[in] storage マウント状態をリセットするストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が AlbumStorage の有効な値
    nn::Result ResetAlbumMountStatus(AlbumStorageType storage) NN_NOEXCEPT;

//! @}

//! @name アルバムエントリー操作
//! @{

    //! @brief ApplicationAlbumEntry から AlbumEntry を取得します。
    //! @param[out] pOutValue AlbumEntry を受け取る変数のポインタ
    //! @param[in] srcEntry 変換元の ApplicationAlbumEntry
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @details
    //! 自アプレットが取得した ApplicationAlbumEntry のみ変換できます。
    nn::Result GetAlbumEntryFromApplicationAlbumEntry(AlbumEntry* pOutValue, const nn::capsrv::ApplicationAlbumEntry& srcEntry) NN_NOEXCEPT;

//! @}

//! @name セッション操作
//! @{

    //! @brief AlbumAccessorSession をオープンします。
    nn::Result OpenAlbumAccessorSession() NN_NOEXCEPT;

    //! @brief AlbumAccessorSession をクローズします。
    void CloseAlbumAccessorSession() NN_NOEXCEPT;

    //! @brief AlbumControlSession をオープンします。
    nn::Result OpenAlbumControlSession() NN_NOEXCEPT;

    //! @brief AlbumControlSession をクローズします。
    void CloseAlbumControlSession() NN_NOEXCEPT;

//! @}

//! @name 動画読込
//! @{

    nn::Result CloseAlbumMovieReadStreamImpl(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

//! @}

//! @name 動画書出
//! @{

    nn::Result DiscardAlbumMovieWriteStreamImpl(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result DiscardAlbumMovieWriteStreamNoDeleteImpl(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;

//! @}


//! @name 動作モード変更
//! @{

    //! @brief 内部エラー値の変換の有効・無効を変更します（テスト用）
    //! @param[in] value 有効にするかの値です。
    //! @details
    //!   初期値は true です。
    //! @pre アルバムへのアクセス機能が初期化されている
    nn::Result SetInternalConversionEnabled(bool value) NN_NOEXCEPT;

//! @}

}}
