﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/capsrv/capsrv_Result.h>

namespace nn{ namespace capsrv{

    //! @brief アルバム内のファイルの作成日時を表す構造体です。
    struct AlbumFileDateTime
    {
        //! @brief 年を表します。
        uint16_t year;
        //! @brief 月を表します。
        uint8_t month;
        //! @brief 日を表します。
        uint8_t day;
        //! @brief 時を表します。
        uint8_t hour;
        //! @brief 分を表します。
        uint8_t minute;
        //! @brief 秒を表します。
        uint8_t second;
        //! @brief 同一時刻内での連番です。
        uint8_t id;

    public:
        bool IsValidDateTime() const NN_NOEXCEPT
        {
            return nn::time::IsValidDate(this->year, this->month, this->day) &&
                   (this->hour   >= 0 && this->hour   <= 23) &&
                   (this->minute >= 0 && this->minute <= 59) &&
                   (this->second >= 0 && this->second <= 59);
        }

        Result ToCalendarTime(nn::time::CalendarTime* pOut) const NN_NOEXCEPT
        {
            // nn::time::CalendarTime に変換可能かをチェック
            NN_RESULT_THROW_UNLESS( this->IsValidDateTime(), ResultAlbumInvalidTimestamp() );

            nn::time::CalendarTime time = {};
            time.year   = this->year;
            time.month  = this->month;
            time.day    = this->day;
            time.hour   = this->hour;
            time.minute = this->minute;
            time.second = this->second;
            *pOut = time;
            NN_RESULT_SUCCESS;
        }

        void FromCalendarTime(const time::CalendarTime& time) NN_NOEXCEPT
        {
            this->year   = time.year;
            this->month  = time.month;
            this->day    = time.day;
            this->hour   = time.hour;
            this->minute = time.minute;
            this->second = time.second;
            this->id     = 0;
        }

        void SetUint64Value(uint64_t value) NN_NOEXCEPT
        {
            this->year   = (value >> 48) & 0xffff;
            this->month  = (value >> 40) & 0xff;
            this->day    = (value >> 32) & 0xff;
            this->hour   = (value >> 24) & 0xff;
            this->minute = (value >> 16) & 0xff;
            this->second = (value >> 8)  & 0xff;
            this->id     = (value >> 0)  & 0xff;
        }

        uint64_t GetUint64Value() const NN_NOEXCEPT
        {
            return 0ull
                | static_cast<uint64_t>(this->year)   << 48
                | static_cast<uint64_t>(this->month)  << 40
                | static_cast<uint64_t>(this->day)    << 32
                | static_cast<uint64_t>(this->hour)   << 24
                | static_cast<uint64_t>(this->minute) << 16
                | static_cast<uint64_t>(this->second) << 8
                | static_cast<uint64_t>(this->id)     << 0
                ;
        }

        // 比較演算子
        friend bool operator==(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return lhs.GetUint64Value() == rhs.GetUint64Value();
        }

        friend bool operator!=(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }

        friend bool operator<(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return lhs.GetUint64Value() < rhs.GetUint64Value();
        }

        friend bool operator>(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return rhs < lhs;
        }

        friend bool operator<=(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return !(rhs < lhs);
        }

        friend bool operator>=(const AlbumFileDateTime& lhs, const AlbumFileDateTime& rhs) NN_NOEXCEPT
        {
            return !(lhs < rhs);
        }
    };

    NN_STATIC_ASSERT(sizeof(AlbumFileDateTime) == 8);

    const AlbumFileDateTime AlbumFileDateTimeMin = {1970,1,1,0,0,0,0};
    const AlbumFileDateTime AlbumFileDateTimeMax = {3000,1,1,0,0,0,0};

}}
