﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "capsrv_Result.h"
#include "capsrv_AlbumEntry.h"
#include "capsrv_AlbumUsage.h"
#include "capsrv_AlbumFileSizeLimit.h"
#include "capsrv_AlbumFileCountLimit.h"
#include "capsrv_ApplicationAlbumEntry.h"
#include "capsrv_ScreenShotAttribute.h"
#include "capsrv_ThumbnailFormat.h"
#include "capsrv_ViewerThumbnailFormat.h"
#include "capsrv_OverlayNotificationRequest.h"
#include "capsrv_ScreenShotDecodeOption.h"
#include "capsrv_AlbumMovieReadStreamHandle.h"
#include "capsrv_AppletData.h"
#include "movie/capsrv_MovieReaderFileSystem.h"

namespace nn{ namespace capsrv{

    //! @brief アルバムへのアクセス機能を初期化します。
    //! @pre アルバムへのアクセス機能が初期化されていない
    //! @post アルバムへのアクセス機能が初期化されている
    //! @details
    //! アルバムにアクセスする関数を呼び出す前にこの関数を呼び出す必要があります。
    nn::Result InitializeAlbumAccess() NN_NOEXCEPT;

    //! @brief アルバムへのアクセス機能の終了処理を行います。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @post アルバムへのアクセス機能が初期化されていない
    //! @details
    //! この関数を呼び出した後にアルバムにアクセスする関数を呼び出してはいけません。
    void FinalizeAlbumAccess() NN_NOEXCEPT;

    //! @brief 指定されたストレージのアルバム内のファイル数の上限値を取得します。
    //! @param[in] storage 調べるアルバムが存在するストレージです。
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @return ファイル数の上限値を返します
    //! @details
    //! 取得される値は AlbumFileCountLimit の該当する値と同一です。
    //! GetAlbumFileCount() により取得される値は必ずこの関数で取得される値以下です。
    int GetAlbumFileCountLimit(AlbumStorageType storage) NN_NOEXCEPT;

    //! @brief アルバム内のファイル数を取得します。
    //! @param[out] pOutFileCount ファイル数を受け取る変数のポインタです。
    //! @param[in] storage 調べるアルバムが存在するストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileCount != nullptr
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutFileCount にアルバム内のファイル数が代入されます。
    //! @post 失敗した場合、 *pOutFileCount に 0 が代入されます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @details
    //! この関数で取得されるファイル数はアルバムで管理されるファイル数です。
    //! GetAlbumFileList() で取得されるファイル数と同じ値が取得されます。
    //! 本関数では以下のファイル種別を対象としてファイル数をカウントします。
    //!
    //! - nn::capsrv::AlbumFileContents_ScreenShot
    //! - nn::capsrv::AlbumFileContents_Movie
    //!
    nn::Result GetAlbumFileCount(
        int* pOutFileCount,
        AlbumStorageType storage
        ) NN_NOEXCEPT;

    //! @brief アルバム内のファイル数を取得します。
    //! @param[out] pOutFileCount ファイル数を受け取る変数のポインタです。
    //! @param[in] storage 調べるアルバムが存在するストレージです。
    //! @param[in] contentsMask 取得対象とするファイルの種類を示すマスクパターン
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileCount != nullptr
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutFileCount にアルバム内のファイル数が代入されます。
    //! @post 失敗した場合、 *pOutFileCount に 0 が代入されます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @details
    //! この関数で取得されるファイル数はアルバムで管理されるファイル数です。
    //! GetAlbumFileList() で取得されるファイル数と同じ値が取得されます。
    //! 本関数では contentsMask で指定されたファイル種別を対象として
    //! ファイル数をカウントします。
    //!
    nn::Result GetAlbumFileCount(
        int* pOutFileCount,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT;

    //! @brief アルバム中のファイルのリストを取得します。
    //! @param[out] pOutFileCount pAlbumEntryBuffer に書き込まれた要素数を受け取る変数のポインタです。
    //! @param[out] pAlbumEntryBuffer 取得したファイルリストを受け取る配列です。
    //! @param[in] albumEntryBufferLength pAlbumEntryBuffer で渡した配列の長さです。
    //! @param[in] storage ファイルリストを取得するアルバムが存在するストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileCount != nullptr
    //! @pre pAlbumEntryBuffer != nullptr
    //! @pre albumEntryBufferLength > 0
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutFileCount に取得したファイルリストの長さが代入されます。
    //! @post 成功した場合、 pAlbumEntryBuffer にファイルリストが代入されます。
    //! @post 失敗した場合、 *pOutFileCount に 0 が代入されます。
    //! @post 失敗した場合、 pAlbumEntryBuffer の全要素は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capstv::ResultAlbumError
    //! @details
    //! この関数で取得されるファイルリストは、アルバムの管理対象となっているファイルのリストです。
    //! この関数で取得されるリストの長さは GetAlbumFileCount() で取得されるファイル数と同一です。@n
    //! @n
    //! pAlbumEntryBuffer に書き込まれる要素の順番は保証されません。@n
    //! albumEntryBufferLength がアルバム中のファイル数よりも大きな値だった場合、0 から (*pOutFileCount - 1) 番目の要素に値が書き込まれ、それ以降の要素は 0 埋めされます。@n
    //! albumEntryBufferLength がアルバム中のファイル数よりも小さな値だった場合、pAlbumEntryBuffer には albumEntryBufferLength 個の要素が書き込まれます。
    //! 書き込まれなかった要素を後から取得することはできません。
    //! アルバム内に存在するファイルの完全なリストを取得するためには albumEntryBufferLength が GetAlbumFileCount() で取得される値以上である必要があります。@n
    //! 本関数では以下のファイル種別を対象としてファイル数をカウントします。
    //!
    //! - nn::capsrv::AlbumFileContents_ScreenShot
    //! - nn::capsrv::AlbumFileContents_Movie
    //!
    nn::Result GetAlbumFileList(
        int* pOutFileCount,
        AlbumEntry* pAlbumEntryBuffer,
        int albumEntryBufferLength,
        AlbumStorageType storage
        ) NN_NOEXCEPT;

    //! @brief アルバム中のファイルのリストを取得します。
    //! @param[out] pOutFileCount pAlbumEntryBuffer に書き込まれた要素数を受け取る変数のポインタです。
    //! @param[out] pAlbumEntryBuffer 取得したファイルリストを受け取る配列です。
    //! @param[in] albumEntryBufferLength pAlbumEntryBuffer で渡した配列の長さです。
    //! @param[in] storage ファイルリストを取得するアルバムが存在するストレージです。
    //! @param[in] contentsMask 取得対象とするファイルの種類を示すマスクパターン
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileCount != nullptr
    //! @pre pAlbumEntryBuffer != nullptr
    //! @pre albumEntryBufferLength > 0
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutFileCount に取得したファイルリストの長さが代入されます。
    //! @post 成功した場合、 pAlbumEntryBuffer にファイルリストが代入されます。
    //! @post 失敗した場合、 *pOutFileCount に 0 が代入されます。
    //! @post 失敗した場合、 pAlbumEntryBuffer の全要素は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capstv::ResultAlbumError
    //! @details
    //! この関数で取得されるファイルリストは、アルバムの管理対象となっているファイルのリストです。
    //! この関数で取得されるリストの長さは GetAlbumFileCount() で取得されるファイル数と同一です。@n
    //! @n
    //! pAlbumEntryBuffer に書き込まれる要素の順番は保証されません。@n
    //! albumEntryBufferLength がアルバム中のファイル数よりも大きな値だった場合、0 から (*pOutFileCount - 1) 番目の要素に値が書き込まれ、それ以降の要素は 0 埋めされます。@n
    //! albumEntryBufferLength がアルバム中のファイル数よりも小さな値だった場合、pAlbumEntryBuffer には albumEntryBufferLength 個の要素が書き込まれます。
    //! 書き込まれなかった要素を後から取得することはできません。
    //! アルバム内に存在するファイルの完全なリストを取得するためには albumEntryBufferLength が GetAlbumFileCount() で取得される値以上である必要があります。@n
    //! 本関数では contentsMask で指定されたファイル種別を対象として
    //! ファイル数をカウントします。
    //!
    nn::Result GetAlbumFileList(
        int* pOutFileCount,
        AlbumEntry* pAlbumEntryBuffer,
        int albumEntryBufferLength,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT;

    //! @brief アルバム中のファイルの画像データ部分を読み込みます。
    //! @param[out] pOutSize pBuffer に書き込まれたデータサイズ（バイト）を受け取る変数のポインタです。
    //! @param[out] pBuffer ファイルの内容を受け取るバッファです。
    //! @param[in] bufferSize pBuffer で渡したバッファのサイズ（バイト）です。
    //! @param[in] pFileId 読み込むファイルの識別子のポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutSize != nullptr
    //! @pre pBuffer != nullptr
    //! @pre pFileId != nullptr
    //! @post 成功した場合、 *pOutSize に読み込んだ画像データのサイズが代入されます。
    //! @post 成功した場合、 pBuffer に読み込んだ画像データが代入されます。
    //! @post 失敗した場合、 *pOutSize に 0 が代入されます。
    //! @post 失敗した場合、 pBuffer の値は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @details
    //! アルバム内に存在するファイルの識別子は GetAlbumFileList() により取得することができます。@n
    //! 画像データのサイズは nn::capsrv::AlbumFileSizeLimit_ScreenShot 以下であることが保証されます。@n
    //! @n
    //! pFileId->contents == AlbumFileContents_ScreenShot の場合、画面写真データ全体を読み込みます。@n
    //! pFileId->contents == AlbumFileContents_Movie の場合、動画に埋め込まれた静止画データ部分を読み込みます。@n
    //! @n
    //! bufferSize が読み込む画像データのサイズよりも大きかった場合、 pBuffer の先頭から画像データを書き込み、以降は 0 埋めされます。
    //! bufferSize が読み込む画像データのサイズよりも小さかった場合、この関数は失敗します。
    nn::Result LoadAlbumScreenShotFile(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT;

    //! @brief アルバム中のファイルのサムネイルを読み込みます。
    //! @param[out] pOutSize pBuffer に書き込まれたデータサイズ（バイト）を受け取る変数のポインタです。
    //! @param[out] pBuffer サムネイルの内容を受け取るバッファです。
    //! @param[in] bufferSize pBuffer で渡したバッファのサイズ（バイト）です。
    //! @param[in] pFileId 読み込むファイルの識別子のポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutSize != nullptr
    //! @pre pBuffer != nullptr
    //! @pre pFileId != nullptr
    //! @post 成功した場合、 *pOutSize に読み込んだサムネイルのデータサイズが代入されます。
    //! @post 成功した場合、 pBuffer に読み込んだサムネイルのデータが代入されます。
    //! @post 失敗した場合、 *pOutSize に 0 が代入されます。
    //! @post 失敗した場合、 pBuffer は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumInvalidFileData
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumNoThumbnail
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @details
    //! アルバム内に存在するファイルの識別子とファイルサイズは GetAlbumFileList() により取得することができます。@n
    //! @n
    //! bufferSize がサムネイルのデータよりも大きかった場合、 pBuffer の先頭からサムネイルデータを書き込み、以降は 0 埋めされます。
    //! bufferSize が読み込むサムネイルのサイズよりも小さかった場合、この関数は ResultAlbumReadBufferShortage により失敗します。
    //! ファイルにサムネイルが含まれていなかった場合、 ResultAlbumNoThumbnail により失敗します。
    //! ファイルの解析に失敗した場合、 ResultAlbumInvalidFileData により失敗します。@n
    //! 正規のファイルのサムネイルの最大サイズは ViewerThumbnailImageDataSize_EncodedLimit で定義された値です。@n
    //! @n
    //! この関数はファイルの先頭部分の検査を行います。@n
    //! この関数はサムネイルの内容を検査しません。@n
    //! 正規のファイルのサムネイルは JPEG エンコードされた画像データであり、サムネイルのヘッダ部分から画像サイズを取得することができます。
    //! 正規のファイルのサムネイルの画像サイズは ViewerThumbnailImageSize で定義された値です。
    //! @n
    //! 正規のファイルに対しても ResultAlbumNoThumbnail が返る場合があります。
    nn::Result LoadAlbumFileThumbnail(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT;

    //! @brief アルバム中の画面写真のデータを読み込みます。
    //! @param[out] pOutWidth      読み込んだ画面写真の幅（ピクセル）を受け取る変数のポインタです。
    //! @param[out] pOutHeight     読み込んだ画面写真の高さ（ピクセル）を受け取る変数のポインタです。
    //! @param[out] pOutAttribute  読み込んだ画面写真の属性を受け取る変数のポインタです。
    //! @param[out] pOutAppletData 読込んだ画面写真のアプレット用データ領域を受け取る変数のポインタです。
    //! @param[out] pBuffer        画面写真データを受け取るバッファです。
    //! @param[in] bufferSize      画面写真データを受け取るバッファの大きさ（バイト）です。
    //! @param[in] pFileId         読み込むファイルの識別子のポインタです。
    //! @param[in] option          デコード時のオプションです。
    //! @param[in] pWorkBuffer     作業用バッファです。
    //! @param[in] workBufferSize  作業用バッファの大きさ（バイト）です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutWidth != nullptr
    //! @pre pOutHeight != nullptr
    //! @pre pOutAttribute != nullptr
    //! @pre pBuffer != nullptr
    //! @pre pFileId != nullptr
    //! @pre pWorkBuffer != nullptr
    //! @post 成功した場合、 *pOutWidth に読み込んだ画面写真の幅が代入されます。
    //! @post 成功した場合、 *pOutHeight に読み込んだ画面写真の高さが代入されます。
    //! @post 成功した場合、 *pOutAttribute に読み込んだ画面写真の属性が代入されます。
    //! @post 成功した場合、 *pOutAppletData に読み込んだアプレット用データ領域の値が代入されます。
    //! @post 成功した場合、 pBuffer に読み込んだ画面写真データが代入されます。
    //! @post 失敗した場合、 *pOutWidth に 0 が代入されます。
    //! @post 失敗した場合、 *pOutHeight に 0 が代入されます。
    //! @post 失敗した場合、 pBuffer は 0 埋めされます。
    //! @post 成功失敗に関わらず pWorkBuffer は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumInvalidFileData
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @details
    //! この関数は画面写真の読み込み専用です。
    //! LoadAlbumScreenShotImage() で読み込んだ画像をデコードした結果を取得します。
    //! 取得される画像データは sRGBA8 フォーマットで、 パディングなしのリニア画像 (stride = 4 * width, size = 4 * width * height) です。
    //! @n
    //! bufferSize が画像データサイズよりも大きかった場合、 pBuffer の先頭から画像データを書き込み、以降は 0 埋めされます。
    //! bufferSize が画像データサイズよりも小さかった場合、 ResultAlbumReadBufferShortage により失敗します。
    //! workBufferSize が必要な大きさに満たなかった場合、 ResultAlbumReadBufferShortage により失敗します。
    //! @n
    //! pBuffer には画面写真を受け取るのに十分な大きさのバッファを渡す必要があります。
    //! 現在の実装では 4 * 1280 * 720 バイト以上のバッファを渡せば十分です。
    //! pBuffer には CPU キャッシュが有効な GPU のメモリプール上のメモリを渡すことができます。
    //! @n
    //! pWorkBuffer には GetAlbumFileSize() で取得されるファイルサイズよりも大きいバッファを渡す必要があります。
    //! AlbumFileSizeLimit_ScreenShot バイト以上のバッファを渡せば十分です。
    //! pWorkBuffer に GPU のメモリプール上のメモリを渡すことはできません。@n
    //! @n
    //! 現在の実装では成功した場合に *pOutWidth に代入される値は 1280 です。
    //! 現在の実装では成功した場合に *pOutHeight に代入される値は 720 です。
    nn::Result LoadAlbumScreenShotImage(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId,
        const ScreenShotDecodeOption& option,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT;

    //! @brief アルバム中の画面写真のサムネイル画像データを読み込みます。
    //! @param[out] pOutWidth      読み込んだサムネイル画像の幅（ピクセル）を受け取る変数のポインタです。
    //! @param[out] pOutHeight     読み込んだサムネイル画像の高さ（ピクセル）を受け取る変数のポインタです。
    //! @param[out] pOutAttribute  読み込んだ画面写真の属性を受け取る変数のポインタです。
    //! @param[out] pOutAppletData 読込んだ画面写真のアプレット用データ領域を受け取る変数のポインタです。
    //! @param[out] pBuffer        サムネイル画像データを受け取るバッファです。
    //! @param[in] bufferSize      サムネイル画像データを受け取るバッファの大きさ（バイト）です。
    //! @param[in] pFileId         読み込むファイルの識別子のポインタです。
    //! @param[in] option          デコード時のオプションです。
    //! @param[in] pWorkBuffer     作業用バッファです。
    //! @param[in] workBufferSize  作業用バッファの大きさ（バイト）です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutWidth != nullptr
    //! @pre pOutHeight != nullptr
    //! @pre pOutAttribute != nullptr
    //! @pre pBuffer != nullptr
    //! @pre pFileId != nullptr
    //! @pre pWorkBuffer != nullptr
    //! @post 成功した場合、 *pOutWidth に読み込んだサムネイル画像の幅が代入されます。
    //! @post 成功した場合、 *pOutHeight に読み込んだサムネイル画像の高さが代入されます。
    //! @post 成功した場合、 *pOutAttribute に読み込んだ画面写真の属性が代入されます。
    //! @post 成功した場合、 *pOutAppletData に読み込んだアプレット用データ領域の値が代入されます。
    //! @post 成功した場合、 pBuffer に読み込んだサムネイル画像データが代入されます。
    //! @post 失敗した場合、 *pOutWidth に 0 が代入されます。
    //! @post 失敗した場合、 *pOutHeight に 0 が代入されます。
    //! @post 失敗した場合、 pBuffer は 0 埋めされます。
    //! @post 成功失敗に関わらず pWorkBuffer は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumInvalidFileData
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @retval nn::capsrv::ResultAlbumNoThumbnail
    //! @details
    //! この関数は画面写真の読み込み専用です。
    //! LoadAlbumFileThumbnail() で読み込んだ画像をデコードした結果を取得します。
    //! 取得される画像データは sRGBA8 フォーマットで、 パディングなしのリニア画像 (stride = 4 * width, size = 4 * width * height) です。
    //! @n
    //! bufferSize がサムネイル画像のデータサイズよりも大きかった場合、 pBuffer の先頭からサムネイル画像データを書き込み、以降は 0 埋めされます。
    //! bufferSize がサムネイル画像のデータサイズよりも小さかった場合、 ResultAlbumReadBufferShortage により失敗します。
    //! workBufferSize が必要な大きさに満たなかった場合、 ResultAlbumReadBufferShortage により失敗します。
    //! @n
    //! pBuffer にはサムネイル画像を受け取るのに十分な大きさのバッファを渡す必要があります。
    //! ViewerThumbnailImageDataSize_Raw バイト以上のバッファを渡せば十分です。
    //! pBuffer には CPU キャッシュが有効な GPU のメモリプール上のメモリを渡すことができます。
    //! @n
    //! pWorkBuffer にはデコード前のサムネイルデータのサイズよりも大きいバッファを渡す必要があります。
    //! ViewerThumbnailImageDataSize_EncodedLimit バイト以上のバッファを渡せば十分です。
    //! pWorkBuffer に GPU のメモリプール上のメモリを渡すことはできません。
    //! @n
    //! 成功した場合に *pOutWidth に代入される値は ViewerThumbnailImageSize_Width です。
    //! 成功した場合に *pOutHeight に代入される値は ViewerThumbnailImageSize_Height です。
    //! @n
    //! *pOutAttribute には画面写真全体の属性が代入されます。
    //! 属性から取得できる画面写真の大きさはフルサイズ画像の大きさです。
    //! @n
    //! 正規のファイルに対しても ResultAlbumNoThumbnail が返る場合があります。
    nn::Result LoadAlbumScreenShotThumbnailImage(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId,
        const ScreenShotDecodeOption& option,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT;

    //! @brief アルバム中のファイルを削除します。
    //! @param[in] pFileId 削除するファイルの識別子のポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pFileId != nullptr
    //! @post 成功した場合、 pFileId で指定したファイルは削除されます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @details
    //! アルバム内に存在するファイルの識別子は GetAlbumFileList() により取得することができます。@n
    nn::Result DeleteAlbumFile(
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT;

    //! @brief アルバム内のファイルをストレージ間でコピーします。
    //! @param[in] pSrcFileId コピー元のファイルの識別子のポインタです。
    //! @param[in] destStorage コピー先のストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pSrcFileId != nullptr
    //! @pre destStorage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 pSrcFileId で指定したファイルが destStorage 上のアルバムにコピーされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsFull
    //! @retval nn::capsrv::ResultAlbumFileCountLimit
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @details
    //! アルバム内に存在するファイルの識別子は GetAlbumFileList() により取得することができます。@n
    //! コピー先のストレージに同じ識別子を持つファイルが既に存在した場合、コピー元のファイルで上書きされます。
    nn::Result StorageCopyAlbumFile(
        const AlbumFileId* pSrcFileId,
        AlbumStorageType destStorage
        ) NN_NOEXCEPT;

    //! @brief アルバム内の全ファイルをストレージ間でコピーする場合に、コピー先のストレージに必要な空き容量を計算します。
    //! @param[out] pOutValue 空き容量を受け取る変数のポインタです。
    //! @param[in] dstStorage コピー先のストレージです。
    //! @param[in] srcStorage コピー元のストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutValue != nullptr
    //! @pre dstStorage が nn::capsrv::AlbumStorage のいずれかの値
    //! @pre srcStorage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutValue に全ファイルをコピーするのに必要なストレージの空き容量が代入されます。
    //! @post 失敗した場合、 *pOutValue に 0 が代入されます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidStorage
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @details
    //! コピー元のストレージとコピー先のストレージを比較して全ファイルをコピーするために必要な空き容量を計算します。
    //! この API で取得される値は実際に必要な空き容量よりも若干大きな値です。
    //! 実際の空き容量が取得された値より小さくても全ファイルのコピーに成功する場合があります。
    //! @n
    //! ストレージ間でファイルに差分がない場合でも非 0 の値を返します。
    //! コピー先のストレージの空き容量が十分な場合でも、アルバムのファイル数制限によりすべてのファイルがコピーできない場合があります。
    nn::Result GetRequiredStorageFreeSpaceSizeToCopyAll(
        size_t* pOutValue,
        AlbumStorageType dstStorage,
        AlbumStorageType srcStorage
        ) NN_NOEXCEPT;

    //! @brief 指定されたストレージのアルバムがマウントされているかを取得します。
    //! @param[in] storage 調べるストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @retval true アルバムがマウントされている。
    //! @retval false アルバムがマウントされていない。
    //! @details
    //! この関数の呼び出し前に指定したストレージのアルバムへのアクセスが行われていなかった場合、アルバムのマウントを試みてその結果を返します。
    //! それ以外の場合、このアルバムへのアクセスは行わずに前回のアクセス時にストレージがマウントされていたかを返します。
    //! この関数を使用して他の API やユーザーの操作によるストレージのアンマウントを検出することはできません。
    //!
    //! マウントに失敗した理由を取得する場合、 GetAlbumMountResult() を使用します。
    bool IsAlbumMounted(
        AlbumStorageType storage
        ) NN_NOEXCEPT;

    //! @brief 指定されたストレージのアルバムのマウント結果を取得します。
    //! @param[in] storage 調べるストレージです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @retval ResultSuccess アルバムがマウントされている。
    //! @retval ResultAlbumIsNotMounted アルバムがマウントされていない。
    //! @retval ResultAlbumStructureCorrupted アルバムのディレクトリ構成が不正なためマウントに失敗した。
    //! @details
    //! この関数の呼び出し前に指定したストレージのアルバムへのアクセスが行われていなかった場合、アルバムのマウントを試みてその結果を返します。
    //! それ以外の場合、直前のマウントの結果を返します。
    //!
    //! GetAlbumMountResult(storage).IsSuccess() は IsAlbumMounted(storage) と等価です。
    //!
    //! この関数を使用して他の API やユーザーの操作によるストレージのアンマウントを検出することはできません。
    nn::Result GetAlbumMountResult(
        AlbumStorageType storage
        ) NN_NOEXCEPT;

    //! @brief 指定されたストレージのアルバムの使用状況を取得します。
    //! @param[out] pOutValue   アルバムの使用状況を受け取る変数のポインタです。
    //! @param[in] storage      対象のストレージです。
    //! @param[in] contentsMask 対象のファイルの種類の組み合わせです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutValue に指定したストレージの使用状況が代入されます。
    //! @post 失敗した場合、 *pOutValue はクリアされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capstv::ResultAlbumError
    //! @details
    //! 指定されたストレージのアルバム中のファイルの数と容量の合計を取得します。
    //!
    //! contentsMask で指定されたファイルの種類が個別に集計されます。
    //! contentsMask で指定されていない種類のファイルは未知のファイルとして集計されます。
    //! 例えば、 contentsMask[AlbumFileContentsFlag_ScreenShot] = true とした場合には画面写真ファイルは個別に集計され、動画ファイルは UnknownUsage に集計されます。
    //! contentsMask[AlbumFileContentsFlag_ScreenShot] = contentsMask[AlbumFileContentsFlag_Movie] = true とした場合には画面写真と動画がそれぞれ個別に集計されます。
    //!
    //! ファイルの集計はアルバムの管理対象のディレクトリに存在するファイルだけが対象となります。
    //! アルバム中に管理対象外のサブディレクトリがあった場合、サブディレクトリ中のファイルはカウントせずに AlbumContentsUsageFlag_HasGreaterUsage フラグを立てます。
    nn::Result GetAlbumUsage(
        AlbumUsage* pOutValue,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT;

    //! @brief 指定されたストレージのアルバムの使用状況を取得します。
    //! @param[out] pOutValue アルバムの使用状況を受け取る変数のポインタです。
    //! @param[in] storage 調べるストレージです。
    //! @details
    //! 互換性のために残されています。
    //! contentsMask[AlbumFileContentsFlag_ScreenShot] = true として呼び出した場合と同様です。
    nn::Result GetAlbumUsage(
        AlbumUsage* pOutValue,
        AlbumStorageType storage
        ) NN_NOEXCEPT;

    //! @brief 指定された種類のファイルの大きさの上限値（バイト）を取得します。
    //! @param[in] contents 調べるファイルの種類です。
    //! @pre contents が nn::capsrv::AlbumFileContents のいずれかの値
    //! @details
    //! 指定された種類のファイルの最大サイズを取得します。
    //! 取得される値は AlbumFileSizeLimit の該当する値と同一です。
    size_t GetAlbumFileSizeLimit(AlbumFileContentsType contents) NN_NOEXCEPT;

    //! @brief 指定したファイルの大きさ（バイト）を取得します。
    //! @param[out] pOutValue ファイルサイズを受け取る変数へのポインタです。
    //! @param[in] pFileId 調べるファイルの識別子のポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pFileId != nullptr
    //! @post 成功した場合、 *pOutValue にファイルサイズが代入されます。
    //! @post 失敗した場合、 *pOutValue に 0 が代入されます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @details
    //! 指定されたファイルのファイルサイズ（バイト）を取得します。
    nn::Result GetAlbumFileSize(size_t* pOutValue, const AlbumFileId* pFileId) NN_NOEXCEPT;

    //! @brief 画像データを画面写真を画面写真として保存します。
    //! @param[out] pOutEntry     新しく保存した画面写真のエントリーを受け取る変数へのポインタです。
    //! @param[in] imageData      保存する画面写真の画像データ (R8G8B8A8) です。
    //! @param[in] imageDataSize  imageData の大きさ（バイト）です。
    //! @param[in] attribute     保存する画面写真の属性です。
    //! @param[in] overlayRequest 保存の成否をオーバーレイに通知する条件です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileId != nullptr
    //! @pre imageData != nullptr
    //! @pre imageDataSize == 4 * 1280 * 720
    //! @post 成功した場合、 *pOutEntry に保存したファイルの情報が代入されます。
    //! @post 失敗した場合、 *pOutEntry はクリアされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultScreenShotEncodingFailed
    //! @retval nn::capsrv::ResultAlbumIsFull
    //! @retval nn::capsrv::ResultAlbumFileCountLimit
    //! @retval nn::capsrv::ResultAlbumError
    //! @details
    //! 画像データを画面写真を保存します。
    //! 画面写真はこの関数を呼び出したアプレットの画面写真となります。
    //!
    //! 画面写真が保存されるストレージは自動的に決定されます。
    //! 保存先のストレージは GetAutoSavingStorage() で取得することができます。
    nn::Result SaveScreenShot(
        ApplicationAlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        const ScreenShotAttribute& attribute,
        OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT;

    //! @brief 編集した画面写真を新しい画面写真として保存します。
    //! @param[out] pOutEntry    新しく保存した画面写真のエントリーを受け取る変数へのポインタです。
    //! @param[in] imageData     保存する画面写真の画像データ (R8G8B8A8) です。
    //! @param[in] imageDataSize imageData の大きさ（バイト）です。
    //! @param[in] width         保存する画面写真の幅（ピクセル）です。
    //! @param[in] height        保存する画面写真の高さ（ピクセル）です。
    //! @param[in] thumbData     保存する画面写真のサムネイルの画像データ (R8G8B8A8) です。
    //! @param[in] thumbDataSize thumbData の大きさ（バイト）です。
    //! @param[in] thumbWidth    サムネイルの幅（ピクセル）です。
    //! @param[in] thumbHeight   サムネイルの高さ（ピクセル）です。
    //! @param[in] attribute     保存する画面写真の属性です。
    //! @param[in] appletData    保存する画面写真に書き込むアプレット用データ領域の値です。
    //! @param[in] pOriginalFileId 編集元の画面写真の識別子へのポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileId != nullptr
    //! @pre imageData != nullptr
    //! @pre imageDataSize == 4 * width * height
    //! @pre width == 1280
    //! @pre height == 720
    //! @pre thumbData != nullptr
    //! @pre thumbDataSize == 4 * thumbWidth * thumbHeight
    //! @pre thumbWidth  == ViewerThumbnailImageSize_Width
    //! @pre thumbHeight == ViewerThumbnailImageSize_Height
    //! @pre pOriginalFileId != nullptr
    //! @pre pOriginalFileId->contents == AlbumFileContents_ScreenShot
    //! @pre pOriginalFileId->storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutEntry に保存したファイルの情報が代入されます。
    //! @post 失敗した場合、 *pOutEntry はクリアされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultScreenShotEncodingFailed
    //! @retval nn::capsrv::ResultAlbumIsFull
    //! @retval nn::capsrv::ResultAlbumFileCountLimit
    //! @retval nn::capsrv::ResultAlbumError
    //! @details
    //! 編集した画面写真を保存します。
    //! 新しい画面写真として保存され、編集元の画面写真は変更されません。
    //!
    //! 新しい画面写真が保存されるストレージは自動的に決定されます。
    //! 保存先のストレージは GetAutoSavingStorage() で取得することができます。
    nn::Result SaveEditedScreenShot(
        AlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        int width,
        int height,
        const void* thumbData,
        size_t thumbDataSize,
        int thumbWidth,
        int thumbHeight,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const AlbumFileId* pOriginalFileId
        ) NN_NOEXCEPT;

    //! @brief 動画データの画像を新しい画面写真として保存します。
    //!
    //! @param[out] pOutEntry    新しく保存した画面写真のエントリーを受け取る変数へのポインタです。
    //! @param[in] imageData     保存する画面写真の画像データ (R8G8B8A8) です。
    //! @param[in] imageDataSize imageData の大きさ（バイト）です。
    //! @param[in] width         保存する画面写真の幅（ピクセル）です。
    //! @param[in] height        保存する画面写真の高さ（ピクセル）です。
    //! @param[in] thumbData     保存する画面写真のサムネイルの画像データ (R8G8B8A8) です。
    //! @param[in] thumbDataSize thumbData の大きさ（バイト）です。
    //! @param[in] thumbWidth    サムネイルの幅（ピクセル）です。
    //! @param[in] thumbHeight   サムネイルの高さ（ピクセル）です。
    //! @param[in] attribute     保存する画面写真の属性です。
    //! @param[in] appletData    保存する画面写真に書き込むアプレット用データ領域の値です。
    //! @param[in] pOriginalFileId 編集元の画面写真の識別子へのポインタです。
    //!
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileId != nullptr
    //! @pre imageData != nullptr
    //! @pre imageDataSize == 4 * width * height
    //! @pre width == 1280
    //! @pre height == 720
    //! @pre thumbData != nullptr
    //! @pre thumbDataSize == 4 * thumbWidth * thumbHeight
    //! @pre thumbWidth  == ViewerThumbnailImageSize_Width
    //! @pre thumbHeight == ViewerThumbnailImageSize_Height
    //! @pre pOriginalFileId != nullptr
    //! @pre pOriginalFileId->contents == AlbumFileContents_ScreenShot
    //! @pre pOriginalFileId->storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post 成功した場合、 *pOutEntry に保存したファイルの情報が代入されます。
    //! @post 失敗した場合、 *pOutEntry はクリアされます。
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultScreenShotEncodingFailed
    //! @retval nn::capsrv::ResultAlbumIsFull
    //! @retval nn::capsrv::ResultAlbumFileCountLimit
    //! @retval nn::capsrv::ResultAlbumError
    //!
    //! @details
    //! 動画データから抽出した静止画を画面写真として保存します。
    //! 新しい画面写真として保存され、編集元の動画データは変更されません。
    //!
    //! attribute には、nn::capsrv::ReadAttributeFromAlbumMovieReadStream() で
    //! 動画データから取得した ScreenShotAttribute を渡すようにして下さい。
    //!
    //! 新しい画面写真が保存されるストレージは自動的に決定されます。
    //! 保存先のストレージは GetAutoSavingStorage() で取得することができます。
    nn::Result SaveScreenShotOfMovie(
        AlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        int width,
        int height,
        const void* thumbData,
        size_t thumbDataSize,
        int thumbWidth,
        int thumbHeight,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const AlbumFileId* pOriginalFileId
        ) NN_NOEXCEPT;

    //! @deprecated 最後に撮影した静止画のサムネイルを取得します。
    //!
    //! @details
    //! 本 API は削除する予定です。@n
    //! 最新の nn::capsrv::GetLastOverlayScreenShotThumbnail() をご使用下さい。
    //!
    nn::Result GetLastThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT;

    //! @brief 最後に撮影した静止画のサムネイルを取得します。
    //! @param[out] pOutFileId   サムネイルに対応するファイル ID を受け取る変数のポインタです。
    //! @param[out] pOutDataSize 取得したサムネイルの画像データのサイズ（バイト）を受け取る変数のポインタです。
    //! @param[out] pOutBuffer   サムネイルの画像データを受け取るバッファです。
    //! @param[in]  bufferSize   pOutBuffer で渡したバッファの大きさ（バイト）です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileId != nullptr
    //! @pre pOutDataSize != nullptr
    //! @pre pOutBuffer != nullptr
    //! @post 成功した場合、 *pOutFileId にサムネイルに対応するファイル ID が代入されます。
    //! @post 成功した場合、 *pOutDataSize にサムネイル画像のデータサイズが代入されます。
    //! @post 成功した場合、 pOutBuffer にサムネイル画像データが代入されます。
    //! @post 失敗した場合、 *pOutFileId はクリアされます。
    //! @post 失敗した場合、 *pOutDataSize に 0 が代入されます。
    //! @post 失敗した場合、 pOutBuffer は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @details
    //! 最後に撮影した静止画のサムネイルを取得します。
    //! 連続で撮影された場合、取得されるサムネイルが期待したファイルと対応しない可能性があります。
    //!
    //! サムネイル画像の幅と高さは ThumbnailImageSize で定義されています。
    //! サムネイル画像のデータサイズは ThumbnailImageDataSize で定義されています。
    //! bufferSize がサムネイル画像のデータサイズに満たなかった場合、 ResultAlbumReadBufferShortage が返されます。
    //!
    //! 取得されたサムネイルがどのファイルに対応するかは *pOutFileId に代入されるファイル ID で確認することができます。
    //! サムネイルが撮影されていなかった場合、 *pOutFileId の値は 0 クリアされ、白一色のサムネイル画像が取得されます。
    //! @see ThumbnailImageSize
    //! @see ThumbnailImageFormat
    //! @see ThumbnailImageDataSize
    nn::Result GetLastOverlayScreenShotThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT;

    //! @brief 最後に撮影した動画のサムネイルを取得します。
    //! @param[out] pOutFileId   サムネイルに対応するファイル ID を受け取る変数のポインタです。
    //! @param[out] pOutDataSize 取得したサムネイルの画像データのサイズ（バイト）を受け取る変数のポインタです。
    //! @param[out] pOutBuffer   サムネイルの画像データを受け取るバッファです。
    //! @param[in]  bufferSize   pOutBuffer で渡したバッファの大きさ（バイト）です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutFileId != nullptr
    //! @pre pOutDataSize != nullptr
    //! @pre pOutBuffer != nullptr
    //! @post 成功した場合、 *pOutFileId にサムネイルに対応するファイル ID が代入されます。
    //! @post 成功した場合、 *pOutDataSize にサムネイル画像のデータサイズが代入されます。
    //! @post 成功した場合、 pOutBuffer にサムネイル画像データが代入されます。
    //! @post 失敗した場合、 *pOutFileId はクリアされます。
    //! @post 失敗した場合、 *pOutDataSize に 0 が代入されます。
    //! @post 失敗した場合、 pOutBuffer は 0 埋めされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @details
    //! 最後に撮影した動画のサムネイルを取得します。
    //! 連続で撮影された場合、取得されるサムネイルが期待したファイルと対応しない可能性があります。
    //!
    //! サムネイル画像の幅と高さは ThumbnailImageSize で定義されています。
    //! サムネイル画像のデータサイズは ThumbnailImageDataSize で定義されています。
    //! bufferSize がサムネイル画像のデータサイズに満たなかった場合、 ResultAlbumReadBufferShortage が返されます。
    //!
    //! 取得されたサムネイルがどのファイルに対応するかは *pOutFileId に代入されるファイル ID で確認することができます。
    //! サムネイルが撮影されていなかった場合、 *pOutFileId の値は 0 クリアされ、白一色のサムネイル画像が取得されます。
    //! @see ThumbnailImageSize
    //! @see ThumbnailImageFormat
    //! @see ThumbnailImageDataSize
    nn::Result GetLastOverlayMovieThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT;

    //! @brief 自動保存の保存先ストレージを取得します。
    //! @param[out] pOutValue 保存先ストレージを受け取る変数のポインタです。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutValue != nullptr
    //! @post 成功した場合、 *pOutValue に自動的に保存されるストレージの値が代入されます。
    //! @post 失敗した場合、 *pOutValue はクリアされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumIsNotMounted 保存先ストレージの取得に失敗
    //! @details
    //! 撮影ボタンの押下や編集済画面写真の保存の際にファイルが保存されるストレージを取得します。
    //! 優先的に保存するストレージがマウントされていればそのストレージが取得されます。
    //! 優先的に保存するストレージがマウントされていなければ本体内蔵メモリが取得されます。
    //! capsrv による自動保存ではストレージのマウント状態のみが参照されます。
    nn::Result GetAutoSavingStorage(
        AlbumStorageType* pOutValue
        ) NN_NOEXCEPT;

    //! @brief アプリケーション用のアルバムエントリーから通常のアルバムエントリーを取得します。
    //! @param[out] pOutValue    ファイルエントリーを受け取る変数のポインタです。
    //! @param[in] entry         アプリケーション用のアルバムエントリーです。
    //! @param[in] applicationId アプリケーション ID です。
    //! @pre アルバムへのアクセス機能が初期化されている
    //! @pre pOutValue != nullptr
    //! @post 成功した場合、 *pOutValue にファイルエントリーの値が代入されます。
    //! @post 失敗した場合、 *pOutValue はクリアされます。
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError 正しいアルバムエントリーが取得できませんでした。
    //! @details
    //! アプリケーション用のアルバムエントリーから通常の AlbumEntry を取得します。
    //! 不正な ApplicationAlbumEntry が渡された場合、 ResultAlbumError を返します。
    nn::Result GetAlbumEntryFromApplicationAlbumEntry(AlbumEntry* pOutValue, const ApplicationAlbumEntry& entry, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

//! @name 動画読込ストリーム
//! @{

    //! @brief 動画読込ストリームを開きます。
    //! @param[out] pOutHandle 動画読込ストリームのハンドルを受け取る変数のポインタです。
    //! @param[in]  fileId     開く動画ファイルの識別子です。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @pre pOutValue != nullptr
    //! @post 成功した場合、 *pOutHandle に有効な動画読込ストリームのハンドルが代入されます。
    //! @retval ResultAlbumResourceLimit 同時にオープンできるストリーム数の上限に達しています。
    //! @details
    //! 動画読込ストリームを開きます。
    //! 読込が終わった後は CloseAlbumMovieReadStream() でストリームを閉じる必要があります。
    nn::Result OpenAlbumMovieReadStream(AlbumMovieReadStreamHandle* pOutHandle, const AlbumFileId& fileId) NN_NOEXCEPT;

    //! @brief 動画読込ストリームを閉じます。
    //! @param[in] handle 閉じる動画読込ストリームのハンドルです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @pre handle が有効な動画読込ストリームのハンドル。
    //! @post handle は無効な動画読込ストリームのハンドルになります。
    void CloseAlbumMovieReadStream(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

    //! @brief 動画ファイルの動画データ部分のサイズを取得します。
    //! @param[out] pOutSize 動画データ部分のサイズ（バイト）を受け取る変数のポインタです。
    //! @param[in]  handle   動画読込ストリームのハンドルです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @post 成功した場合、 *pOutSize に動画ファイルの動画データ部分のサイズが代入されます。
    //! @pre pOutSize != nullptr
    //! @retval ResultAlbumNotFound handle が無効です。
    //! @details
    //! 動画ファイルの動画データ部分のサイズを取得します。
    //! ReadDataFromAlbumMovieReadStream() では先頭からこの API で取得されるサイズの範囲で有効なデータが取得できます。
    //!
    //! @see GetAlbumMovieReadStream()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    nn::Result GetAlbumMovieReadStreamDataSize(int64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

    //! @brief 動画ファイルの動画データ部分を読み込みます。
    //! @param[out] pOutReadSize 読込んだバイト数を受け取る変数のポインタです。
    //! @param[in]  buffer       読込んだデータを受け取るバッファです。
    //! @param[in]  size         読込むデータの大きさ（バイト）です。
    //! @param[in]  handle       動画読込ストリームのハンドルです。
    //! @param[in]  offset       データの読込の開始位置の動画読込ストリーム中のオフセットです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @pre pOutReadSize != nullptr
    //! @pre size >= 0
    //! @pre size % AlbumMovieDataUnitSize == 0
    //! @pre offset >= 0
    //! @pre offset % AlbumMovieDataUnitSize == 0
    //! @pre size == 0 || buffer != nullptr
    //! @retval ResultAlbumNotFound handle が不正です。
    //! @details
    //! 動画ファイルの動画データ部分を読み込みます。
    //! オフセット及びサイズはともに AlbumMovieDataUnitSize の整数倍でなければなりません。
    //! 動画データの大きさは GetAlbumMovieReadStreamDataSize() で取得します。
    //! 読込範囲が動画データの大きさを超える場合、範囲外の部分はゼロクリアされます。
    //!
    //! 任意のオフセット・サイズでデータ部分を読込む場合、 nn::capsrv::movie::MovieReaderFileSystem を利用します。
    //!
    //! @see GetAlbumMovieReadStream()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    nn::Result ReadDataFromAlbumMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT;

    //! @brief 動画ファイルの静止画データ部分のサイズを取得します。
    //! @param[out] pOutSize 静止画データ部分のサイズ（バイト）を受け取る変数のポインタです。
    //! @param[in]  handle   動画読込ストリームのハンドルです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @post 成功した場合、 *pOutSize に動画ファイルの静止画データ部分のサイズが代入されます。
    //! @pre pOutSize != nullptr
    //! @retval ResultAlbumNotFound handle が無効です。
    //! @details
    //! 動画ファイルの静止画データ部分のサイズを取得します。
    //! ReadImageDataFromAlbumMovieReadStream() では先頭からこの API で取得されるサイズの範囲で有効なデータが取得できます。
    //!
    //! @see GetAlbumMovieReadStream()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    nn::Result GetAlbumMovieReadStreamImageDataSize(int64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

    //! @brief 動画ファイルの静止画データ部分を読み込みます。
    //! @param[out] pOutReadSize 読込んだバイト数を受け取る変数のポインタです。
    //! @param[in]  buffer       読込んだデータを受け取るバッファです。
    //! @param[in]  size         読込むデータの大きさ（バイト）です。
    //! @param[in]  handle       動画読込ストリームのハンドルです。
    //! @param[in]  offset       データの読込の開始位置の動画読込ストリーム中のオフセットです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @pre pOutReadSize != nullptr
    //! @pre size >= 0
    //! @pre offset >= 0
    //! @pre size == 0 || buffer != nullptr
    //! @retval ResultAlbumNotFound handle が不正です。
    //! @details
    //! 動画ファイルの静止画データ部分を読み込みます。
    //! 静止画データの大きさは GetAlbumMovieReadStreamDataSize() で取得します。
    //! 読込範囲が静止画データの大きさを超える場合、範囲外の部分はゼロクリアされます。
    //!
    //! @see GetAlbumMovieReadStream()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    nn::Result ReadImageDataFromAlbumMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT;

    //! @brief 動画ファイルの属性を読み込みます。
    //! @param[out] pOutAttribute 読込んだ属性情報を受け取る変数のポインタです。
    //! @param[in]  handle        動画読込ストリームのハンドルです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @pre pOutAttribute != nullptr
    //! @retval ResultAlbumNotFound handle が不正です。
    //! @details
    //! 動画ファイルの属性情報を読み込みます。
    nn::Result ReadAttributeFromAlbumMovieReadStream(ScreenShotAttribute* pOutAttribute, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

    //! @brief 動画読込ストリームの内部エラーの値を取得します。
    //! @param[in] handle 動画読込ストリームのハンドルです。
    //! @pre アルバムへのアクセス機能が初期化されている。
    //! @details
    //! 動画読込ストリームの内部エラーの値が返ります。
    //! ストリームが正常な場合、 nn::ResultSuccess() が返ります。
    nn::Result GetAlbumMovieReadStreamBrokenReason(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

//! @}

}}
