﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  アプリケーション独自のアルバムファイル操作関連機能に関する定義
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/audio/audio_SampleFormat.h>
#include <nn/album/album_CommonApi.h>
#include <nn/album/album_ImageSize.h>
#include <nn/album/album_ImageOrientation.h>
#include <nn/album/album_ExtraMediaFileAccess.private.h>

namespace nn{ namespace album{

    class MovieMakerMovieParameter
    {
    public:
        //! @brief 既定の値を取得します。
        static MovieMakerMovieParameter GetDefaultValue() NN_NOEXCEPT;

        //! @brief デフォルトコンストラクタです。規定の値に設定されます。
        MovieMakerMovieParameter() NN_NOEXCEPT;

        //! @brief 画像のビットレート (bps) を取得します。
        int GetVideoBitRate() const NN_NOEXCEPT;

        //! @brief 画像のビットレート (bps) を設定します。
        void SetVideoBitRate(int value) NN_NOEXCEPT;

        //! @brief 画像の幅 (px) を取得します。
        int GetVideoWidth() const NN_NOEXCEPT;

        //! @brief 画像の幅 (px) を設定します。
        //! @details
        //! 1280 固定です。
        void SetVideoWidth(int value) NN_NOEXCEPT;

        //! @brief 画像の高さ (px) を取得します。
        int GetVideoHeight() const NN_NOEXCEPT;

        //! @brief 画像の高さ (px) を設定します。
        //! @details
        //! 720 固定です。
        void SetVideoHeight(int value) NN_NOEXCEPT;

        //! @brief 画像のフレームレート (fps) を取得します。
        int GetVideoFrameRate() const NN_NOEXCEPT;

        //! @brief 画像のフレームレート (fps) を設定します。
        //! @details
        //! 30 または 60 を指定します。
        void SetVideoFrameRate(int value) NN_NOEXCEPT;

        //! @brief 画像のキーフレーム間隔 (frames) を取得します。
        int GetVideoKeyFrameInterval() const NN_NOEXCEPT;

        //! @brief 画像のキーフレーム間隔 (frames) を設定します。
        //! @details
        //! 値にはキーフレームを含みます。
        //! 1 以上の値を指定します。
        void SetVideoKeyFrameInterval(int value) NN_NOEXCEPT;

        //! @brief 画像の回転方向をを設定します。
        void SetVideoImageOrientation(ImageOrientation orientation) NN_NOEXCEPT;

        //! @brief 音声のビットレート (bps) を取得します。
        int GetAudioBitRate() const NN_NOEXCEPT;

        //! @brief 音声のビットレート (bps) を設定します。
        void SetAudioBitRate(int value) NN_NOEXCEPT;

        //! @brief 音声のサンプルレート (samples/sec) を取得します。
        int GetAudioSampleRate() const NN_NOEXCEPT;

        //! @brief 音声のサンプルレート (samples/sec) を設定します。
        void SetAudioSampleRate(int value) NN_NOEXCEPT;

        //! @brief 音声のチャンネル数を取得します。
        int GetAudioChannelCount() const NN_NOEXCEPT;

        //! @brief 音声のチャンネル数を設定します。
        //! @details
        //! 2 固定です。
        void SetAudioChannelCount(int value) NN_NOEXCEPT;

        //! @brief 音声のサンプルフォーマットを取得します。
        nn::audio::SampleFormat GetAudioSampleFormat() const NN_NOEXCEPT;

        //! @brief 音声のサンプルフォーマットを設定します。
        //! @details
        // nn::audio::SampleFormat_PcmInt16 固定です。
        void SetAudioSampleFormat(nn::audio::SampleFormat value) NN_NOEXCEPT;

    private:
        NN_ALIGNAS(8) char m_Data[128] NN_IS_UNUSED_MEMBER;
    };

    //! @brief MovieMaker を初期化するために必要なワークメモリのサイズ（バイト）を取得します。
    size_t GetMovieMakerRequiredWorkMemorySize() NN_NOEXCEPT;

    //! @brief MovieMaker を初期化するために必要なワークメモリのアライメント（バイト）を取得します。
    size_t GetMovieMakerRequiredWorkMemoryAlignment() NN_NOEXCEPT;

    //! @brief MovieMaker が初期化されているかを取得します。
    bool IsMovieMakerInitialized() NN_NOEXCEPT;

    //! @brief MovieMaker が動作中かを取得します。
    bool IsMovieMakerRunning() NN_NOEXCEPT;

    //! @brief MovieMaker を初期化します。
    //!
    //! @pre !IsMovieMakerInitialized()
    //! @post IsMovieMakerInitialized()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! MovieMaker を初期化します。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result InitializeMovieMaker(void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

    //! @brief MovieMaker を破棄します。
    //!
    //! @pre IsMovieMakerInitialzied()
    //! @post !IsMovieMakerInitialized()
    //! @post !IsMovieMakerRunning()
    //!
    //! @details
    //! MovieMaker を破棄します。
    //! MovieMaker が動作中だった場合、中断されます。
    void FinalizeMovieMaker() NN_NOEXCEPT;

    //! @brief MovieMaker で使用する NativeWindow を取得します。
    //!
    //! @param[out] pOutNativeWindow NativeWindow を受け取る変数のポインタ
    //!
    //! @pre IsMovieMakerInitialized()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! この NativeWindow を使用してグラフィクス API からバッファを提出することでエンコーダに画像が入力されます。
    //! この NativeWindow は Vsync に同期していないため、システムの負荷により動作速度が変動します。
    //! 描画ループを一定の速度で回す場合にはタイマーなどを用いて実時間と同期を取ってください。
    //!
    //! 動画に記録される画像の長さは StartMovieMaker で指定したフレームレートと
    //! NativeWindow にバッファを提出した回数から計算されます。
    //!
    //! この NativeWindow は FinalizeMovieMaker() を呼び出すまで有効です。
    //! MovieMaker が動作中でない場合、提出したバッファはエンコードされずに返却されます。
    //! FinalizeMovieMaker() の呼出後にこの NativeWindow に対して操作を行った場合の動作は未定義です。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result GetMovieMakerNativeWindow(nn::vi::NativeWindowHandle* pOutNativeWindow) NN_NOEXCEPT;

    //! @brief 動画の作成開始前の事前チェックを行ないます。
    //!
    //! @param[in] size  動画を保存するのに必要とされるストレージの空き容量
    //!
    //! @pre  IsMovieMakerInitialized()
    //! @pre  !IsMovieMakerRunning()
    //! @post !IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の作成を開始するにあたって、必要最低限の事前チェックを行います。
    //! size には動画を保存する際に必要とされるであろうストレージの空き容量を
    //! 概算値で指定して下さい。
    //!
    //! 本 API で ResultSuccess が返る場合のみ album::StartMovieMaker() 以降の
    //! 動画作成処理へ進んで下さい。ただし、本 API で ResultSuccess が返っても
    //! 後続の動画作成処理の過程でエラーとなることがあります。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result PrecheckToStartMovieMaker(uint64_t size) NN_NOEXCEPT;

    //! @brief 動画の作成を開始します。
    //!
    //! @pre IsMovieMakerInitialized()
    //! @pre PrecheckToStartMovieMaker(size_t size).IsSuccess() == true
    //! @pre !IsMovieMakerRunning()
    //! @post IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の作成を開始します。
    //!
    //! 動画は本体アルバムに保存されます。
    //! 動画ファイルのパラメータを param で指定します。
    //!
    //! この API の呼出後 NativeWindow に提出された画像バッファがエンコードされて保存されます。
    //! それ以前に提出されていた画像バッファはエンコードされずに返却されます。
    //!
    //! 動画の作成を終了させるには FinishMovieMaker() または AbortMovieMaker() のいずれかを呼び出します。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result StartMovieMaker(const MovieMakerMovieParameter& param) NN_NOEXCEPT;

    //! @brief 動画の作成を終了します。
    //!
    //! @pre IsMovieMakerRunning()
    //! @post !IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の作成を終了します。
    //!
    //! この API の呼出より前に NativeWindow に提出されたバッファがエンコードされて保存されます。
    //! それ以降に提出されたバッファはエンコードされずに返却されます。
    //!
    //! 保存中にエラーが発生していた場合、エラーを返します。
    //! エラーが発生した場合、動画の作成は中断され保存中のファイルは破棄されます。追加で AbortMovieMaker() を呼び出す必要はありません。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result FinishMovieMaker() NN_NOEXCEPT;

    //! @brief 動画の作成を終了します。
    //!
    //! @param[in] pUserData    動画ファイルに埋め込む任意データです。
    //! @param[in] userDataSize 動画ファイルに埋め込む任意データのサイズです。
    //!
    //! @pre IsMovieMakerRunning()
    //! @pre userDataSize <= nn::album::AlbumUserDataSizeMax
    //! @post !IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の作成を終了します。
    //! 機能的には nn::album::FinishMovieMaker() と等価です。
    //!
    //! 本関数では動画ファイルにプログラムとして独自に埋め込んでおきたい
    //! 任意のデータを pUserData と userDataSize で指定することができます。
    //! システムは本 API で設定される任意データ列の内容には一切関与しません。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result FinishMovieMaker(const void* pUserData, size_t userDataSize) NN_NOEXCEPT;

    //! @brief サムネイル画像を指定して、動画の作成を終了します。
    //!
    //! @param[in] pUserData      動画ファイルに埋め込む任意データです。
    //! @param[in] userDataSize   動画ファイルに埋め込む任意データのサイズです。
    //! @param[in] pImageData     サムネイルとして保存する画像データです。
    //! @param[in] imageDataSize  サムネイル画像データの大きさ（バイト）です。
    //! @param[in] imageSize      pImageData で指定した画像のサイズです。
    //!
    //! @pre IsMovieMakerRunning()
    //! @pre userDataSize <= nn::album::AlbumUserDataSizeMax
    //! @pre imageSize == nn::album::ImageSize_1280x720
    //! @post !IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の作成を終了します。
    //! 機能的には nn::album::FinishMovieMaker() と等価です。
    //!
    //! 本関数では動画ファイルにプログラムとして独自に埋め込んでおきたい
    //! 任意のデータを pUserData と userDataSize で指定することができます。
    //! システムは本 API で設定される任意データ列の内容には一切関与しません。
    //!
    //! また、pImageData, imageDataSize, imageSize でサムネイル画像として
    //! 埋め込む画像を指定します。imageSize には nn::album::ImageSize_1280x720
    //! を指定して下さい。ここで指定した画像は以下の API で取得できます。
    //!
    //!   - nn::album::LoadAlbumImage()
    //!   - nn::album::LoadAlbumThumbnailImage()
    //!
    //! pImageData に渡すべき画像のフォーマットは sRGB 色空間で表現された
    //! 各チャンネルが 8 ビットの RGBA 画像です。イメージフォーマットとして
    //! nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb を指定し、タイルモードを
    //! nn::gfx::TileMode_Linear とした左上原点・パディングなし
    //! （ stride == 4 * GetImageWidth(screenshotSize) ）の画像データに
    //! 相当します。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result FinishMovieMaker(
        const void* pUserData,
        size_t userDataSize,
        const void* pImageData,
        size_t imageDataSize,
        ImageSize imageSize
        ) NN_NOEXCEPT;

    //! @brief 動画の作成を中断します。
    //!
    //! @pre IsMovieMakerRunning()
    //! @post !IsMovieMakerRunning()
    //!
    //! @details
    //! 動画の作成を中断します。
    //! 保存中のファイルは破棄されます。
    void AbortMovieMaker() NN_NOEXCEPT;

    //! @brief 動画の作成中にエラーが発生していないか調べます。
    //!
    //! @pre IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! エラーが発生している場合、そのエラーを返します。
    //! エラーが発生していない場合、成功を返します。
    //!
    //! エラーが発生した場合、 AbortMovieMaker() により動画の作成を中断してください。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result CheckMovieMakerError() NN_NOEXCEPT;

    //! @brief 動画の音声データをエンコードします。
    //!
    //! @param[in] buffer サンプルデータのバッファ
    //! @param[in] size   buffer の大きさ（バイト）
    //!
    //! @pre IsMovieMakerRunning()
    //!
    //! @return nn::Result 型で結果を返します。
    //!
    //! @details
    //! 動画の音声のサンプルデータをエンコーダに入力します。
    //!
    //! 入力する音声のパラメータは StartMovieMaker で指定した値です。
    //! サンプルデータのフォーマットは nn::audio::AudioOutBuffer と同様です。
    //!
    //! 動画に記録される音声の長さは StartMovieMaker で指定したサンプルレートと
    //! エンコーダに入力したサンプルデータの総サンプル数から計算されます。
    //!
    //! この API は buffer で与えたすべてのサンプルデータをエンコーダに入力するまでブロックします。
    //!
    //! 返値が nn::Result::IsSuccess() == true でない場合は、
    //! その値を使ってエラービューアを起動することでユーザに告知できます。
    //!
    nn::Result EncodeMovieMakerAudioSample(const void* buffer, size_t size) NN_NOEXCEPT;


}}
