﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  アルバムファイル読み込みに関する定義
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/album/album_AlbumFileEntry.private.h>
#include <nn/album/album_AlbumFileAccess.h>

namespace nn { namespace album {

//! @brief アルバムの静止画のデータサイズ最大値です。
const uint64_t AlbumScreenShotImageDataSize = 4 * 1280 * 720;

//! @brief アルバムの静止画データの取得に必要なワークメモリのサイズです。
const size_t RequiredWorkMemorySizeToLoadImage = 500 * 1024;


//! @brief アルバムのサムネイル画像のデータサイズです。
const uint64_t AlbumThumbnailImageDataSize = 4 * 320 * 180;

//! @brief アルバムのサムネイル画像の取得に必要なワークメモリのサイズです。
const size_t RequiredWorkMemorySizeToLoadThumbnailImage = 40 * 1024;


//! @brief 動画ファイルの読込みストリームを扱うためのハンドルです。
//!
struct MovieStreamHandle
{
    uint64_t value;
};

//! @brief 動画ファイルのデータ読込みを行なう際の読込み単位を表す定数です。
//!
//! @details
//! 動画ファイルのデータを読込む際のオフセットとサイズは、ともにこの値の
//! 整数倍を指定しなければなりません。
//!
const uint64_t MovieFileDataUnitSize = 256 * 1024;


//! @name アルバムの静止画ファイル関連機能
//! @{

//----------------------------------------------------------------------------
//! @brief アルバムの静止画ファイルのファイルリストを取得します。
//!
//! @param[out] pOutCount     取得した静止画ファイル数の格納先ポインタ
//! @param[out] buffer        取得した静止画ファイルリストを受け取る配列
//! @param[in]  bufferLength  buffer で渡す配列の長さ（要素数）
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre bufferLength >= 0
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! アルバムに保存されている静止画ファイルのファイルリストを
//! buffer が指す AlbumFileEntry 配列に取得します。
//! buffer に書き込まれる要素の順番は保証されません。
//!
//! bufferLength が取得可能な静止画ファイル数よりも大きな値だった場合は、
//! 0 から (*pOutCount - 1) 番目の要素に値が書き込まれます。@n
//! bufferLength が取得可能な静止画ファイル数よりも小さな値だった場合は、
//! buffer には bufferLength 個の要素だけが書き込まれます。
//!
//! アルバムに保存可能な静止画ファイルの最大数については別途マニュアルを
//! 参照して下さい。また、将来のファームウェアでは最大数が増加することも
//! 考えられるため、本関数の buffer および bufferLength には、
//! アプリケーションにとって都合のよいバッファ容量を渡すようにして下さい。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
Result GetAlbumScreenshotFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength) NN_NOEXCEPT;

//! @}


//! @name アルバムの動画ファイル関連機能
//! @{

//----------------------------------------------------------------------------
//! @brief アルバムの動画ファイルのファイルリストを取得します。
//!
//! @param[out] pOutCount     取得した動画ファイル数の格納先ポインタ
//! @param[out] buffer        取得した動画ファイルリストを受け取る配列
//! @param[in]  bufferLength  buffer で渡す配列の長さ（要素数）
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre bufferLength >= 0
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! アルバムに保存されている動画ファイルのファイルリストを
//! buffer が指す AlbumFileEntry 配列に取得します。
//! buffer に書き込まれる要素の順番は保証されません。
//!
//! bufferLength が取得可能な動画ファイル数よりも大きな値だった場合は、
//! 0 から (*pOutCount - 1) 番目の要素に値が書き込まれます。@n
//! bufferLength が取得可能な動画ファイル数よりも小さな値だった場合は、
//! buffer には bufferLength 個の要素だけが書き込まれます。
//!
//! アルバムに保存可能な動画ファイルの最大数については別途マニュアルを
//! 参照して下さい。また、将来のファームウェアでは最大数が増加することも
//! 考えられるため、本関数の buffer および bufferLength には、
//! アプリケーションにとって都合のよいバッファ容量を渡すようにして下さい。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
Result GetAlbumMovieFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength) NN_NOEXCEPT;

//! @}


//! @name アルバムの静止画・動画ファイル共通機能
//! @{

//! @brief 静止画もしくは動画ファイルに記録された画像データを読み込みます。
//!
//! @param[out] pOutWidth      読み込んだ画像データの横幅ピクセル数の格納先
//! @param[out] pOutHeight     読み込んだ画像データの高さピクセル数の格納先
//! @param[out] pBuffer        画像データを受け取るバッファ
//! @param[in]  bufferSize     画像データを受け取るバッファサイズ（バイト）
//! @param[in]  pWorkBuffer    作業用バッファ
//! @param[in]  workBufferSize 作業用バッファサイズ（バイト）
//! @param[in]  entry          対象ファイルの識別子
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre bufferSize >= nn::album::AlbumScreenShotImageDataSize
//! @pre workBufferSize >= nn::album::RequiredWorkMemorySizeToLoadImage
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! entry で指定された静止画ファイルもしくは動画ファイルに記録されている
//! 画像データをデコードし、pBuffer と bufferSize で指定された
//! バッファ領域に読み込みます。
//!
//! 取得される画像データは、sRGB 色空間で表現された各チャンネルが
//! 8 ビットの RGBA 画像フォーマットで、左上原点、パディングなしの
//! 線形タイリング画像 (stride = 4 * 横幅, size = 4 * 横幅 * 高さ) です。
//! デコード時のオプションには既定値が使用されます。
//!
//! 本関数は対象ファイルに静止画ファイルだけでなく、動画ファイルを
//! 指定することも可能です。その場合、動画ファイルに埋め込まれた
//! 静止画の画像データを読み込みます。
//!
//! pBuffer には画像データを受け取るバッファを渡します。pBuffer には
//! CPU キャッシュが有効な GPU のメモリプール上のメモリを渡すことができます。
//!
//! bufferSize が画像データサイズよりも大きかった場合、
//! pBuffer の先頭から画像データを書き込み、以降は 0 埋めされます。
//! bufferSize が画像データサイズよりも小さかった場合、
//! エラーを示す Result を返します。
//!
//! pWorkBuffer には GPU のメモリプール上のメモリを渡すことはできません。
//! workBufferSize が必要な大きさに満たなかった場合、
//! エラーを示す Result を返します。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
Result LoadAlbumImage(
    int* pOutWidth,
    int* pOutHeight,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT;

//! @brief 静止画もしくは動画ファイルに記録されたサムネイル画像データを読み込みます。
//!
//! @param[out] pOutWidth      読み込んだサムネイル画像の横幅ピクセル数の格納先
//! @param[out] pOutHeight     読み込んだサムネイル画像の高さピクセル数の格納先
//! @param[out] pBuffer        サムネイル画像データを受け取るバッファ
//! @param[in]  bufferSize     サムネイル画像データを受け取るバッファサイズ
//! @param[in]  pWorkBuffer    作業用バッファです。
//! @param[in]  workBufferSize 作業用バッファの大きさ（バイト）です。
//! @param[in]  entry          対象ファイルの識別子
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre bufferSize >= nn::album::AlbumThumbnailImageDataSize
//! @pre workBufferSize >= nn::album::RequiredWorkMemorySizeToLoadThumbnailImage
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! entry で指定された静止画ファイルもしくは動画ファイルに記録されている
//! サムネイル画像をデコードし、pBuffer と bufferSize で指定された
//! バッファ領域に読み込みます。
//!
//! 取得される画像データは、sRGB 色空間で表現された各チャンネルが
//! 8 ビットの RGBA 画像フォーマットで、左上原点、パディングなしの
//! 線形タイリング画像 (stride = 4 * 横幅, size = 4 * 横幅 * 高さ) です。
//! デコード時のオプションには既定値が使用されます。
//!
//! 本関数は対象ファイルに静止画ファイルだけでなく、動画ファイルを
//! 指定することも可能です。その場合、動画ファイルに埋め込まれた
//! サムネイル画像データを読み込みます。
//!
//! pBuffer にはサムネイル画像を受け取るバッファを渡します。pBuffer には
//! CPU キャッシュが有効な GPU のメモリプール上のメモリを渡すことができます。
//!
//! bufferSize がサムネイル画像のデータサイズよりも大きかった場合、
//! pBuffer の先頭からサムネイル画像を書き込み、以降は 0 埋めされます。
//! bufferSize がサムネイル画像のデータサイズよりも小さかった場合、
//! エラーを示す Result を返します。
//!
//! pWorkBuffer には GPU のメモリプール上のメモリを渡すことはできません。
//! workBufferSize が必要な大きさに満たなかった場合、
//! エラーを示す Result を返します。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
Result LoadAlbumThumbnailImage(
    int* pOutWidth,
    int* pOutHeight,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT;


//! @name 動画ファイルの読み込みストリーム操作機能
//! @{

//! @brief  動画ファイルの読み込みストリームを開きます。
//!
//! @param[out] pOutHandle  動画ファイルの読み込みストリームハンドルの格納先
//! @param[in]  entry       対象動画ファイルの識別子
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 動画ファイルの読み込みストリームを開き、そのハンドルを返します。
//! 読み込みが終わった後は nn::album::CloseAlbumMovieStream() で
//! ストリームを閉じる必要があります。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
Result OpenAlbumMovieStream(MovieStreamHandle* pOutHandle, const AlbumFileEntry& entry) NN_NOEXCEPT;


//! @brief 動画ファイルの読み込みストリームを閉じます。
//!
//! @param[in] handle   動画ファイルの読み込みストリームハンドル
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//! @pre handle が有効なストリームハンドルである
//!
//! @details
//! 動画ファイルの読み込みストリームを閉じます。
//!
//! @see nn::album::OpenAlbumMovieStream()
//!
void CloseAlbumMovieStream(MovieStreamHandle handle) NN_NOEXCEPT;


//! @brief 動画ファイルの動画データ部分のサイズを取得します。
//!
//! @param[out] pOutSize    動画データ部分のサイズ（バイト）の格納先
//! @param[in]  handle      動画ファイルの読み込みストリームハンドル
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//! @pre handle が有効なストリームハンドルである
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 動画ファイルの動画データ部分のサイズを取得します。
//! nn::album::ReadAlbumMovieStream() では、
//! 先頭からこの API で取得されるサイズの範囲で有効なデータを取得できます。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
//! @see nn::album::OpenAlbumMovieStream()
//!
Result GetAlbumMovieStreamSize(uint64_t* pOutSize, MovieStreamHandle handle) NN_NOEXCEPT;


//! @brief 動画ファイルの動画データ部分を読み込みます。
//!
//! @param[out] pOutReadSize 読み込んだバイト数の格納先
//! @param[in]  buffer       読み込んだデータを受け取るバッファ
//! @param[in]  size         読み込むデータの大きさ（バイト）
//! @param[in]  handle       動画ファイルの読み込みストリームハンドル
//! @param[in]  offset       データ読み込みの開始位置オフセット
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//! @pre handle が有効なストリームハンドルである
//! @pre size >= 0
//! @pre size % nn::album::MovieFileDataUnitSize == 0
//! @pre offset >= 0
//! @pre offset % nn::album::MovieFileDataUnitSize == 0
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 動画ファイルの動画データ部分を読み込みます。
//! offset 及び size はともに nn::album::MovieFileDataUnitSize の
//! 整数倍でなければなりません。動画データの実際の大きさは
//! nn::album::GetAlbumMovieStreamSize() で取得できます。
//!
//! offset と size で指定される読み込み範囲が実際の動画データの大きさを
//! 超える場合は、範囲外の部分はゼロで埋められます。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
//! @see nn::album::OpenAlbumMovieStream()
//!
Result ReadAlbumMovieStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamHandle handle, int64_t offset) NN_NOEXCEPT;


//! @}

}}  // namespace nn::album
