﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SettingsManager
{
    using System;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using Properties;

    /// <summary>
    /// リソースを管理します。
    /// </summary>
    internal static class ResourceManager
    {
        private const string SettingsManagerName = "SettingsManager";

        private static readonly Lazy<string>
            assemblyDirectoryPath = new Lazy<string>(GetAssemblyDirectoryPath);

        private static readonly Lazy<string>
            resourceDirectoryPath = new Lazy<string>(GetResourceDirectoryPath);

        private static readonly Lazy<string>
            resourceLocalizerPath = new Lazy<string>(GetResourceLocalizerPath);

        private static readonly Lazy<string>
            runOnTargetPath = new Lazy<string>(GetRunOnTargetPath);

        private static readonly Lazy<string[]>
            enginePaths = new Lazy<string[]>(GetEnginePaths);

        /// <summary>
        /// リソースローカライズ用 DLL のファイルパスです。
        /// </summary>
        internal static string ResourceLocalizerPath
        {
            get
            {
                return resourceLocalizerPath.Value;
            }
        }

        /// <summary>
        /// RunOnTarget.exe のファイルパスです。
        /// </summary>
        internal static string RunOnTargetPath
        {
            get
            {
                return runOnTargetPath.Value;
            }
        }

        /// <summary>
        /// 設定マネージャエンジンのファイルパスを返します。
        /// </summary>
        /// <param name="platform">プラットフォームです。</param>
        /// <returns>設定マネージャエンジンのファイルパスです。</returns>
        internal static string GetEnginePath(Platform platform)
        {
            var paths = enginePaths.Value;

            string extension = GetExecutableExtension(platform);

            var auths = new[] { "Full", "Core" };

            foreach (var auth in auths)
            {
                var suffix = $"{auth}{extension}";

                string path = paths.LastOrDefault(x => x.EndsWith(suffix));

                if (path != null)
                {
                    return path;
                }
            }

            throw new SettingsManagerException(
                string.Format(
                    Resources.ErrorFileNotFound,
                    $"{SettingsManagerName}{auths.Last()}{extension}"));
        }

        private static string GetExecutableExtension(Platform platform)
        {
            switch (platform)
            {
                case Platform.Horizon:
                    return ".nsp";

                case Platform.Windows:
                default:
                    return ".exe";
            }
        }

        private static string GetAssemblyDirectoryPath()
        {
            return Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
        }

        private static string GetResourceDirectoryName()
        {
            return Path.Combine("Resources", SettingsManagerName);
        }

        private static string GetResourceDirectoryPath()
        {
            var path = Path.Combine(
                assemblyDirectoryPath.Value, GetResourceDirectoryName());

            if (Directory.Exists(path))
            {
                return path;
            }
            else
            {
                throw new SettingsManagerException(
                    string.Format(
                        Resources.ErrorDirectoryNotFound,
                        GetResourceDirectoryName()));
            }
        }

        private static string GetResourceLocalizerPath()
        {
            return Path.Combine(
                assemblyDirectoryPath.Value,
                "en", $"{SettingsManagerName}.resources.dll");
        }

        private static string GetRunOnTargetPath()
        {
            string path = Path.GetFullPath(Path.Combine(
                assemblyDirectoryPath.Value, "..", "RunOnTarget.exe"));

            if (File.Exists(path))
            {
                return path;
            }
            else
            {
                throw new SettingsManagerException(
                    string.Format(
                        Resources.ErrorFileNotFound, Path.GetFileName(path)));
            }
        }

        private static string[] GetEnginePaths()
        {
            return Directory.EnumerateFiles(
                resourceDirectoryPath.Value,
                $"{SettingsManagerName}*",
                SearchOption.AllDirectories).ToArray();
        }
    }
}
