﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SettingsManager.Command
{
    using System.IO;
    using Nintendo.ToolFoundation.CommandLine;
    using Properties;

    /// <summary>
    /// Import コマンドを提供します。
    /// </summary>
    internal sealed class Import : ICommand
    {
        private CommonOptions commonArgs = new CommonOptions();

        private string filePath = string.Empty;

        /// <summary>
        /// コマンドが選択されたか否かを表す値を返します。
        /// </summary>
        public bool IsSelected { get; private set; }

        /// <summary>
        /// コマンドラインパーサにコマンドを登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        public void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            var context = parser
                .AddCommand("Import", () => this.IsSelected = true)
                .SetDescription(Properties.Resources.ImportCommand);

            this.commonArgs.RegiserWith(context);

            context
                .AddValue(this.AcceptFilePathOption, 1)
                .SetValueName("path")
                .SetDescription(Resources.ImportValueFilePath)
                .SetRequired();
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        public void Execute()
        {
            var platform = this.commonArgs.Platform;

            var enginePath = ResourceManager.GetEnginePath(platform);

            var executer = new Executer(platform);

            executer.Target = this.commonArgs.Target;

            executer.Ignite(enginePath, new[] { "Import", this.filePath });
        }

        private void AcceptFilePathOption(string filePath)
        {
            filePath = Path.GetFullPath(filePath);

            if (File.Exists(filePath))
            {
                this.filePath = filePath;
            }
            else
            {
                throw new SettingsManagerException(
                    string.Format(
                        Resources.ErrorFileNotFound,
                        Path.GetFileName(filePath)));
            }
        }
    }
}
