﻿namespace MakeFirmwareDebugSettings
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using YamlDotNet.Core;
    using YamlDotNet.RepresentationModel;

    internal static class YamlParser
    {
        internal static IDictionary<string, object> ParseFile(string path)
        {
            var dict = new Dictionary<string, object>();

            using (var sr = new StreamReader(path, Encoding.UTF8))
            {
                var ys = new YamlStream();

                try
                {
                    ys.Load(sr);
                }
                catch (YamlException ex)
                {
                    throw new FirmwareDebugSettingsException(
                        string.Format("failed to load '{0}': {1}", path, ex.Message), ex);
                }

                if (ys.Documents.Count < 1)
                {
                    throw new FirmwareDebugSettingsException(
                        string.Format("document not found in '{0}'", path));
                }

                var rootMark = ys.Documents[0].RootNode.Start;

                var rootNode = ys.Documents[0].RootNode as YamlMappingNode;

                if (rootNode == null)
                {
                    throw new FirmwareDebugSettingsException(
                        GetUnexpectedNodeTypeErrorMessage(path, rootMark));
                }

                foreach (var child in rootNode.Children)
                {
                    var valueMark = child.Value.Start;

                    var valueNode = child.Value as YamlScalarNode;

                    if (valueNode == null)
                    {
                        throw new FirmwareDebugSettingsException(
                            GetUnexpectedNodeTypeErrorMessage(path, valueMark));
                    }

                    var key = ((YamlScalarNode)child.Key).Value;

                    var value = valueNode.Value;

                    if (valueNode.Style != ScalarStyle.Plain)
                    {
                        dict[key] = value;
                    }
                    else if (new[] { "true", "True", "TRUE" }.Contains(value))
                    {
                        dict[key] = true;
                    }
                    else if (new[] { "false", "False", "FALSE" }.Contains(value))
                    {
                        dict[key] = false;
                    }
                    else if (Regex.IsMatch(value, @"^[-+]?(0|[1-9][0-9]*)$"))
                    {
                        dict[key] = Convert.ToInt32(value);
                    }
                    else if (Regex.IsMatch(value, @"^0[Xx][0-9A-Fa-f]+$"))
                    {
                        dict[key] = Convert.ToInt32(value, 16);
                    }
                    else
                    {
                        dict[key] = value;
                    }
                }
            }

            return dict;
        }

        private static string GetUnexpectedNodeTypeErrorMessage(string path, Mark mark)
        {
            return string.Format(
                "failed to load '{0}': unexpected node type (Line:{1}, Offset:{2})",
                path, mark.Line, mark.Column);
        }
    }
}
