﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using Nintendo.Foundation.IO;

namespace MakeRebootlessSystemUpdateVersion
{
    public class Arguments
    {
        [CommandLineOption('o', "output", Description = "出力するファイルパスを指定します", IsRequired = true)]
        public string OutputPath { get; set; }

        [CommandLineOption('t', "title-version", Description = "タイトルのバージョン値(32ビット)を指定します", IsRequired = true)]
        public string TitleVersion { get; set; }

        [CommandLineOption('r', "rebootless-version", Description = "再起動不要 NUP バージョン値(32ビット)を指定します", IsRequired = true)]
        public string RebootlessVersion { get; set; }

        [CommandLineOption('d', "display-version", Description = "再起動不要 NUP 表記バージョンを明示的に指定します(明示的な指定がない場合 rebootless-version を元に自動的に設定されます)", IsRequired = false)]
        public string DisplayVersion { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
            // コマンドライン引数の解釈
            Arguments parameters = new Arguments();
            if(!InterpretCommandLineArguments(args, out parameters))
            {
                Environment.Exit(-1);
            }

            // 作業ディレクトリの作成
            // 出力ファイルのディレクトリに作成する
            var workDirectory = new WorkDirectory(Directory.GetParent(parameters.OutputPath).FullName);

            // nmeta の作成
            var nmetaFilePath = workDirectory.GetPath() + @"\RebootlessSystemUpdateVersion.nmeta";
            {
                var nmeta = new RebootlessSystemUpdateVersionNmeta(UInt32.Parse(parameters.TitleVersion));
                nmeta.Save(nmetaFilePath);
            }

            // data の作成
            var dataDirectoryPath = workDirectory.GetPath() + @"\data";
            {
                var data = new RebootlessSystemUpdateVersionData(UInt32.Parse(parameters.RebootlessVersion), parameters.DisplayVersion);
                data.Save(dataDirectoryPath);
            }

            // AuthoringTool の実行
            var executor = new AuthoringExecutor();
            executor.CreateSystemDataNsp(parameters.OutputPath, nmetaFilePath, dataDirectoryPath);
        }

        static bool InterpretCommandLineArguments(string[] args, out Arguments parameters)
        {
            parameters = null;
            try
            {
                CommandLineParser.Default.ParseArgs<Arguments>(args, out parameters);
            }
            catch (ArgumentException)
            {
                return false;
            }

            // title-version の形式チェック
            try
            {
                UInt32.Parse(parameters.TitleVersion);
            }
            catch (FormatException)
            {
                Console.Error.WriteLine("\'title-version\' は数値を指定する必要があります。");
                return false;
            }
            catch (OverflowException)
            {
                Console.Error.WriteLine("\'title-version\' は 32bit 符号無し整数で表現できる範囲の数値を指定する必要があります。");
                return false;
            }

            // rebootless-version の形式チェック
            try
            {
                UInt32.Parse(parameters.RebootlessVersion);
            }
            catch (FormatException)
            {
                Console.Error.WriteLine("\'rebootless-version\' は数値を指定する必要があります。");
                return false;
            }
            catch (OverflowException)
            {
                Console.Error.WriteLine("\'rebootless-version\' は 32bit 符号無し整数で表現できる範囲の数値を指定する必要があります。");
                return false;
            }

            // display-version の形式チェック
            if (parameters.DisplayVersion != null && parameters.DisplayVersion.Length >= 32)
            {
                Console.Error.WriteLine("\'display-version\' は 32 文字未満の文字列を指定する必要があります。");
                return false;
            }

            return true;
        }
    }

    public class WorkDirectory
    {
        private const string WorkDirectoryName = "_MakeRebootlessSystemUpdateVersion_WorkDirectory";

        public WorkDirectory(string path)
        {
            this.path = path + @"\" + WorkDirectoryName;
            Directory.CreateDirectory(this.path);
        }

        ~WorkDirectory()
        {
            Directory.Delete(this.path, true);
        }

        public string GetPath()
        {
            return this.path;
        }

        private string path;
    }

    public class AuthoringExecutor
    {
        private const string AuthoringToolDirectoryPath = @"Tools\CommandLineTools\AuthoringTool";
        private const string KeyconfigFileName = "AuthoringTool.repository.keyconfig.xml";
        private const int KeyIndex = 2;
        private const string SigloRootMarkFileName = "SigloRootMark";
        private const string NintendoSdkRootMarkFileName = "NintendoSdkRootMark";

        public void CreateSystemDataNsp(string outFilePath, string nmetaFilePath, string dataFilePath)
        {
            var args = string.Format("creatensp -o \"{0}\" --meta \"{1}\" --type \"{2}\" --data \"{3}\"", outFilePath, nmetaFilePath, "SystemData", dataFilePath);
            InternalExecuteAuthoringTool(args, GetSigloRoot() + AuthoringToolDirectoryPath + @"\" + KeyconfigFileName, KeyIndex.ToString());
        }

        private void InternalExecuteAuthoringTool(string args, string inKeyPath, string keyIndex)
        {
            if (inKeyPath != null && inKeyPath != string.Empty)
            {
                args += string.Format(" --keyconfig \"{0}\"", inKeyPath);
            }
            if (keyIndex != null && keyIndex != string.Empty)
            {
                args += string.Format(" --keyindex \"{0}\"", keyIndex);
            }

            using (var process = new Process())
            {
                var authoringToolDirPath = GetSigloRoot() + AuthoringToolDirectoryPath;
                process.StartInfo.FileName = Path.Combine(authoringToolDirPath, "AuthoringTool.exe");
                process.StartInfo.Arguments = args;

                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                process.Start();

                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    Console.Error.WriteLine("作成に失敗しました。");
                    Console.Error.WriteLine(process.StandardError.ReadToEnd());
                    throw new Exception("Creation failed");
                }
            }
        }


        private static string GetSigloRoot()
        {
            // ルートパスを確定させるために "～RootMark" ファイルを検索する
            var rootPath = FindSigloRoot(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location));
            if (rootPath == null)
            {
                // もし取得できない場合は、環境変数"NINTENDO_SDK_ROOT"を取得
                rootPath = Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
                if (rootPath == null)
                {
                    throw new InvalidDataException("Rootパスを見つけられませんでした");
                }
            }

            if (rootPath.EndsWith(@"\") == false)
            {
                rootPath += @"\";
            }

            return rootPath;
        }

        private static string FindSigloRoot(string path)
        {
            string startPath = path;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                // "SigloRootMark"の存在確認
                var sigloRootMarkFilePath = Path.Combine(currentDirectory, SigloRootMarkFileName);
                if (File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                // "NintendoSdkRootMark"の存在確認
                var nintendoSdkRootMarkFilePath = Path.Combine(currentDirectory, NintendoSdkRootMarkFileName);
                if (File.Exists(nintendoSdkRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = Path.GetFullPath(Path.Combine(currentDirectory, ".."));
            }

            return null;
        }
    }

    public class RebootlessSystemUpdateVersionData
    {
        public RebootlessSystemUpdateVersionData(UInt32 version, string displayVersion)
        {
            this.version = version;

            if (displayVersion != null)
            {
                this.displayVersion = displayVersion;
            }
            else
            {
                this.displayVersion = version.ToString();
            }
        }

        /**
         * 以下の構造体のファイルを作成する
         *
         * struct RebootlessSystemUpdateVersion
         * {
         *    uint32_t version;           //!< 再起動不要 NUP のバージョン
         *    uint8_t reserved[28];
         *    char displayVersion[32];    //!< 再起動不要 NUP の表示バージョン
         * };
         */
        public void Save(string saveDirectoryPath)
        {
            Byte[] data = new Byte[64];

            Byte[] versionBytes = BitConverter.GetBytes(this.version);
            versionBytes.CopyTo(data, 0);

            Byte[] displayVersionBytes = System.Text.Encoding.ASCII.GetBytes(this.displayVersion);
            displayVersionBytes.CopyTo(data, 32);

            Directory.CreateDirectory(saveDirectoryPath);
            File.WriteAllBytes(saveDirectoryPath + @"\version", data);
        }

        private UInt32 version;
        private string displayVersion;
    }

    public class RebootlessSystemUpdateVersionNmeta
    {
        private const string Id = "0x0100000000000826";
        private const string Name = "RebootlessSystemUpdateVersion";

        public RebootlessSystemUpdateVersionNmeta(UInt32 titleVersion)
        {
            this.titleVersion = titleVersion;
        }

        public void Save(string saveFilePath)
        {
            var nmetaXmlDocument = new XmlDocument();

            var head = nmetaXmlDocument.CreateXmlDeclaration("1.0", "utf-8", null);
            nmetaXmlDocument.AppendChild(head);

            var root = nmetaXmlDocument.CreateElement("NintendoSdkMeta");
            nmetaXmlDocument.AppendChild(root);

            var systemData = nmetaXmlDocument.CreateElement("SystemData");
            root.AppendChild(systemData);

            Action<string, string> addSystemDataElementAction = (string inElementName, string inSetValue) =>
            {
                if (inSetValue != null)
                {
                    var element = nmetaXmlDocument.CreateElement(inElementName);
                    element.InnerText = inSetValue;
                    systemData.AppendChild(element);
                }
            };

            addSystemDataElementAction("Id", Id);
            addSystemDataElementAction("Name", Name);
            addSystemDataElementAction("Version", titleVersion.ToString());
            addSystemDataElementAction("ContentMetaAttribute", "Rebootless");

            nmetaXmlDocument.Save(saveFilePath);
        }

        private UInt32 titleVersion;
    }
}
