﻿using System;
using System.IO;
using System.IO.Compression;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using CommandUtility;
using MakeFirmwareArchive;

namespace MakeQspiBootImage
{
    public class MakeQspiBootImageArguments
    {
        [CommandLineOption('o', "output", Description = "output qspi-boot image path", IsRequired = true)]
        public string OutputImagePath { get; set; }

        [CommandLineOption('i', "input", Description = "input nfa path", IsRequired = true)]
        public string InputNfaPath { get; set; }

        [CommandLineOption("system-partition", Description = "input system parition image", IsRequired = false)]
        public string SystemPartitionPath { get; set; }

        [CommandLineOption("system-partition-alignment", Description = "input system parition image alignment", IsRequired = false)]
        public string SystemPartitionAlignment { get; set; }

        [CommandLineOption("size", Description = "system partition size(MB)", IsRequired = false, DefaultValue = 11)]
        public int SystemPartitionSize { get; set; }

        [CommandLineOption("paddingsize", Description = "padding size", IsRequired = false, DefaultValue = 0x500000)]
        public int PaddingSize { get; set; }

        [CommandLineOption("key", Description = "input key of fat image", IsRequired = true)]
        public string InputKeyFile { get; set; }
    }

    class Program
    {
        public static void ReadFirmwareArchive(MakeQspiBootImageArguments parameters)
        {
            var inputNfa = new FileInfo(parameters.InputNfaPath);

            using (var tempHolder = new TemporaryFileHolder("MakeQspiBootImage"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var workingDirectory = tempHolder.CreateTemporaryDirectory("WorkingDir");

                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputNfa);

                List<FileInfo> systemNspInfos = archiveDirectory.GetSystemNspList(extractedNfa);

                List<string> installList = new List<string>();
                foreach(var systemNsp in systemNspInfos)
                {
                    installList.Add("-i");
                    installList.Add(systemNsp.FullName);
                }

                MakeQspiBootImage(parameters, workingDirectory, installList, archiveDirectory);
            }
        }

        public static void MakeSystemPartition(FileInfo outputPath, List<string> installList, FirmwareArchiveDirectory archiveDirectory, int systemPartitionSize, FileInfo fatKey)
        {
            MakeInstalledFatImage(outputPath, systemPartitionSize, fatKey, installList);
        }

        public static void MakeQspiBootImage(MakeQspiBootImageArguments parameters, DirectoryInfo workingDirectory, List<string> installList, FirmwareArchiveDirectory archiveDirectory)
        {
            var outputFile = new FileInfo(parameters.OutputImagePath);

            var recoveryWriterProgram =
                archiveDirectory.GetRecoveryFileList("RecoveryWriter").First();

            var gptPartition =
                archiveDirectory.GetRecoveryFileList("gpt-recovery.img").First();

            var package1 = MakePackage1(workingDirectory, archiveDirectory);

            var package2 =
                archiveDirectory.GetNormalPackage2().First();

            var fatKey = new FileInfo(parameters.InputKeyFile);
            if (!fatKey.Exists)
            {
                throw new Exception(string.Format("fat key({0}) is not found.", fatKey.FullName));
            }

            var systemPartitionSize = parameters.SystemPartitionSize;
            var paddingSize = parameters.PaddingSize;

            FileInfo systemPartition;
            if (parameters.SystemPartitionPath == null)
            {
                systemPartition = new FileInfo(Path.Combine(workingDirectory.FullName, "systemPartition.img"));
                MakeSystemPartition(systemPartition, installList, archiveDirectory, systemPartitionSize, fatKey);
            }
            else
            {
                systemPartition = new FileInfo(parameters.SystemPartitionPath);
            }

            var parameterFile = new FileInfo(Path.Combine(workingDirectory.FullName, "params.bin"));
            var recoveryWriterParameter = new RecoveryWriterParameter(
                    recoveryWriterProgram, gptPartition, package1, package2, paddingSize, systemPartition.Length);
            recoveryWriterParameter.MakeParameterFile(parameterFile);

            var intermediate1 = new FileInfo(Path.Combine(workingDirectory.FullName, "Intermediate1.img"));

            ConcatenateFiles(intermediate1.FullName,
                new string[] {
                    recoveryWriterProgram.FullName,
                    parameterFile.FullName,
                    gptPartition.FullName,
                    package1.FullName,
                    package2.FullName,
                }
            );

            var intermediate2 = new FileInfo(Path.Combine(workingDirectory.FullName, "Intermediate2.img"));

            MakePaddedImage(intermediate2, intermediate1, paddingSize);

            var actualSystemPartitionSize = new FileInfo(systemPartition.FullName).Length;
            var systemPartitionAlignmentSize = ByteUnitExpression.Parse(parameters.SystemPartitionAlignment).Bytes;
            var systemPartitionPaddingSize = systemPartitionAlignmentSize - actualSystemPartitionSize;

            if (systemPartitionPaddingSize < 0)
            {
                throw new Exception($"[ERROR] SystemPartitionSize size is too large.\n" +
                                    $"    expected: size < {systemPartitionAlignmentSize}, {parameters.SystemPartitionAlignment}\n" +
                                    $"    actual: {actualSystemPartitionSize}");
            }

            using (var writer = outputFile.OpenWrite())
            {
                FileUtility.WriteToStream(writer, intermediate2);
                FileUtility.WriteToStreamWithPad(writer, new FileInfo(systemPartition.FullName), systemPartitionAlignmentSize);
            }
        }

        private static FileInfo MakePackage1(DirectoryInfo workingDirectory, FirmwareArchiveDirectory archiveDirectory)
        {
            var outputPackage1Path = new FileInfo(Path.Combine(workingDirectory.FullName, "Package1.img"));
            var bctPath = GetPackage1BctPath(archiveDirectory);
            var bootLoaderPath = GetPackage1BootLoaderPath(archiveDirectory);

            CommandUtility.SdkTool.Execute("MakePackage1.exe",
                "--output", outputPackage1Path.FullName,
                "--bct", bctPath.FullName,
                "--bootloader", bootLoaderPath.FullName);

            return outputPackage1Path;
        }

        private static FileInfo GetPackage1BootLoaderPath(FirmwareArchiveDirectory archiveDirectory)
        {
            var package1Files = archiveDirectory.GetNormalPackage1();

            try
            {
                return package1Files.Where((fileInfo) => fileInfo.Extension == ".bl" || fileInfo.Extension == ".bin").Single();
            }
            catch
            {
                throw new Exception(string.Format("Bootloader(.bl or .bin) file is not found."));
            }
        }

        private static FileInfo GetPackage1BctPath(FirmwareArchiveDirectory archiveDirectory)
        {
            var package1Files = archiveDirectory.GetNormalPackage1();

            try
            {
                return package1Files.Where((fileInfo) => fileInfo.Extension == ".bct").Single();
            }
            catch
            {
                throw new Exception(string.Format("Bct file(.bct) is not found."));
            }
        }

        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            MakeQspiBootImageArguments parameters = new MakeQspiBootImageArguments();
            if (CommandLineParser.Default.ParseArgs<MakeQspiBootImageArguments>(args, out parameters))
            {
                ReadFirmwareArchive(parameters);
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("エラー: {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void MakePaddedImage(FileInfo outputImagePath, FileInfo sourceImagePath, int paddingSize)
        {
            SdkTool.Execute(
                SdkPath.FindToolPath("Pad.exe", "Externals/NxSystemImages/Recovery/Pad.exe", "Externals/NxSystemImages/Recovery/Pad.exe"),
                new string[] {
                    "--input", sourceImagePath.FullName,
                    "--output", outputImagePath.FullName,
                    "--size", paddingSize.ToString(),
                }
            );
        }

        public static void ConcatenateFiles(string outputPath, string[] inputPathList)
        {
            try
            {
                foreach (var inputPath in inputPathList)
                {
                    if (!File.Exists(inputPath))
                    {
                        throw new Exception(string.Format("{0} is not found.", inputPath));
                    }
                }

                using (var outputFile = File.OpenWrite(outputPath))
                {
                    foreach (var inputPath in inputPathList)
                    {
                        using (var inputFile = File.OpenRead(inputPath))
                        {
                            inputFile.CopyTo(outputFile);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                Console.Error.WriteLine("error: {0}", e.Message);
                throw;
            }
        }

        public static void MakeInstalledFatImage(FileInfo outputPath, int partitionSize, FileInfo keyFile, List<string> installedPrograms)
        {
            var arguments = new List<string>
            {
                "-o", outputPath.FullName,
                "--size", partitionSize.ToString(),
                "--delete", "contents/meta",
                "--delete", "save"
            };

            if (keyFile != null)
            {
                var keyarg = new List<string>
                {
                    "--key-file",
                    keyFile.FullName
                };

                arguments = arguments.Concat(keyarg).ToList();
            }

            arguments = arguments.Concat(installedPrograms.ToArray()).ToList();

            SdkTool.Execute(
                SdkPath.FindToolPath("MakeInstalledFatImage.exe", "Tools/CommandLineTools/MakeInstalledFatImage.exe", "Tools/CommandLineTools/MakeInstalledFatImage.exe"),
                arguments.ToArray()
            );
        }
    }
}
