﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using CommandUtility;
using System.IO;
using Nintendo.Foundation.IO;
using MakeFirmwareArchive;

namespace MakeSignedSystemPartition
{
    public class MakeSignedSystemPartition
    {
        [CommandLineOption('o', "output", Description = "output qspi-boot image path", IsRequired = true)]
        public string OutputImagePath { get; set; }

        [CommandLineOption('i', "input", Description = "input nfa path", IsRequired = true)]
        public string InputNfaPath { get; set; }

        [CommandLineOption('a', "append", Description = "input package2 path", IsRequired = false, DefaultValue = null)]
        public string InputPackage2Path { get; set; }

        [CommandLineOption("size", Description = "system partition size(MB)", IsRequired = false, DefaultValue = 11)]
        public string PartitionSize { get; set; }

        [CommandLineOption("free-space", Description = "ensuree free space size(bytes)", IsRequired = false, DefaultValue = "0B")]
        public string EnsuredFreeSpace { get; set; }

        [CommandLineOption("key", Description = "input key for signed binary.", IsRequired = true)]
        public string InputKeyFile { get; set; }

        [CommandLineOption("enable-develop-image", Description = "create for develop image.", DefaultValue = false)]
        public bool EnableDevelopImage { get; set; }

        [CommandLineOption("encrypt", Description = "create encrypted system partition (for testing purpose only).", DefaultValue = false)]
        public bool Encrypt { get; set; }

        private FileInfo Package2 { get; set; }

        public int Run()
        {
            var inputNfa = new FileInfo(this.InputNfaPath);

            using (var tempHolder = new TemporaryFileHolder("MakeQspiBootImage"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var workingDirectory = tempHolder.CreateTemporaryDirectory("WorkingDir");

                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputNfa);

                List<FileInfo> systemNspInfos = archiveDirectory.GetSystemNspList(extractedNfa);

                List<string> installList = new List<string>();
                foreach (var systemNsp in systemNspInfos)
                {
                    installList.Add("-i");
                    installList.Add(systemNsp.FullName);
                }

                if (InputPackage2Path != null)
                {
                    Package2 = new FileInfo(InputPackage2Path);
                    EnableDevelopImage = true;
                }
                else if (EnableDevelopImage || Encrypt)
                {
                    Package2 = archiveDirectory.GetNormalPackage2().First();
                }

                var systemPartition = new FileInfo(Path.Combine(workingDirectory.FullName, "systemPartition.img"));
                MakeSystemPartition(systemPartition, installList, archiveDirectory);

                var signedSystemPartition = new FileInfo(OutputImagePath);
                MakeSignedBinary(signedSystemPartition, systemPartition, new FileInfo(InputKeyFile));
            }

            return 0;
        }

        private void MakeSignedBinary(FileInfo signedSystemPartition, FileInfo systemPartition, FileInfo keyFile)
        {
            if (EnableDevelopImage)
            {
                SdkTool.Execute(
                    "MakeSignedBinary.exe",
                    "sign",
                    "-o", signedSystemPartition.FullName,
                    "-i", systemPartition.FullName,
                    "-k", keyFile.FullName,
                    "-a", Package2.FullName
                    );
            }
            else
            {
                if (Encrypt)
                {
                    SdkTool.Execute(
                        "MakeSignedBinary.exe",
                        "sign",
                        "-o", signedSystemPartition.FullName,
                        "-i", systemPartition.FullName,
                        "-k", keyFile.FullName,
                        "-a", Package2.FullName
                        );
                }
                else
                {
                    SdkTool.Execute(
                        "MakeSignedBinary.exe",
                        "sign",
                        "-o", signedSystemPartition.FullName,
                        "-i", systemPartition.FullName,
                        "-k", keyFile.FullName,
                        "--disable-encryption"
                        );
                }
            }
        }

        private void MakeSystemPartition(FileInfo outputPath, List<string> installList, FirmwareArchiveDirectory archiveDirectory, FileInfo fatKey = null)
        {
            MakeInstalledFatImage(outputPath, fatKey, installList);
        }

        private void MakeInstalledFatImage(FileInfo outputPath, FileInfo keyFile, List<string> installedPrograms)
        {
            var arguments = new List<string>
            {
                "-o", outputPath.FullName,
                "--size", PartitionSize,
                "--delete", "contents/meta",
                "--delete", "save",
                "--copy-file", "save/8000000000000120/0/meta/imkvdb.arc,cnmtdb.arc",
                "--ensure-free-space", EnsuredFreeSpace
            };

            if (keyFile != null)
            {
                var keyarg = new List<string>
                {
                    "--key-file",
                    keyFile.FullName
                };

                arguments = arguments.Concat(keyarg).ToList();
            }

            arguments = arguments.Concat(installedPrograms.ToArray()).ToList();

            SdkTool.Execute(
                "MakeInstalledFatImage.exe",
                arguments.ToArray()
            );
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            try
            {
                SingleCommandRunner<MakeSignedSystemPartition>.Run(args);
            }
            catch
            {
                Environment.Exit(1);
            }
        }
    }
}
