﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using CommandUtility;
using System.IO;
using System.Security.Cryptography;
using Nintendo.Authoring.CryptoLibrary;

namespace MakeSignedBinary
{
    public class Utility
    {
        const int MaxDataSize = 32 * 1024 * 1024;

        public static void EncryptAesCtr128(Stream output, Stream input, byte[] key, byte[] iv)
        {
            var inputBuffer = new byte[MaxDataSize + 1];
            var outputBuffer = new byte[MaxDataSize + 1];

            var readSize = input.Read(inputBuffer, 0, inputBuffer.Length);
            if(readSize == inputBuffer.Length)
            {
                throw new Exception($"Too large data(max:{MaxDataSize}).");
            }

            var aesCtr = new Aes128CtrCryptoDriver(key);
            aesCtr.EncryptBlock(iv, inputBuffer, 0, readSize, outputBuffer, 0);

            output.Write(outputBuffer, 0, readSize);
        }

        public static void DecryptAesCtr128(Stream output, Stream input, byte[] key, byte[] iv)
        {
            var inputBuffer = new byte[MaxDataSize + 1];
            var outputBuffer = new byte[MaxDataSize + 1];

            var readSize = input.Read(inputBuffer, 0, inputBuffer.Length);
            if (readSize == inputBuffer.Length)
            {
                throw new Exception($"Too large data(max:{MaxDataSize}).");
            }

            var aesCtr = new Aes128CtrCryptoDriver(key);
            aesCtr.DecryptBlock(iv, inputBuffer, 0, readSize, outputBuffer, 0);

            output.Write(outputBuffer, 0, readSize);
        }

        public static void EncryptContent(FileInfo outputFile, string inputFile, byte[] encryptionKey, byte[] iv)
        {
            using (var writer = outputFile.OpenWrite())
            using (var reader = File.OpenRead(inputFile))
            {
                EncryptAesCtr128(writer, reader, encryptionKey, iv);
            }
        }

        public static byte[] SignBlock(byte[] data, KeyConfig keyConfig)
        {
            var rsa = new Rsa2048PssSha256SignCryptoDriver(keyConfig.ModulusBytes, keyConfig.PublicExponentBytes, keyConfig.PrivateExponentBytes);

            return rsa.SignBlock(data, 0, data.Length);
        }

        public static bool VerifySignBinary(KeyConfig keyConfig, string inputFile, string[] appendFiles)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeSignedBinary"))
            using (var reader = File.OpenRead(inputFile))
            {
                var content = tempHolder.CreateTemporaryFilePath("encryptedContent");

                var expectedSignature = BinaryUtility.ReadBytes(reader, 256);

                var header = SignedBinaryHeader.Load(reader);

                using (var writer = content.OpenWrite())
                {
                    reader.CopyTo(writer);
                }

                var headerBytes = header.MakeBinary();
                var headerHash = Utility.CalculateHash(header.MakeBinary());
                var actualSignature = Utility.SignBlock(headerBytes, keyConfig);

                if (!expectedSignature.SequenceEqual(actualSignature))
                {
                    Console.Error.WriteLine("Signature is mismatched");
                    return false;
                }

                var rsa = new Rsa2048PssSha256SignCryptoDriver(keyConfig.ModulusBytes, keyConfig.PublicExponentBytes, keyConfig.PrivateExponentBytes);
                if (!rsa.VerifyBlock(headerBytes, 0, headerBytes.Length, actualSignature))
                {
                    Console.Error.WriteLine("Signature is mismatched");
                    return false;
                }

                var contentSize = new FileInfo(content.FullName).Length;
                if (header.Size != contentSize)
                {
                    Console.Error.WriteLine($"Content size is mismatched. expected:{header.Size} != actual:{contentSize}");
                    return false;
                }

                var sha256hash = Utility.CalculateHash(content.FullName);
                if (!sha256hash.SequenceEqual(header.ContentHash))
                {
                    Console.Error.WriteLine($"Content hash is mismatched.");
                    return false;
                }

                if (appendFiles.Length != header.AppendHashCount)
                {
                    Console.Error.WriteLine($"Append Content Count is mismatched. header expected: {header.AppendHashCount}, input files: {appendFiles.Length}");
                    return false;
                }

                foreach (var target in header.AppendHashInfos.Zip(appendFiles, (info, file) => new { info = info, file = new FileInfo(file) }))
                {
                    if(target.file.Length != target.info.Size)
                    {
                        Console.Error.WriteLine($"Append Content Size is mismatched.");
                        Console.Error.WriteLine($"header expected: {target.info.Size}, input file size: {target.file.Length}");
                        return false;
                    }

                    var fileHash = Utility.CalculateHash(target.file.FullName);
                    if(!fileHash.SequenceEqual(target.info.Hash))
                    {
                        Console.Error.WriteLine($"Append Content Hash is mismatched.");
                        Console.Error.WriteLine($"header expected: {BinaryUtility.ToPrettyHexString(target.info.Hash)}");
                        Console.Error.WriteLine($"input file hash: {BinaryUtility.ToPrettyHexString(fileHash)}");
                        return false;
                    }
                }
            }

            return true;
        }

        public static byte[] CalculateHash(IEnumerable<string> filenames)
        {
            var sha256 = SHA256.Create();

            foreach (var filename in filenames)
            {
                using (var stream = File.OpenRead(filename))
                {
                    sha256.ComputeHash(stream);
                }
            }

            return sha256.Hash;
        }

        public static byte[] CalculateHash(string filename)
        {
            var sha256 = SHA256.Create();

            using (var stream = File.OpenRead(filename))
            {
                sha256.ComputeHash(stream);
            }

            return sha256.Hash;
        }

        public static byte[] CalculateHash(byte[] data)
        {
            var sha256 = SHA256.Create();

            sha256.ComputeHash(data);

            return sha256.Hash;
        }

        public static void SignBinary(string outputFile, string inputFile, string[] appendFiles, KeyConfig keyConfig, TemporaryFileHolder tempHolder, bool disableEncryption = false)
        {
            var iv = CryptUtility.GenerateIV();

            var encryptedFile = tempHolder.CreateTemporaryFilePath("encryptedFile");

            if (disableEncryption)
            {
                File.Copy(inputFile, encryptedFile.FullName);
            }
            else
            {
                Utility.EncryptContent(encryptedFile, inputFile, keyConfig.EncryptionKeyBytes, iv);
            }

            var sha256hash = Utility.CalculateHash(encryptedFile.FullName);

            var appendHashInfos = appendFiles.Select(filename => {
                var appendHash = Utility.CalculateHash(filename);
                var fileInfo = new FileInfo(filename);
                return new AppendHashInfo()
                {
                    Size = fileInfo.Length,
                    Hash = appendHash
                };
            }).ToArray();

            var header = new SignedBinaryHeader(
                keyConfig.Version,
                new FileInfo(inputFile).Length,
                iv,
                sha256hash,
                keyConfig.KeyGeneration,
                appendHashInfos);

            var headerBytes = header.MakeBinary();

            var signature = Utility.SignBlock(headerBytes, keyConfig);

            using (var output = File.OpenWrite(outputFile))
            {
                output.Write(signature, 0, signature.Count());

                output.Write(headerBytes, 0, headerBytes.Length);

                using (var content = encryptedFile.OpenRead())
                {
                    content.CopyTo(output);
                }
            }
        }

        public static void RemoveSignBinary(KeyConfig keyConfig, string outputFile, string inputFile)
        {
            using (var reader = File.OpenRead(inputFile))
            {
                var signature = BinaryUtility.ReadBytes(reader, 256);

                var header = SignedBinaryHeader.Load(reader);

                using (var writer = File.OpenWrite(outputFile))
                {
                    DecryptAesCtr128(writer, reader, keyConfig.EncryptionKeyBytes, header.IV);
                }
            }
        }
    }
}
