﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using CommandUtility;

namespace MakeSignedBinary
{
    public class AppendHashInfo
    {
        public Int64 Size { get; set; }
        public byte[] Hash { get; set; }
    }

    public class SignedBinaryHeader
    {
        public Int64 Version { get; private set; }
        public Int64 Size { get; private set; }
        public byte[] IV { get; private set; }
        public byte[] ContentHash { get; private set; }
        public Int64 KeyGeneration { get; private set; }
        public Int64 AppendHashCount { get; private set; }
        public AppendHashInfo[] AppendHashInfos { get; private set; }

        public const int HeaderSize = 256;
        public const int MaxAppendHashCount = 4;

        public SignedBinaryHeader(Int64 version, Int64 size, byte[] iv, byte[] contentHash, Int64 keyGeneration, AppendHashInfo[] appendHashInfos)
        {
            this.Version = version;
            this.Size = size;
            this.IV = iv;
            this.ContentHash = contentHash;
            this.KeyGeneration = keyGeneration;
            this.AppendHashCount = appendHashInfos.Length;
            this.AppendHashInfos = appendHashInfos;

            if(MaxAppendHashCount < appendHashInfos.Length)
            {
                throw new Exception($"Too many append hash. count = {appendHashInfos.Length}, max = {MaxAppendHashCount}");
            }

            if (IV.Length != 16)
            {
                throw new Exception($"Invalid IV length: expected=16 != actual={iv.Length}");
            }

            if (ContentHash.Length != 32)
            {
                throw new Exception($"Invalid hash length: expected=32 != actual={ContentHash.Length}");
            }
        }

        public static SignedBinaryHeader Load(FileStream reader)
        {
            var headerBytes = new byte[HeaderSize];
            var readSize = reader.Read(headerBytes, 0, HeaderSize);

            if(readSize != HeaderSize)
            {
                throw new Exception($"Too small header. readSize={readSize}");
            }

            using (var headerReader = new MemoryStream(headerBytes))
            {
                var version = BinaryUtility.ReadBinary<Int64>(headerReader);
                var size = BinaryUtility.ReadBinary<Int64>(headerReader);
                var iv = BinaryUtility.ReadBytes(headerReader, 16);
                var hash = BinaryUtility.ReadBytes(headerReader, 32);
                var keyGeneration = BinaryUtility.ReadBinary<Int64>(headerReader);
                var appendHashCount = BinaryUtility.ReadBinary<Int64>(headerReader);

                if(MaxAppendHashCount < appendHashCount)
                {
                    throw new Exception($"Too many append hash. count = {appendHashCount}, max = {MaxAppendHashCount}");
                }

                var appendHashInfos = Enumerable.Range(0, (int)appendHashCount).Select(i =>
                {
                    var appendContentSize = BinaryUtility.ReadBinary<Int64>(headerReader);
                    var appendContentHash = BinaryUtility.ReadBytes(headerReader, 32);
                    return new AppendHashInfo()
                    {
                        Size = appendContentSize,
                        Hash = appendContentHash
                    };
                }).ToArray();

                return new SignedBinaryHeader(
                    version,
                    size,
                    iv,
                    hash,
                    keyGeneration,
                    appendHashInfos);
            }
        }

        public byte[] MakeBinary()
        {
            using (var stream = new MemoryStream())
            {
                BinaryUtility.WriteBinary<Int64>(stream, Version);
                BinaryUtility.WriteBinary<Int64>(stream, Size);
                stream.Write(IV, 0, IV.Count());
                stream.Write(ContentHash, 0, ContentHash.Count());
                BinaryUtility.WriteBinary<Int64>(stream, KeyGeneration);

                BinaryUtility.WriteBinary<Int64>(stream, AppendHashCount);
                foreach (var appendHashInfo in this.AppendHashInfos)
                {
                    BinaryUtility.WriteBinary<Int64>(stream, appendHashInfo.Size);
                    stream.Write(appendHashInfo.Hash, 0, appendHashInfo.Hash.Count());
                }

                var restSize = HeaderSize - (int)stream.Position;
                if(restSize < 0)
                {
                    throw new Exception("Too large header");
                }

                var padding = new byte[restSize];
                stream.Write(padding, 0, restSize);

                return stream.ToArray();
            }
        }
    }
}
