﻿using System;
using CommandUtility;
using Nintendo.Foundation.IO;
using System.Text.RegularExpressions;
using System.IO;

namespace MakeRecoveryWriter.Commands
{
    public class MakeRecoveryWriterCommand : SubCommandCommon
    {
        [CommandLineOption("program", Description = "input recovery writer program image", IsRequired = true)]
        public string InputProgram { get; set; }

        [CommandLineOption("gpt", Description = "input gpt image", IsRequired = true)]
        public string InputGpt { get; set; }

        [CommandLineOption("bct", Description = "input bct image", IsRequired = true)]
        public string InputBct { get; set; }

        [CommandLineOption("bootloader", Description = "input boot loader image", IsRequired = true)]
        public string InputBootLoader { get; set; }

        [CommandLineOption("package2", Description = "input package2", IsRequired = true)]
        public string InputPackage2 { get; set; }

        [CommandLineOption("system-partition", Description = "input system partition", IsRequired = true)]
        public string InputSystemPartition { get; set; }

        [CommandLineOption("system-partition-alignment", Description = "input system parition image alignment", IsRequired = false, DefaultValue = "0B")]
        public string SystemPartitionAlignmentString { get; set; }

        [CommandLineOption("min", Description = "min size of recovery writer image. example: 16777216, 16777216B, 16384KB, 16MB", IsRequired = true)]
        public string MinSizeString { get; set; }

        [CommandLineOption("max", Description = "max size of recovery writer image. example: 16777216, 16777216B, 16384KB, 16MB", IsRequired = true)]
        public string MaxSizeString { get; set; }

        [CommandLineOption('o', "output", Description = "output recovery writer image file", IsRequired = true)]
        public string OutputFile { get; set; }

        public long MaxSize
        {
            get
            {
                return BinaryUtility.ConvertFromByteUnitExpression(MaxSizeString);
            }
        }

        public long MinSize
        {
            get
            {
                return BinaryUtility.ConvertFromByteUnitExpression(MinSizeString);
            }
        }

        public long SystemPartitionAlignment
        {
            get
            {
                return BinaryUtility.ConvertFromByteUnitExpression(SystemPartitionAlignmentString);
            }
        }

        public override int Run()
        {
            var outputFileInfo = new FileInfo(OutputFile);
            using (var writer = outputFileInfo.OpenWrite())
            {
                var programInfo = new FileInfo(this.InputProgram);
                var gptInfo = new FileInfo(InputGpt);
                var bctInfo = new FileInfo(InputBct);
                var bootLoaderInfo = new FileInfo(InputBootLoader);
                var package2Info = new FileInfo(InputPackage2);
                var systemPartitionInfo = new FileInfo(InputSystemPartition);

                RecoveryWriterImage.MakeImage(writer, programInfo, gptInfo, bctInfo, bootLoaderInfo, package2Info, systemPartitionInfo, SystemPartitionAlignment, MinSize, MaxSize);
            }

            return 0;
        }
    }
}
