﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;

namespace MakeInitialImage
{
    public class InitialImageConstructor
    {
        public InitialImageConstructor(Stream output, InitialImageConfig config, DirectoryInfo workingDirectory, FileInfo makeFatImagePath, FileInfo installNspPath)
        {
            this.outputStream = output;
            this.config = config;
            imageWriter = new InitialImageWriter(outputStream);
            gptWriter = new GuidPartitionTableWriter();
            this.makeFatImagePath = makeFatImagePath;
            this.installNspPath = installNspPath;
            WorkingDirectory = workingDirectory;
        }

        public void ConstructImage()
        {
            using (TemporaryFileHolder = new TemporaryFileHolder("MakeInitialImage", WorkingDirectory.FullName))
            {
                ValidateConfig();
                ConstructCommands();
                ConstructPartitionAddresses();
                ConstructGpt();
                ConstructSpecialPartitions();
                ConstructPartitions();
                ConstructPreprocessCommands();
                ConstructPostprocessCommands();
                ConstructAlternativeCommands();
                imageWriter.WritePartitions();
            }
        }

        private void ConstructAlternativeCommands()
        {
            if (config.StorageDefinitions[0].Commands != null)
            {
                alternativeCommands = Parse(config, imageWriter, gptWriter);
            }
        }

        private void ConstructPostprocessCommands()
        {
            commands.AddRange(postProcessCommands);
        }

        private void ConstructPreprocessCommands()
        {
            commands.Insert(0, InitialImageCommand.MakeEnsureProtectProductionInfoCommand(
                config.InitialImageDefinition.AllowOverwritingProductionInfo,
                imageWriter.GetPartitionIndex(GptHeaderString),
                imageWriter.GetPartitionIndex(GptEntriesString),
                imageWriter.GetPartitionIndex(PartitionAddressString)));
        }

        private void ValidateConfig()
        {
            if (config.StorageDefinitions.Count != 1)
            {
                throw new InvalidDataException(string.Format("Unsupported StorageDefinitions: count = {0}", config.StorageDefinitions.Count));
            }
        }

        private void ConstructPartitionAddresses()
        {
            var partitionAddressList = PartitionAddress.MakePartitionAddress(config.StorageDefinitions[0]);
            imageWriter.AddPartition(PartitionAddressString, (writer) =>
            {
                Utility.WriteBinary<int>(writer, partitionAddressList.Count);
                foreach (var partitionAddress in partitionAddressList)
                {
                    Utility.WriteBinary<int>(writer, (int)partitionAddress.Type);
                    switch (partitionAddress.Type)
                    {
                        case PartitionAddressType.Absolute:
                            Utility.WriteBinary<int>(writer, 0);
                            Utility.WriteBinary<long>(writer, partitionAddress.AbsoluteAddress.Address);
                            break;
                        case PartitionAddressType.Relative:
                            Utility.WriteBinary<int>(writer, partitionAddress.RelativeAddress.BasePartitionIndex);
                            Utility.WriteBinary<long>(writer, partitionAddress.RelativeAddress.Address);
                            break;
                        case PartitionAddressType.Keep:
                            Utility.WriteBinary<int>(writer, partitionAddress.RelativeAddress.BasePartitionIndex);
                            Utility.WriteBinary<long>(writer, partitionAddress.RelativeAddress.Address);
                            break;
                        default:
                            throw new InvalidDataException("Unsupported type");
                    }
                    Utility.WriteBinary<long>(writer, partitionAddress.Size);
                }
            });
        }

        private void ConstructCommands()
        {
            imageWriter.AddPartition(CommandPartitionString, WriteCommands);
        }

        private void ConstructGpt()
        {
            imageWriter.AddPartition(ProtectiveHeaderString, gptWriter.WriteMbr);

            imageWriter.AddPartition(GptHeaderString, gptWriter.WriteHeader);

            imageWriter.AddPartition(GptEntriesString, gptWriter.WritePartitionEntries);

            commands.Add(InitialImageCommand.MakeWriteGpt(
                imageWriter.GetPartitionIndex(ProtectiveHeaderString),
                imageWriter.GetPartitionIndex(GptHeaderString),
                imageWriter.GetPartitionIndex(GptEntriesString),
                imageWriter.GetPartitionIndex(PartitionAddressString)));
        }

        private void ConstructSpecialPartitions()
        {
            if (config.StorageDefinitions[0].SpecialPartitions == null)
            {
                return;
            }

            foreach (var partition in config.StorageDefinitions[0].SpecialPartitions)
            {
                if (ProcessedPartition.Contains(partition.Name))
                {
                    return;
                }

                if (partition.UpdateSecureInfo)
                {
                    postProcessCommands.Add(MakeUpdateSecureInfoCommand(partition));
                }

                switch (partition.GeneratingType)
                {
                    case PartitionGeneratingType.RawImage:
                        ConstructRawImagePartition(partition, false);
                        break;
                    case PartitionGeneratingType.Installed:
                        ConstructPartitionFromContents(partition, false);
                        break;
                    case PartitionGeneratingType.PairedInstalled:
                        ConstructPartitionFromContentsWithPair(partition, false);
                        break;
                    case PartitionGeneratingType.Clean:
                        ConstructCleanPartition(partition, false);
                        break;
                    case PartitionGeneratingType.Keep:
                        ConstructKeepPartition(partition, false);
                        break;
                    default:
                        throw new Exception(
                            string.Format("Not supported GenerationType: name={0}, type={1}",
                                partition.Name,
                                partition.GeneratingType.ToString()));
                }
            }
        }

        private void ConstructPartitionFromContentsWithPair(Partition primaryPartition, bool asGptPartition)
        {
            var pairedPartitionName = primaryPartition.PairedPartition;

            ProcessedPartition.Add(pairedPartitionName);

            if(pairedPartitionName == null
                || !HasPartition(pairedPartitionName))
            {
                throw new Exception($"Invalid paired partition: partitionName={primaryPartition.Name}, pairedPartition={pairedPartitionName}");
            }

            var pairedPartition = GetPartition(pairedPartitionName);

            if(!(primaryPartition.GeneratingType == PartitionGeneratingType.PairedInstalled &&
                 pairedPartition.GeneratingType == PartitionGeneratingType.Installed))
            {
                throw new Exception($"Invalid generating type: partitionName={primaryPartition.Name}, pairedPartition={pairedPartitionName}");
            }


            // ユーザパーティション用のディレクトリの作成
            var pairedPartitionDirectory = TemporaryFileHolder.CreateTemporaryDirectory("PairedPartition");

            // ユーザーのパーティションの baseDirectory のコピー
            if(pairedPartition.BaseDirectory != null)
            {
                Utility.CopyDirectory(new DirectoryInfo(pairedPartition.BaseDirectory), pairedPartitionDirectory);
            }

            // システムパーティションのアプリインストール処理
            var preprocessors = new List<IFatImagePreprocessor>() {
                new ApplicationInstallProcessor(
                    pairedPartitionDirectory,
                    pairedPartition.InstallContents.Select(filePath => new FileInfo(filePath)).ToList(),
                    TemporaryFileHolder)
            };

            ConstructPartitionFromContents(primaryPartition, asGptPartition, preprocessors);

            // 後ろのパーティションの FatImage の作成
            FileInfo fatImageFile = TemporaryFileHolder.CreateTemporaryFilePath("FatImage");
            var imageCreateor = new InstalledFatImage(makeFatImagePath, installNspPath, pairedPartition.DeleteFiles);

            imageCreateor.Create(
                    fatImageFile,
                    pairedPartitionDirectory,
                    null,
                    MakeInstallStorageName(pairedPartition.InstallStorageType),
                    pairedPartition.Size.MebiBytes,
                    true,
                    null,
                    0);

            ConstructSparseImagePartition(pairedPartition, fatImageFile, asGptPartition);
        }

        private Partition GetPartition(string partitionName)
        {
            return config.StorageDefinitions[0].Partitions.First(partition => partition.Name == partitionName);
        }

        private bool HasPartition(string pairedPartition)
        {
            return this.config.StorageDefinitions[0].Partitions.Any(partition => partition.Name == pairedPartition);
        }

        public static InitialImageCommand MakeUpdateSecureInfoCommand(Partition partition)
        {
            long secureInfoIndex = CalculateSecureInfoIndex(partition);

            return InitialImageCommand.MakeUpdateSecureInfoCommand(secureInfoIndex);
        }

        private static long CalculateSecureInfoIndex(Partition partition)
        {
            if (partition.FsPartitionId != "BootPartition1Root")
            {
                throw new Exception("Unsupported partition(FsPartitionId != BootPartition1Root): " + partition.FsPartitionId);
            }

            if (!(0x0 <= partition.OffsetAsNumber && partition.OffsetAsNumber <= 0x10000))
            {
                throw new Exception("Invalid offset range(0-0x10000): " + partition.Offset);
            }

            if (partition.OffsetAsNumber % 0x4000 != 0)
            {
                throw new Exception("Invalid offset alignment(0x4000): " + partition.Offset);
            }

            long SecureInfoIndex = partition.OffsetAsNumber / 0x4000;
            return SecureInfoIndex;
        }

        private void ConstructPartitions()
        {
            if (config.StorageDefinitions[0].Partitions == null)
            {
                return;
            }

            foreach (var partition in config.StorageDefinitions[0].Partitions)
            {
                ConstructPartition(partition);
            }
        }

        private void ConstructPartition(Partition partition)
        {
            if (ProcessedPartition.Contains(partition.Name))
            {
                return;
            }

            switch (partition.GeneratingType)
            {
                case PartitionGeneratingType.RawImage:
                    ConstructRawImagePartition(partition, true);
                    break;
                case PartitionGeneratingType.Installed:
                    ConstructPartitionFromContents(partition, true);
                    break;
                case PartitionGeneratingType.PairedInstalled:
                    ConstructPartitionFromContentsWithPair(partition, true);
                    break;
                case PartitionGeneratingType.Clean:
                    ConstructCleanPartition(partition, true);
                    break;
                case PartitionGeneratingType.Keep:
                    ConstructKeepPartition(partition, true);
                    break;
                default:
                    throw new Exception("Not supported GenerationType: " + partition.GeneratingType.ToString());
            }
        }

        private void ConstructCleanPartition(Partition partition, bool asGptPartition)
        {
            DirectoryInfo temporaryDirectory = TemporaryFileHolder.CreateTemporaryDirectory("FatImageRoot");

            FileInfo fatImageFile = TemporaryFileHolder.CreateTemporaryFilePath("FatImage");

            new InstalledFatImage(makeFatImagePath, installNspPath, new List<string>()).CreateFatImage(fatImageFile, temporaryDirectory, partition.Size.MebiBytes, true);

            ConstructSparseImagePartition(partition, fatImageFile, asGptPartition);
        }

        private void ConstructKeepPartition(Partition partition, bool asGptPartition)
        {
            if (asGptPartition)
            {
                gptWriter.AddPartition(
                    partition.Name,
                    0,
                    Guid.Parse(partition.TypeGuid),
                    Guid.Parse(partition.Guid),
                    true);
            }
        }

        private void ConstructPartitionFromContents(Partition partition, bool asGptPartition, List<IFatImagePreprocessor> preprocessors = null)
        {
            FileInfo fatImageFile = TemporaryFileHolder.CreateTemporaryFilePath("FatImage");
            var baseDirectory = partition.BaseDirectory == null ? null : new DirectoryInfo(partition.BaseDirectory);
            var installContents = partition.InstallContents == null ? null : (from content in partition.InstallContents select new FileInfo(content)).ToList();

            var extractingSaveInfos = MakeExtractedSaveInfos(partition);
            var saveDataExtractor = new SaveDataExtractor(partition, TemporaryFileHolder);

            var fatimageMaker = new InstalledFatImage(makeFatImagePath, installNspPath, partition.DeleteFiles);

            if(preprocessors != null)
            {
                foreach (var processor in preprocessors)
                {
                    fatimageMaker.AddPreprocessor(processor);
                }
            }

            fatimageMaker.Create(
                    fatImageFile,
                    baseDirectory,
                    installContents,
                    MakeInstallStorageName(partition.InstallStorageType),
                    partition.Size.MebiBytes,
                    true,
                    null,
                    0,
                    extractingSaveInfos,
                    saveDataExtractor
                    );

            ConstructSparseImagePartition(partition, fatImageFile, asGptPartition);

            if(partition.NeedsExtractingSaveData)
            {
                ConstructDatabasePartition(partition, extractingSaveInfos, saveDataExtractor);
            }
        }

        private void ConstructDatabasePartition(Partition partition, List<Tuple<FileInfo, string>> extractingSaveInfos, SaveDataExtractor saveDataExtractor)
        {

            imageWriter.AddPartition($"{partition.Name}-CNTMDB", (writer) =>
            {
                saveDataExtractor.Export(writer);
            });
        }

        private List<Tuple<FileInfo, string>> MakeExtractedSaveInfos(Partition partition)
        {
            if (partition.ExtractedSaves == null)
            {
                return new List<Tuple<FileInfo, string>>();
            }
            else
            {
                return (from savepath in partition.ExtractedSaves
                        where true
                        select new Tuple<FileInfo, string>(
                            TemporaryFileHolder.CreateTemporaryFilePath("ExtractedSave"),
                            savepath)).ToList();
            }
        }

        private string MakeInstallStorageName(InstallStorageType installStorageType)
        {
            if (installStorageType == InstallStorageType.System)
            {
                return "system";
            }
            else if (installStorageType == InstallStorageType.User)
            {
                return "user";
            }

            throw new Exception(string.Format("Unknown install storage type: {0}", installStorageType));
        }

        private void ConstructSparseImagePartition(Partition partition, FileInfo fatImageFile, bool asGptPartition)
        {
            imageWriter.AddPartition(partition.Name, (outStr) =>
            {
                using (var inputStream = new FileStream(fatImageFile.FullName, FileMode.Open, FileAccess.Read))
                {
                    inputStream.CopyTo(outStr);
                }
            });

            if (asGptPartition)
            {
                gptWriter.AddPartition(
                    partition.Name,
                    partition.Size.Bytes,
                    Guid.Parse(partition.TypeGuid),
                    Guid.Parse(partition.Guid),
                    true);
            }

            if (partition.FsPartitionId == null)
            {
                commands.Add(
                    InitialImageCommand.MakeWriteSparsePartitionImageCommand(
                        gptWriter.GetPartitionIndex(partition.Name),
                        imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
            }
            else
            {
                commands.Add(
                    InitialImageCommand.MakeWriteFsSparsePartitionImageCommand(
                        partition.FsPartitionId,
                        imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
            }
        }

        private void ConstructRawImagePartition(Partition partition, bool asGptPartition)
        {
            imageWriter.AddPartition(partition.Name, (outStr) =>
            {
                using (var inputStream = new FileStream(partition.RawImageFile, FileMode.Open, FileAccess.Read))
                {
                    inputStream.CopyTo(outStr);
                }
            });

            if (asGptPartition)
            {
                gptWriter.AddPartition(
                    partition.Name,
                    partition.Size.Bytes,
                    Guid.Parse(partition.TypeGuid),
                    Guid.Parse(partition.Guid),
                    true);
            }

            if (partition.FsPartitionId == null)
            {
                commands.Add(
                    InitialImageCommand.MakeWritePartitionImageCommand(
                        gptWriter.GetPartitionIndex(partition.Name),
                        imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
            }
            else
            {
                commands.Add(
                    InitialImageCommand.MakeWriteFsPartitionImageCommand(
                        partition.FsPartitionId,
                        imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
            }
        }

        private void WriteCommands(Stream writer)
        {
            if (alternativeCommands == null)
            {
                WriteCommands(writer, commands);
            }
            else
            {
                WriteCommands(writer, alternativeCommands);
            }
        }

        private void WriteCommands(Stream writer, List<InitialImageCommand> commands)
        {
            Utility.WriteBinary<int>(writer, commands.Count);
            foreach (var command in commands)
            {
                Utility.WriteBinary<InitialImageCommand>(writer, command);
            }
        }

        private Guid ConvertParitionTypeGuid(PartitionType partitionType)
        {
            switch (partitionType)
            {
                case PartitionType.Required:
                    return RequiredTypeGuid;
                case PartitionType.Optional:
                    return OptionalTypeGuid;
                default:
                    throw new NotSupportedException("Not supported parititoin type: " + partitionType.ToString());
            }
        }

        public InitialImageCommand.CommandType ParseCommandType(string commandName)
        {
            return (InitialImageCommand.CommandType)Enum.Parse(typeof(InitialImageCommand.CommandType), commandName);
        }

        public List<InitialImageCommand> Parse(InitialImageConfig config, InitialImageWriter imageWriter, GuidPartitionTableWriter gptWriter)
        {
            var ret = new List<InitialImageCommand>();

            foreach (var rawCommand in config.StorageDefinitions[0].Commands)
            {
                if (!rawCommand.ContainsKey("Command"))
                {
                    throw new Exception(string.Format("Command is not found."));
                }

                switch (ParseCommandType(rawCommand["Command"]))
                {
                    case InitialImageCommand.CommandType.WriteGpt:
                        {
                            ret.Add(InitialImageCommand.MakeWriteGpt(
                                imageWriter.GetPartitionIndex(InitialImageConstructor.ProtectiveHeaderString),
                                imageWriter.GetPartitionIndex(InitialImageConstructor.GptHeaderString),
                                imageWriter.GetPartitionIndex(InitialImageConstructor.GptEntriesString),
                                imageWriter.GetPartitionIndex(InitialImageConstructor.PartitionAddressString)));
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteRawImage:
                        throw new NotImplementedException();
                    case InitialImageCommand.CommandType.UpdateGptHeader:
                        throw new NotImplementedException();
                    case InitialImageCommand.CommandType.WritePartitionImage:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeWritePartitionImageCommand(
                                    gptWriter.GetPartitionIndex(partition.Name),
                                    imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteSparsePartitionImage:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeWriteSparsePartitionImageCommand(
                                    gptWriter.GetPartitionIndex(partition.Name),
                                    imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteFsPartitionImage:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeWriteFsPartitionImageCommand(
                                    partition.FsPartitionId,
                                    imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteFsSparsePartitionImage:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeWriteFsSparsePartitionImageCommand(
                                    partition.FsPartitionId,
                                    imageWriter.GetPartitionIndex(partition.Name), partition.OffsetAsNumber));
                        }
                        break;
                    case InitialImageCommand.CommandType.EnsureProtectProductionInfo:
                        {
                            ret.Add(
                                InitialImageCommand.MakeEnsureProtectProductionInfoCommand(
                                    config.InitialImageDefinition.AllowOverwritingProductionInfo,
                                    imageWriter.GetPartitionIndex(InitialImageConstructor.GptHeaderString),
                                    imageWriter.GetPartitionIndex(InitialImageConstructor.GptEntriesString),
                                    imageWriter.GetPartitionIndex(InitialImageConstructor.PartitionAddressString)));
                        }
                        break;
                    case InitialImageCommand.CommandType.UpdateSecureInfo:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageConstructor.MakeUpdateSecureInfoCommand(partition));
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteBatchedBootPartitions:
                        {
                            var bctPartition = GetTargetPartition(config, "BctPartitionName", rawCommand);
                            var bootLoaderPartition = GetTargetPartition(config, "BootLoaderPartitionName", rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeWriteBatchedBootPartitionsCommand(
                                    bctPartition.FsPartitionId,
                                    imageWriter.GetPartitionIndex(bctPartition.Name), bctPartition.OffsetAsNumber,
                                    bootLoaderPartition.FsPartitionId,
                                    imageWriter.GetPartitionIndex(bootLoaderPartition.Name), bootLoaderPartition.OffsetAsNumber,
                                    CalculateSecureInfoIndex(bctPartition)));
                        }
                        break;
                    case InitialImageCommand.CommandType.EraseEmmcWithEscapingData:
                        {
                            var targetPartition1 = gptWriter.GetPartitionIndex(GetTargetPartition(config, "TargetPartition1" , rawCommand).Name);
                            var targetPartition2 = gptWriter.GetPartitionIndex(GetTargetPartition(config, "TargetPartition2", rawCommand).Name);
                            ret.Add(
                                InitialImageCommand.MakeEraseEmmcWithEscapingDataCommand(
                                    targetPartition1,
                                    targetPartition2));
                        }
                        break;
                    case InitialImageCommand.CommandType.InitializeZero:
                        {
                            var partition = GetTargetPartition(config, rawCommand);
                            ret.Add(
                                InitialImageCommand.MakeInitializeZeroCmmand(
                                    partition.FsPartitionId));
                        }
                        break;
                    case InitialImageCommand.CommandType.EnableUpdatingFromBootImagePackage:
                        {
                            var target = rawCommand["Target"];
                            if (target == "Normal")
                            {
                                ret.Add(InitialImageCommand.MakeEnableUpdatingFromBootImagePackage(0));
                            }
                            else if(target == "Safe")
                            {
                                ret.Add(InitialImageCommand.MakeEnableUpdatingFromBootImagePackage(1));
                            }
                            else
                            {
                                throw new Exception(string.Format("Unknown Target: {0}", target));
                            }
                        }
                        break;
                    case InitialImageCommand.CommandType.WriteDatabase:
                        {
                            ret.Add(InitialImageCommand.MakeWriteDatabaseCommand(
                                GetParameter(rawCommand, "FsPartitionId"),
                                imageWriter.GetPartitionIndex(GetParameter(rawCommand, "PartitionName"))));
                        }
                        break;
                    default:
                        throw new InvalidDataException("Unknown command name: " + rawCommand["Command"]);
                }
            }

            return ret;
        }

        private static string GetParameter(Dictionary<string, string> rawCommand, string parameterName)
        {
            if(!rawCommand.ContainsKey(parameterName))
            {
                throw new Exception($"Parameter={parameterName} is not found. command={rawCommand["Command"]}");
            }

            return rawCommand[parameterName];
        }

        private static Partition GetTargetPartition(InitialImageConfig config, Dictionary<string, string> rawCommand)
        {
            return GetTargetPartition(config, "PartitionName", rawCommand);
        }

        private static Partition GetTargetPartition(InitialImageConfig config, string key, Dictionary<string, string> rawCommand)
        {
            if (!rawCommand.ContainsKey(key))
            {
                throw new Exception(string.Format("Parameter={0} is not found. command={1}", key, rawCommand["Command"]));
            }
            var partitionName = rawCommand[key];
            var partition = config.StorageDefinitions[0].FindPartition(partitionName);
            if (partition == null)
            {
                throw new Exception(string.Format("Found no partitions. name = {0}", partitionName));
            }
            return partition;
        }

        public const string PartitionAddressString = "PartitionAddress";
        public const string CommandPartitionString = "Command";
        public const string ProtectiveHeaderString = "ProtectiveMbr";
        public const string GptHeaderString = "GptHeader";
        public const string GptEntriesString = "GptEntries";

        private List<InitialImageCommand> commands = new List<InitialImageCommand>();
        private List<InitialImageCommand> postProcessCommands = new List<InitialImageCommand>();
        private List<InitialImageCommand> alternativeCommands = null;
        private Stream outputStream;
        private InitialImageConfig config;
        public InitialImageWriter imageWriter;
        private TemporaryFileHolder TemporaryFileHolder = null;
        private DirectoryInfo WorkingDirectory;
        public GuidPartitionTableWriter gptWriter;
        private readonly Guid RequiredTypeGuid = new Guid("a102efcc-4fd0-4fce-88c1-4a2085006c3d");
        private readonly Guid OptionalTypeGuid = new Guid("f5993f83-174a-4546-ad1c-9c7f3ac071f5");
        private FileInfo makeFatImagePath;
        private FileInfo installNspPath;
        private HashSet<string> ProcessedPartition = new HashSet<string>();
    }
}
