﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using CommandUtility;
using Nintendo.Foundation.IO;
using System.IO;
using YamlDotNet;
using System.Security.Cryptography;
using System.Text.RegularExpressions;
using MakeInitialImage;

namespace Commands
{
    public class MakePartitionCommand : SubCommandCommon
    {
        [CommandLineOption('i', "input", Description = "input config file")]
        public string InputFile { get; set; }

        [CommandLineOption('o', "output", Description = "output file")]
        public string OutputFile { get; set; }

        public override int Run()
        {
            var config = InitialImageConfigParser.Parse(new FileInfo(InputFile));

            Verify(config);

            var storage = config.StorageDefinitions[0];

            using (var writer = new FileInfo(OutputFile).OpenWrite())
            {
                var gptWriter = new GuidPartitionTableWriter();

                gptWriter.WriteMbr(writer);

                var diskGuid = GetDiskGuid(config);
                var rawGptHeader = GptHeader.Create(diskGuid, storage.Partitions.Count());

                var rawPartitions = (from partitionConfig in storage.Partitions
                                     select MakeRawPartition(partitionConfig, config)).ToList();

                var partitionEntryArrayBytesForCrc = MakePartitionEntryArrayBytesForCrc(rawPartitions);
                var partitionEntryArrayBytes = MakePartitionEntryArrayBytes(rawPartitions);

                rawGptHeader.HeaderCrc32 = 0;
                rawGptHeader.AlternateLba = (storage.Size.Bytes / storage.SectorSize.Bytes) - 1;
                rawGptHeader.FirstUsableLba = 34 ;
                rawGptHeader.LastUsableLba = rawGptHeader.AlternateLba - 32 - 1;
                rawGptHeader.PartitionEntryLba = 2;
                rawGptHeader.NumberOfPartitionEntries = storage.Partitions.Count();
                rawGptHeader.PartitionEntryArrayCrc32 = CalculateCrc32(partitionEntryArrayBytesForCrc);

                var headerBytesForCrc = MakeGptHeaderBytesForCrc(rawGptHeader);
                rawGptHeader.HeaderCrc32 = CalculateCrc32(headerBytesForCrc);

                Utility.WriteBinary<GptHeader>(writer, rawGptHeader);

                var padding = new byte[512 * 2 - writer.Position];
                writer.Write(padding, 0, padding.Length);

                writer.Write(partitionEntryArrayBytes, 0, partitionEntryArrayBytes.Length);
            }

            return 0;
        }

        private byte[] MakeGptHeaderBytes(GptHeader rawGptHeader)
        {
            using (var writer = new MemoryStream())
            {
                Utility.WriteBinary<GptHeader>(writer, rawGptHeader);

                var padding = new byte[512 - writer.Position];
                writer.Write(padding, 0, padding.Length);

                return writer.ToArray();
            }
        }

        private byte[] MakeGptHeaderBytesForCrc(GptHeader rawGptHeader)
        {
            using (var writer = new MemoryStream())
            {
                Utility.WriteBinary<GptHeader>(writer, rawGptHeader);
                var rawGptHeaderArray = writer.ToArray();
                var rawGptHeaderArrayWithoutReserved = new byte[rawGptHeader.HeaderSize];
                Array.Copy(rawGptHeaderArray, 0, rawGptHeaderArrayWithoutReserved, 0, rawGptHeader.HeaderSize);
                return rawGptHeaderArrayWithoutReserved;
            }
        }


        private byte[] MakePartitionEntryArrayBytes(List<GptPartitionEntry> rawPartitions)
        {
            using (var writer = new MemoryStream())
            {
                foreach (var partition in rawPartitions)
                {
                    Utility.WriteBinary<GptPartitionEntry>(writer, partition);
                }

                for (int i = 0; i < (128 - rawPartitions.Count()); i++)
                {
                    var nullPartition = new GptPartitionEntry();
                    Utility.WriteBinary<GptPartitionEntry>(writer, nullPartition);
                }

                return writer.ToArray();
            }
        }

        private byte[] MakePartitionEntryArrayBytesForCrc(List<GptPartitionEntry> rawPartitions)
        {
            using (var writer = new MemoryStream())
            {
                foreach (var partition in rawPartitions)
                {
                    Utility.WriteBinary<GptPartitionEntry>(writer, partition);
                }

                return writer.ToArray();
            }
        }

        private GptPartitionEntry MakeRawPartition(Partition partitionConfig, InitialImageConfig config)
        {
            return new GptPartitionEntry()
            {
                Attributes = GuidPartitionTableWriter.MakeAttribute(true),
                PartitionTypeGuid = Guid.Parse(partitionConfig.TypeGuid),
                UniquePartitionGuid = Guid.Parse(partitionConfig.Guid),
                PartitionName = partitionConfig.Name,
                StartingLba = partitionConfig.AddressAsNumber / config.StorageDefinitions[0].SectorSize.Bytes,
                EndingLba = (partitionConfig.AddressAsNumber + partitionConfig.Size.Bytes) / config.StorageDefinitions[0].SectorSize.Bytes - 1
            };
        }

        private uint CalculateCrc32(byte[] bytes)
        {
            uint r = 0xFFFFFFFF;

            foreach (var c in bytes)
            {
                r ^= c;

                for (int j = 0; j < 8; j++)
                {
                    if ((r & 1) != 0)
                    {
                        r = (r >> 1) ^ 0xEDB88320;
                    }
                    else
                    {
                        r >>= 1;
                    }
                }
            }

            return r ^ 0xFFFFFFFF;
        }

        private void Verify(bool expectedCondition, string errorMessage)
        {
            if(!expectedCondition)
            {
                throw new InvalidDataException(errorMessage);
            }
        }

        private void Verify(InitialImageConfig config)
        {
            Verify(config.StorageDefinitions.Count() == 1, "Number of storages must be one.");

            var storage = config.StorageDefinitions[0];

            Verify(storage.SectorSize.Bytes == 512,   "SectorSize must be 512 bytes.");
            Verify(storage.Size != null,              "Needs StorageDefinition.Size.");
            Verify(storage.Commands == null,          "Not supported StorageDefinition.Commands.");
            Verify(storage.SpecialPartitions == null, "Not supported StorageDefinition.SpecialPartitions.");

            foreach (var partition in storage.Partitions)
            {
                Verify(partition.Guid != null,     "Needs Guid.");
                Verify(partition.TypeGuid != null, "Needs TypeGuid.");
                Verify(partition.Address != null,  "Needs address.");
                Verify(partition.AllocationType == PartitionAllocationType.Fixed, "AllocationType must be Fixed.");
                Verify(partition.Size != null,     "Needs Size.");
            }
        }

        private Guid GetDiskGuid(InitialImageConfig config)
        {
            var storage = config.StorageDefinitions[0];
            if (storage.DiskGuid == null)
            {
                return Guid.NewGuid();
            }
            else
            {
                return new Guid(storage.DiskGuid);
            }
        }
    }
}
