﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <string>
#include <vector>
#include <list>
#include <algorithm>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fat/fat_FatFileSystem.h>

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include "MakeFatImage.h"

namespace
{
    const size_t EncryptionBlockSize = 16 * 1024;
    const size_t WorkBufferSize = 512 * 1024;
}

bool IsFile(std::string path)
{
    nn::fs::DirectoryEntryType directoryEntryType;
    NN_ABORT_UNLESS(nn::fs::GetEntryType(&directoryEntryType, path.c_str()).IsSuccess());

    return directoryEntryType == nn::fs::DirectoryEntryType_File;
}

bool IsDirectory(std::string path)
{
    nn::fs::DirectoryEntryType directoryEntryType;
    NN_ABORT_UNLESS(nn::fs::GetEntryType(&directoryEntryType, path.c_str()).IsSuccess());

    return directoryEntryType == nn::fs::DirectoryEntryType_Directory;
}

int64_t GetFileSize(std::string filename)
{
    nn::fs::FileHandle fileHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenFile(&fileHandle, filename.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT {
        nn::fs::CloseFile(fileHandle);
    };

    int64_t fileSize;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::GetFileSize(&fileSize, fileHandle));

    return fileSize;
}

nn::Result CopyFileData(nn::fs::FileHandle destinationHandle, int64_t destinationStartAddress, nn::fs::FileHandle sourceHandle, int64_t sourceStartAddress, int64_t copySize, size_t blockSize = 4 * 1024 * 1024)
{
    std::unique_ptr<uint8_t[]> buffer = std::unique_ptr<uint8_t[]>(new uint8_t[blockSize]);

    int64_t restSize = copySize;
    int64_t currentDestinationAddress = destinationStartAddress;
    int64_t currentSourceAddress = sourceStartAddress;

    while (0LL < restSize)
    {
        int64_t currentCopySize = std::min<int64_t>(restSize, blockSize);

        NN_UTILTOOL_LOG_DEBUG("Copy data: %08llx -> %08llx (size: %d).", currentSourceAddress, currentDestinationAddress, currentCopySize);

        NN_UTILTOOL_RESULT_DO(nn::fs::ReadFile(sourceHandle, currentSourceAddress, buffer.get(), static_cast<size_t>(currentCopySize)));
        NN_UTILTOOL_RESULT_DO(nn::fs::WriteFile(destinationHandle, currentDestinationAddress, buffer.get(), static_cast<size_t>(currentCopySize), nn::fs::WriteOption()));

        currentSourceAddress += currentCopySize;
        currentDestinationAddress += currentCopySize;
        restSize -= currentCopySize;
    }

    NN_RESULT_SUCCESS;
}

nn::Result CopyFile(std::string destinationPath, std::string sourceFile)
{
    NN_UTILTOOL_LOG_VERBOSE("copy file: %s <- %s", destinationPath.c_str(), sourceFile.c_str());

    NN_ABORT_UNLESS(IsFile(sourceFile));

    NN_UTILTOOL_RESULT_DO(nn::fs::CreateFile(destinationPath.c_str(), GetFileSize(sourceFile)));

    nn::fs::FileHandle destinationHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenFile(&destinationHandle, destinationPath.c_str(), nn::fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(destinationHandle);
    };

    nn::fs::FileHandle sourceHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenFile(&sourceHandle, sourceFile.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(sourceHandle);
    };

    int64_t fileSize;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::GetFileSize(&fileSize, sourceHandle));

    NN_UTILTOOL_RESULT_DO(nn::fs::SetFileSize(destinationHandle, fileSize));

    NN_UTILTOOL_RESULT_DO(CopyFileData(destinationHandle, 0, sourceHandle, 0, fileSize));

    NN_UTILTOOL_RESULT_DO(nn::fs::FlushFile(destinationHandle));

    NN_RESULT_SUCCESS;
}

bool ExistsFileSystemEntry(std::string path)
{
    nn::fs::DirectoryEntryType entryType;
    nn::Result result = nn::fs::GetEntryType(&entryType, path.c_str());
    if (result.IsSuccess())
    {
        return true;
    }
    if (result <= nn::fs::ResultPathNotFound())
    {
        return false;
    }
    else
    {
        NN_ABORT("unexpected error: result(module:%d, desc:%d), path=%s", result.GetModule(), result.GetDescription(), path.c_str());
        return false;
    }
}

std::string MakeTempFilePath(std::string path)
{
    return path + ".tmp";
}

nn::Result EnsureFileDeleted(std::string path)
{
    if (ExistsFileSystemEntry(path))
    {
        NN_UTILTOOL_LOG_DEBUG("Delete file: %s", path.c_str());
        NN_UTILTOOL_RESULT_DO(
            nn::fs::DeleteFile(path.c_str()));
    }

    NN_RESULT_SUCCESS;
}

nn::Result EnsureDirectory(std::string directoryName)
{
    if (!ExistsFileSystemEntry(directoryName.c_str()))
    {
        NN_UTILTOOL_RESULT_DO(nn::fs::CreateDirectory(directoryName.c_str()));
    }

    NN_RESULT_SUCCESS;
}

std::string CombinePath(std::string head, std::string tail)
{
    return head + "/" + tail;
}

bool IsArchiveRootPath(std::string path)
{
    return *path.rbegin() == ':';
}

nn::Result CopyDirectory(std::string destinationPath, std::string inputDirectory)
{
    NN_UTILTOOL_LOG_VERBOSE("copy directory: %s <- %s", destinationPath.c_str(), inputDirectory.c_str());

    if (!IsArchiveRootPath(destinationPath))
    {
        NN_UTILTOOL_RESULT_DO(EnsureDirectory(destinationPath));
    }

    nn::fs::DirectoryHandle directoryHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenDirectory(
        &directoryHandle,
        inputDirectory.c_str(),
        static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_Directory | nn::fs::OpenDirectoryMode_File)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directoryHandle); };

    int64_t numDirectories;
    NN_UTILTOOL_RESULT_DO(nn::fs::GetDirectoryEntryCount(&numDirectories, directoryHandle));

    NN_UTILTOOL_LOG_DEBUG("found entries: %d", numDirectories);

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(static_cast<int>(numDirectories));

    int64_t readNum;
    NN_UTILTOOL_RESULT_DO(nn::fs::ReadDirectory(&readNum, directoryEntries.data(), directoryHandle, numDirectories));

    NN_UTILTOOL_LOG_DEBUG("read entries: %d", readNum);

    for each (nn::fs::DirectoryEntry directoryEntry in directoryEntries)
    {
        std::string sourcePath = CombinePath(inputDirectory, std::string(directoryEntry.name));
        std::string destinationEntry = CombinePath(destinationPath, std::string(directoryEntry.name));

        switch (directoryEntry.directoryEntryType)
        {
        case nn::fs::DirectoryEntryType_File:
            {
                NN_UTILTOOL_RESULT_DO(CopyFile(destinationEntry, sourcePath));
                break;
            }
        case nn::fs::DirectoryEntryType_Directory:
            {
                NN_UTILTOOL_RESULT_DO(CopyDirectory(destinationEntry, sourcePath));
                break;
            }
        default:
            {
                NN_ABORT("Unsupported DirectoryEntryType: type=%d, path=%s", directoryEntry.directoryEntryType, sourcePath.c_str());
                break;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

struct AccessRange
{
    int64_t address;
    int64_t size;

    static AccessRange MakeFromSize(int64_t address, int64_t size)
    {
        AccessRange ret = { address, size };
        return ret;
    }

    static AccessRange MakeFromRange(int64_t head, int64_t tail)
    {
        NN_ASSERT(head <= tail);
        AccessRange ret = { head, tail - head };
        return ret;
    }

    bool IsZero() const
    {
        return size == 0;
    }

    bool IsOverrapped(const AccessRange &range) const
    {
        return (address - range.size) <= range.address && range.address < (address + size + range.size);
    }

    AccessRange Union(const AccessRange &range)
    {
        NN_ABORT_UNLESS(IsOverrapped(range));

        return AccessRange::MakeFromRange(std::min(address, range.address), std::max(address + size, range.address + range.size));
    }

};

class AccessArea
{
public:
    void Access(int64_t address, int64_t size);
    void Dump();
    void Compact();
    std::list<AccessRange>&Get() { return m_AccessRangeSet; }
private:
    std::list<AccessRange> m_AccessRangeSet;
};

void AccessArea::Access(int64_t address, int64_t size)
{
    m_AccessRangeSet.push_back(AccessRange::MakeFromSize(address, size));
}

void AccessArea::Dump()
{
    NN_UTILTOOL_LOG_DEBUG("==============================");
    NN_UTILTOOL_LOG_DEBUG("DumpAccessRange");
    NN_UTILTOOL_LOG_DEBUG("==============================");
    NN_UTILTOOL_LOG_DEBUG("Number of fragments: %d", m_AccessRangeSet.size());

    for (auto accessAddress : m_AccessRangeSet)
    {
        NN_UTILTOOL_LOG_DEBUG("Range: %08llx-%08llx", accessAddress.address, accessAddress.address + accessAddress.size);
    }
}

void AccessArea::Compact()
{
    m_AccessRangeSet.sort([](AccessRange &a, AccessRange &b) { return a.address < b.address; });

    std::list<AccessRange> compactedArea;

    AccessRange lastRange = { 0, 0 };

    for (auto range : m_AccessRangeSet)
    {
        if (lastRange.IsOverrapped(range))
        {
            lastRange = lastRange.Union(range);
        }
        else
        {
            if (!lastRange.IsZero())
            {
                compactedArea.push_back(lastRange);
            }

            lastRange = range;
        }
    }

    if (!lastRange.IsZero())
    {
        compactedArea.push_back(lastRange);
    }

    m_AccessRangeSet = compactedArea;
}

class RecordingStorageAdapter : public nn::fs::IStorage
{
public:
    RecordingStorageAdapter(nn::fs::IStorage *pStorage, AccessArea *pAccessArea)
        : m_pStorage(pStorage), m_pAccessArea(pAccessArea) {}

    virtual ~RecordingStorageAdapter() NN_NOEXCEPT{}

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_UTILTOOL_LOG_DEBUG("Read storage: %08llx-%08llx(size:%d)", offset, offset + size, size);
        m_pAccessArea->Access(offset, size);
        return m_pStorage->Read(offset, buffer, size);
    }

    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_UTILTOOL_LOG_DEBUG("Write storage: %08llx-%08llx(size:%d)", offset, offset + size, size);
        m_pAccessArea->Access(offset, size);
        return m_pStorage->Write(offset, buffer, size);
    }

    virtual nn::Result Flush() NN_NOEXCEPT
    {
        return m_pStorage->Flush();
    }

    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT
    {
        return m_pStorage->SetSize(size);
    }

    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT
    {
        return m_pStorage->GetSize(outValue);
    }

private:
    nn::fs::IStorage *m_pStorage;
    AccessArea *m_pAccessArea;
};

nn::Result LoadAesStorageSource(AesXtsStorageSource *pAesStorageSource, const char* encryptionKeyFile)
{
    nn::fs::FileHandle handle;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&handle, encryptionKeyFile, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

    size_t readSize = 0;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::ReadFile(&readSize, handle, 0, pAesStorageSource, sizeof(AesXtsStorageSource)));
    NN_ABORT_UNLESS_EQUAL(sizeof(AesXtsStorageSource), readSize);

    NN_RESULT_SUCCESS;
}

nn::Result MakeFatImage(const char* outputPath, AccessArea *pAccessArea, int64_t sizeMiB, const char* inputDirectory, bool enableEncryption, const char *encryptionKeyFile, int64_t ensuredFreeSpace)
{
    nn::fs::FileHandle baseHandle;

    NN_UTILTOOL_RESULT_DO(EnsureFileDeleted(outputPath));

    NN_UTILTOOL_RESULT_DO(
        nn::fs::CreateFile(outputPath, sizeMiB * 1024 * 1024));

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&baseHandle, outputPath, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::FlushFile(baseHandle);
        nn::fs::CloseFile(baseHandle);
    };

    nn::fs::FileHandleStorage baseStorage(baseHandle);

    std::unique_ptr<nn::fs::IStorage> recordingStorageAdapter;
    std::unique_ptr<nn::fs::IStorage> encryptionStorage;
    std::unique_ptr<nn::fs::IStorage> alignedStorage;

    nn::fs::IStorage *pStorageAccessor = nullptr;

    if(enableEncryption)
    {
        AesXtsStorageSource aesStorageSource;
        NN_UTILTOOL_RESULT_DO(
            LoadAesStorageSource(&aesStorageSource, encryptionKeyFile));
        char iv[nn::fssystem::AesXtsStorage::IvSize];
        std::memset(iv, 0, nn::fssystem::AesXtsStorage::IvSize);
        recordingStorageAdapter.reset(new RecordingStorageAdapter(&baseStorage, pAccessArea));
        encryptionStorage.reset(new nn::fssystem::AesXtsStorage(recordingStorageAdapter.get(), aesStorageSource.key1, aesStorageSource.key2, nn::fssystem::AesXtsStorage::KeySize, iv, nn::fssystem::AesXtsStorage::IvSize, EncryptionBlockSize));
        alignedStorage.reset(new nn::fssystem::AlignmentMatchingStoragePooledBuffer<1>(encryptionStorage.get(), EncryptionBlockSize));
        pStorageAccessor = alignedStorage.get();
    }
    else
    {
        recordingStorageAdapter.reset(new RecordingStorageAdapter(&baseStorage, pAccessArea));
        pStorageAccessor = recordingStorageAdapter.get();
    }

    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    NN_ASSERT(fatFs != nullptr);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    NN_UTILTOOL_RESULT_DO(
        fatFs->Initialize(pStorageAccessor, cacheBuffer.get(), cacheBufferSize));

    NN_UTILTOOL_RESULT_DO(
        fatFs->Format());

    NN_UTILTOOL_RESULT_DO(
        fatFs->Mount());

    NN_UTILTOOL_RESULT_DO(
        nn::fs::fsa::Register("fat", std::move(fatFs)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::Unmount("fat");
    };

    NN_UTILTOOL_RESULT_DO(CopyDirectory(std::string("fat:"), std::string(inputDirectory)));

    int64_t freeSpaceSize = 0;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::GetFreeSpaceSize(&freeSpaceSize, "fat:/"));

    int64_t totalSize = 0;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::GetTotalSpaceSize(&totalSize, "fat:/"));

    if (freeSpaceSize < ensuredFreeSpace)
    {
        NN_UTILTOOL_LOG_ERROR("Free space is not enough.");
        NN_UTILTOOL_LOG_ERROR("  Total Size:  %lld bytes", totalSize);
        NN_UTILTOOL_LOG_ERROR("  Free Space:  %lld bytes", freeSpaceSize);
        NN_UTILTOOL_LOG_ERROR("  Needs Space: %lld bytes", ensuredFreeSpace);
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultUsableSpaceNotEnough());
    }

    NN_UTILTOOL_RESULT_DO(nn::fs::FlushFile(baseHandle));

    NN_RESULT_SUCCESS;
}

struct SparseImageHeader
{
    char        signature[8];         // "SPRSIMG\0"
    uint64_t    numChunks;

    static SparseImageHeader MakeInitialHeader()
    {
        SparseImageHeader initialHeader = { "SPRSIMG", 0 };
        return initialHeader;
    }
};

struct SparseImageChunk
{
    int64_t    address;              // 書き込み先の場所（バイト単位）
    int64_t    size;                 // data のサイズのみ表す。構造体のサイズは含めない

    static SparseImageChunk Make(int64_t address, int64_t size)
    {
        SparseImageChunk ret = { address, size };
        return ret;
    }
};

nn::Result MakeSpaseImage(std::string outputPath, AccessArea *pAccessArea, std::string inputImage)
{
    pAccessArea->Compact();

    nn::fs::FileHandle inputHandle;

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&inputHandle, inputImage.c_str(), static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(inputHandle);
    };

    nn::fs::FileHandle outputHandle;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::CreateFile(outputPath.c_str(), 0));
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&outputHandle, outputPath.c_str(), static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(outputHandle);
    };

    SparseImageHeader header = SparseImageHeader::MakeInitialHeader();
    header.numChunks = pAccessArea->Get().size();

    int64_t currentAddress = 0;

    NN_UTILTOOL_RESULT_DO(nn::fs::WriteFile(outputHandle, currentAddress, &header, sizeof(SparseImageHeader), nn::fs::WriteOption()));
    currentAddress += sizeof(SparseImageHeader);

    for (auto accessRange : pAccessArea->Get())
    {
        SparseImageChunk chunk = SparseImageChunk::Make(accessRange.address, accessRange.size);

        NN_UTILTOOL_RESULT_DO(nn::fs::WriteFile(outputHandle, currentAddress, &chunk, sizeof(SparseImageChunk), nn::fs::WriteOption()));
        currentAddress += sizeof(SparseImageChunk);

        NN_UTILTOOL_RESULT_DO(CopyFileData(outputHandle, currentAddress, inputHandle, accessRange.address, accessRange.size));
        currentAddress += accessRange.size;
    }

    NN_UTILTOOL_RESULT_DO(nn::fs::FlushFile(outputHandle));

    NN_UTILTOOL_LOG_VERBOSE("Created sparse image: %s", outputPath.c_str());
    NN_UTILTOOL_LOG_VERBOSE("Raw image size: %lld", GetFileSize(inputImage));
    NN_UTILTOOL_LOG_VERBOSE("Spase image size: %lld", currentAddress);
    NN_UTILTOOL_LOG_VERBOSE("Write chunks: %d", header.numChunks);

    NN_RESULT_SUCCESS;
}

nn::Result MakeFatImage(const char* outputPath, long sizeMiB, const char* inputDirectory, bool makeSparseImage, bool enableEncryption, const char *encryptionKeyFile, int64_t ensuredFreeSpace)
{
    std::string tempFilePath = MakeTempFilePath(outputPath);

    NN_UTILTOOL_RESULT_DO(EnsureFileDeleted(outputPath));
    NN_UTILTOOL_RESULT_DO(EnsureFileDeleted(tempFilePath));

    AccessArea accessArea;

    NN_UTIL_SCOPE_EXIT {
        EnsureFileDeleted(tempFilePath);
    };

    NN_UTILTOOL_RESULT_DO(MakeFatImage(tempFilePath.c_str(), &accessArea, sizeMiB, inputDirectory, enableEncryption, encryptionKeyFile, ensuredFreeSpace));

    if (makeSparseImage)
    {
        NN_UTILTOOL_RESULT_DO(MakeSpaseImage(outputPath, &accessArea, tempFilePath));
    }
    else
    {
        NN_UTILTOOL_RESULT_DO(CopyFile(outputPath, tempFilePath));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListDirectory(std::string path)
{
    NN_LOG("%s\n", path.c_str());

    nn::fs::DirectoryHandle directoryHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenDirectory(
        &directoryHandle,
        path.c_str(),
        static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_Directory | nn::fs::OpenDirectoryMode_File)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directoryHandle); };

    int64_t numDirectories;
    NN_UTILTOOL_RESULT_DO(nn::fs::GetDirectoryEntryCount(&numDirectories, directoryHandle));

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(static_cast<int>(numDirectories));

    int64_t readNum;
    NN_UTILTOOL_RESULT_DO(nn::fs::ReadDirectory(&readNum, directoryEntries.data(), directoryHandle, numDirectories));

    for each (nn::fs::DirectoryEntry directoryEntry in directoryEntries)
    {
        std::string nextPath = CombinePath(path, std::string(directoryEntry.name));

        if (std::string(directoryEntry.name) == std::string(""))
        {
            continue;
        }

        if (directoryEntry.directoryEntryType == nn::fs::DirectoryEntryType_File || std::string(directoryEntry.name) == std::string("Release"))
        {
            NN_LOG("next directory\n", nextPath.c_str());
            NN_UTILTOOL_RESULT_DO(ListDirectory(nextPath));
        }
        else
        {
            NN_LOG("%s\n", nextPath.c_str());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListDirectoryInFatImage(const char* path)
{
    nn::fs::FileHandle baseHandle;

    NN_ABORT_UNLESS(ExistsFileSystemEntry(std::string(path)));

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&baseHandle, path, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)));

    nn::fs::FileHandleStorage baseStorage(baseHandle);

    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    NN_ASSERT(fatFs != nullptr);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    NN_UTILTOOL_RESULT_DO(
        fatFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize));

    NN_UTILTOOL_RESULT_DO(
        fatFs->Mount());

    NN_UTILTOOL_RESULT_DO(
        nn::fs::fsa::Register("fat", std::move(fatFs)));

    NN_UTILTOOL_RESULT_DO(ListDirectory(std::string("fat:/Release")));

    nn::fs::Unmount("fat");

    nn::fs::CloseFile(baseHandle);

    NN_RESULT_SUCCESS;
}

