﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <string>
#include <vector>
#include <list>
#include <algorithm>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fat/fat_FatFileSystem.h>

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include "MakeFatImage.h"
#include "ListFiles.h"

namespace
{
    const size_t EncryptionBlockSize = 16 * 1024;
    const size_t WorkBufferSize = 512 * 1024;
}

nn::Result ListFiles(std::string path)
{
    auto enumeratePath = path;
    if (enumeratePath == "fat:")
    {
        enumeratePath = "fat:/";
    }

    if (!IsDirectory(enumeratePath) && path != "fat:")
    {
        int64_t fileSize;
        nn::fs::FileHandle fileHandle;
        NN_UTILTOOL_RESULT_DO(nn::fs::OpenFile(&fileHandle, path.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::CloseFile(fileHandle);
        };

        NN_UTILTOOL_RESULT_DO(
            nn::fs::GetFileSize(&fileSize, fileHandle));
        NN_UTILTOOL_LOG_MESSAGE("%12lld %s", fileSize, path.c_str());

        NN_RESULT_SUCCESS;
    }

    NN_UTILTOOL_LOG_MESSAGE("%12s %s", "dir", path.c_str());

    nn::fs::DirectoryHandle directoryHandle;
    NN_UTILTOOL_RESULT_DO(nn::fs::OpenDirectory(
        &directoryHandle,
        enumeratePath.c_str(),
        static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_Directory | nn::fs::OpenDirectoryMode_File)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directoryHandle); };

    int64_t numDirectories;
    NN_UTILTOOL_RESULT_DO(nn::fs::GetDirectoryEntryCount(&numDirectories, directoryHandle));

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(static_cast<int>(numDirectories));

    int64_t readNum;
    NN_UTILTOOL_RESULT_DO(nn::fs::ReadDirectory(&readNum, directoryEntries.data(), directoryHandle, numDirectories));

    for each (nn::fs::DirectoryEntry directoryEntry in directoryEntries)
    {
        std::string nextPath = CombinePath(path, std::string(directoryEntry.name));

        if (std::string(directoryEntry.name) == std::string(""))
        {
            continue;
        }

        NN_UTILTOOL_RESULT_DO(ListFiles(nextPath));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListFiles(const char* inputPath, bool enableEncryption, const char *encryptionKeyFile)
{
    nn::fs::FileHandle baseHandle;

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenFile(&baseHandle, inputPath, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(baseHandle);
    };

    nn::fs::FileHandleStorage baseStorage(baseHandle);

    std::unique_ptr<nn::fs::IStorage> encryptionStorage;
    std::unique_ptr<nn::fs::IStorage> alignedStorage;

    nn::fs::IStorage *pStorageAccessor = nullptr;

    if (enableEncryption)
    {
        AesXtsStorageSource aesStorageSource;
        NN_UTILTOOL_RESULT_DO(
            LoadAesStorageSource(&aesStorageSource, encryptionKeyFile));
        char iv[nn::fssystem::AesXtsStorage::IvSize];
        std::memset(iv, 0, nn::fssystem::AesXtsStorage::IvSize);
        encryptionStorage.reset(new nn::fssystem::AesXtsStorage(&baseStorage, aesStorageSource.key1, aesStorageSource.key2, nn::fssystem::AesXtsStorage::KeySize, iv, nn::fssystem::AesXtsStorage::IvSize, EncryptionBlockSize));
        alignedStorage.reset(new nn::fssystem::AlignmentMatchingStoragePooledBuffer<1>(encryptionStorage.get(), EncryptionBlockSize));
        pStorageAccessor = alignedStorage.get();
    }
    else
    {
        pStorageAccessor = &baseStorage;
    }

    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    NN_ASSERT(fatFs != nullptr);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    NN_UTILTOOL_RESULT_DO(
        fatFs->Initialize(pStorageAccessor, cacheBuffer.get(), cacheBufferSize));

    NN_UTILTOOL_RESULT_DO(
        fatFs->Mount());

    NN_UTILTOOL_RESULT_DO(
        nn::fs::fsa::Register("fat", std::move(fatFs)));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::Unmount("fat");
    };

    NN_UTILTOOL_RESULT_DO(ListFiles(std::string("fat:")));

    NN_RESULT_SUCCESS;
}
