﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <random>
#include <string>
#include <vector>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_HostPackageInstallTask.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "InstallNspOnHost.h"

nn::Result InstallNsp(std::string outputDirectory, std::string nspPath, nn::fs::BisPartitionId bisPartitionId, nn::ncm::StorageId storage)
{
    nn::fs::DirectoryEntryType entryType;

    nn::Result result = nn::fs::GetEntryType(&entryType, outputDirectory.c_str());
    if(result <= nn::fs::ResultPathNotFound())
    {
        NN_LOG("[ERROR] OutputPath not found: path=%s", outputDirectory.c_str());
        NN_RESULT_THROW(result);
    }
    else if (result.IsFailure())
    {
        NN_LOG("[ERROR] OutputPath is invalid: path=%s", outputDirectory.c_str());
        NN_RESULT_THROW(result);
    }

    if(entryType != nn::fs::DirectoryEntryType_Directory)
    {
        NN_LOG("[ERROR] OutputPath is not directory: path=%s", outputDirectory.c_str());
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }

    static char s_Buffer[1024 * 1024];

    nn::fs::SetBisRootForHost(bisPartitionId, outputDirectory.c_str());

    nn::ncm::Initialize();

    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, nspPath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{nn::fs::CloseFile(file);};

    nn::ncm::HostPackageInstallTask task;
    task.Initialize(file, storage, s_Buffer, sizeof(s_Buffer), true);
    NN_RESULT_DO(task.Prepare());
    NN_RESULT_DO(task.Execute());
    NN_RESULT_DO(task.Commit());

    NN_RESULT_SUCCESS;
}

nn::Result InstallNsp(std::string installStorage, std::string outputDirectory, std::string nspPath)
{
    if(installStorage == "system")
    {
        NN_RESULT_DO(
            InstallNsp(outputDirectory, nspPath, nn::fs::BisPartitionId::System, nn::ncm::StorageId::BuildInSystem));
    }
    else if(installStorage == "user")
    {
        NN_RESULT_DO(
            InstallNsp(outputDirectory, nspPath, nn::fs::BisPartitionId::User, nn::ncm::StorageId::BuildInUser));
    }
    else
    {
        NN_ABORT("unknown storageType: %s", installStorage.c_str());
    }

    NN_RESULT_SUCCESS;
}
