﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <random>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/os.h>

#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentIdUtil.h>

#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <new>

bool IsFile(std::string path)
{
    nn::fs::DirectoryEntryType directoryEntryType;
    NN_ABORT_UNLESS(nn::fs::GetEntryType(&directoryEntryType, path.c_str()).IsSuccess());

    return directoryEntryType == nn::fs::DirectoryEntryType_File;
}

bool IsDirectory(std::string path)
{
    nn::fs::DirectoryEntryType directoryEntryType;
    NN_ABORT_UNLESS(nn::fs::GetEntryType(&directoryEntryType, path.c_str()).IsSuccess());

    return directoryEntryType == nn::fs::DirectoryEntryType_Directory;
}

int64_t GetFileSize(std::string filename)
{
    nn::fs::FileHandle fileHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, filename.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(fileHandle);
    };

    int64_t fileSize;
    NN_RESULT_DO(
        nn::fs::GetFileSize(&fileSize, fileHandle));

    return fileSize;
}

bool ExistsFileSystemEntry(std::string path)
{
    nn::fs::DirectoryEntryType entryType;
    nn::Result result = nn::fs::GetEntryType(&entryType, path.c_str());
    if (result.IsSuccess())
    {
        return true;
    }
    if (result <= nn::fs::ResultPathNotFound())
    {
        return false;
    }
    else
    {
        NN_ABORT("unexpected error: result(module:%d, desc:%d), path=%s", result.GetModule(), result.GetDescription(), path.c_str());
        return false;
    }
}


nn::Result ReadContentId(nn::ncm::ContentId *pOut, std::string ncaPath)
{
    nn::crypto::Sha256Generator sha256generator;
    sha256generator.Initialize();

    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, ncaPath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    const size_t blockSize = 4 * 1024 * 1024;
    std::unique_ptr<uint8_t[]> buffer = std::unique_ptr<uint8_t[]>(new uint8_t[blockSize]);

    int64_t restSize = GetFileSize(ncaPath);
    int64_t currentSourceAddress = 0;

    while (0LL < restSize)
    {
        int64_t currentSize = std::min<int64_t>(restSize, blockSize);

        NN_RESULT_DO(nn::fs::ReadFile(file, currentSourceAddress, buffer.get(), static_cast<size_t>(currentSize)));

        sha256generator.Update(buffer.get(), static_cast<size_t>(currentSize));

        currentSourceAddress += currentSize;
        restSize -= currentSize;
    }

    nn::Bit8 sha256hash[nn::crypto::Sha256Generator::HashSize];
    sha256generator.GetHash(&sha256hash, sizeof(sha256hash));

    nn::ncm::ContentId contentId = {};
    std::memcpy(contentId.data, sha256hash, sizeof(contentId.data));

    *pOut = contentId;

    NN_RESULT_SUCCESS;
}

nn::Result WritePlaceHolderFromFile(nn::ncm::ContentStorage *pContentStorage, nn::ncm::PlaceHolderId placeHolderId, std::string ncaPath, size_t blockSize = 4 * 1024 * 1024)
{
    std::unique_ptr<uint8_t[]> buffer = std::unique_ptr<uint8_t[]>(new uint8_t[blockSize]);

    nn::fs::FileHandle sourceHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&sourceHandle, ncaPath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(sourceHandle); };

    int64_t restSize = GetFileSize(ncaPath);
    int64_t currentDestinationAddress = 0;
    int64_t currentSourceAddress = 0;

    while (0LL < restSize)
    {
        int64_t currentCopySize = std::min<int64_t>(restSize, blockSize);

        NN_LOG("Copy data: %08llx -> %08llx (size: %d).\n", currentSourceAddress, currentDestinationAddress, currentCopySize);

        NN_RESULT_DO(nn::fs::ReadFile(sourceHandle, currentSourceAddress, buffer.get(), static_cast<size_t>(currentCopySize)));
        NN_RESULT_DO(pContentStorage->WritePlaceHolder(placeHolderId, currentDestinationAddress, buffer.get(), static_cast<size_t>(currentCopySize)));

        currentSourceAddress += currentCopySize;
        currentDestinationAddress += currentCopySize;
        restSize -= currentCopySize;
    }

    NN_RESULT_SUCCESS;
}

nn::Result InstallNca(std::string outputDirectory, std::string ncaPath)
{
    NN_ABORT_UNLESS(ExistsFileSystemEntry(outputDirectory));
    NN_ABORT_UNLESS(ExistsFileSystemEntry(ncaPath));

    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentStorageFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentStorage> ContentStorageShared;

    NN_LOG("Install nca: %s -> %s\n", ncaPath.c_str(), outputDirectory.c_str());

    NN_RESULT_DO(nn::ncm::ContentStorageImpl::InitializeBase(outputDirectory.c_str()));

    auto contentStorageShared = ContentStorageFactory::CreateSharedEmplaced<nn::ncm::IContentStorage, nn::ncm::ContentStorageImpl>();
    ContentStorageFactory::GetEmplacedImplPointer<nn::ncm::ContentStorageImpl>(contentStorageShared)->Initialize(outputDirectory.c_str(), nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat4KCluster, nn::ncm::MakeFlatPlaceHolderFilePath, false);
    nn::ncm::ContentStorage contentStorage(contentStorageShared);

    nn::ncm::ContentId contentId = {};
    NN_RESULT_DO(ReadContentId(&contentId, ncaPath));

    bool hasContent = false;
    NN_RESULT_DO(contentStorage.Has(&hasContent, contentId));
    if (hasContent)
    {
        char idString[nn::ncm::ContentIdStringLength + 1] = {};
        GetStringFromContentId(idString, sizeof(idString), contentId);
        NN_LOG("[ERROR] Content already exist: contentId=%s\n", idString);
    }

    auto placeHolderId = contentStorage.GeneratePlaceHolderId();

    NN_RESULT_DO(contentStorage.CreatePlaceHolder(placeHolderId, contentId, GetFileSize(ncaPath)));

    NN_RESULT_DO(WritePlaceHolderFromFile(&contentStorage, placeHolderId, ncaPath));

    NN_RESULT_DO(contentStorage.Register(placeHolderId, contentId));

    NN_RESULT_SUCCESS;
}

namespace{

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return free(p);
    }

} // namespace


extern "C" void nnMain()
{
    char **argv = nn::os::GetHostArgv();
    int argc = nn::os::GetHostArgc();

    if (argc != 3)
    {
        NN_LOG("usage:\n");
        NN_LOG("    InstallNcaOnHost <OutputDirecotry> <Nca>\r\n");

        for(int i=0; i< argc;i++)
        {
            NN_LOG("input[%d]: %s\r\n", i, argv[i]);
        }

        return;
    }

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

    nn::Result result = InstallNca(argv[1], argv[2]);

    if (result.IsFailure())
    {
        NN_LOG("failure: module=%d, description=%d\n", result.GetModule(), result.GetDescription());
        NN_LOG("Arguments:\n");
        int numArgs = nn::os::GetHostArgc();
        for(int i = 0; i < numArgs; i++)
        {
            NN_LOG("  %d: %s\n", i, nn::os::GetHostArgv()[i]);
        }

        std::exit(1);
    }
    else
    {
        NN_LOG("done.\n");
    }
}

