﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using CommandUtility;
using System.Text.RegularExpressions;

namespace CmacFileTool.Commands
{
    public class ChangeKeyCommand
    {
        [CommandLineOption('i', "input",
            Description = "set raw intial image or initial image with cmac",
            IsRequired = true)]
        public string InputFile { get; set; }

        [CommandLineOption('o', "output",
            Description = "set output as initial image with cmac",
            IsRequired = true)]
        public string OutputFile { get; set; }

        [CommandLineOption("new-key",
            Description = "set new key.(binary-file-path or text-file-path or 32-hexchars)",
            IsRequired = true)]
        public string NewKey { get; set; }

        [CommandLineOption("cmac-key",
            Description = "set cmac key for test.(default: genarate key from random number)")]
        public string CmacKey { get; set; }

        [CommandLineOption("source-key",
            Description = "set source key.(binary-file-path or text-file-path or 32-hexchars)")]
        public string SourceKey { get; set; }

        public void Run()
        {
            var sourceKek = AppendCmacCommand.LoadKeyFromArgument(SourceKey);
            var newKek = AppendCmacCommand.LoadKeyFromArgument(NewKey);
            var inputFileInfo = new FileInfo(InputFile);
            var outputFileInfo = new FileInfo(OutputFile);

            byte[] cmacKey;
            if (CmacKey == null)
            {
                cmacKey = CryptUtility.GenerateKey();
            }
            else
            {
                cmacKey = AppendCmacCommand.LoadKeyFromArgument(CmacKey);
            }

            using (var reader = inputFileInfo.OpenRead())
            using (var writer = outputFileInfo.OpenWrite())
            {
                // ヘッダ読む
                var oldHeader = CmacFileHeader.Load(reader, sourceKek);

                // ヘッダを書く
                var newHeader = new CmacFileHeader(oldHeader.Signature, oldHeader.BlockSize, oldHeader.NumberOfBlocks, newKek, cmacKey);
                newHeader.WriteToStream(writer);

                for (int blockIndex = 0; blockIndex < oldHeader.NumberOfBlocks; blockIndex++)
                {
                    // CMAC の読み込み
                    var cmac = new byte[16];
                    reader.Read(cmac, 0, cmac.Length);

                    // データの読み込み
                    var indexData = BinaryUtility.ToBinary<int>(blockIndex);
                    var buffer = new byte[indexData.Length + oldHeader.BlockSize];
                    indexData.CopyTo(buffer, 0);
                    reader.Read(buffer, indexData.Length, oldHeader.BlockSize);

                    // 検証
                    CmacUtility.Verify(buffer, cmac, oldHeader.CmacKey);

                    // 新しい鍵での書き込み
                    byte[] newCmac = CmacUtility.CalculateCmac(buffer, newHeader.CmacKey);
                    writer.Write(newCmac, 0, newCmac.Length);
                    writer.Write(buffer, indexData.Length, newHeader.BlockSize);
                }
            }
        }
    }
}
