﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Xml;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using MakeFirmwareArchive;
using System.Xml.Linq;

namespace MakeCupArchive
{
    public class MakeCupArchiveArguments
    {
        [CommandLineOption('o', "output", Description = "output cup nsp path", IsRequired = true)]
        public string OutputNspPath { get; set; }

        [CommandLineOption('i', "input", Description = "input nfa path", IsRequired = true)]
        public string InputNfaPath { get; set; }

        [CommandLineOption("system-meta-id", Description = " ", IsRequired = false)]
        public string Id { get; set; }

        [CommandLineOption("system-meta-version", Description = " ", IsRequired = false)]
        public string Version { get; set; }

        [CommandLineOption("keyconfig", Description = " ", IsRequired = true)]
        public string KeyConfigPath { get; set; }

    }

    class Program
    {
        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            MakeCupArchiveArguments parameters = new MakeCupArchiveArguments();
            if (CommandLineParser.Default.ParseArgs<MakeCupArchiveArguments>(args, out parameters))
            {
                ValidateParameters(parameters);
                MakeCupArchive(new FileInfo(parameters.OutputNspPath), new FileInfo(parameters.InputNfaPath), new FileInfo(parameters.KeyConfigPath), parameters.Id, parameters.Version);
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[Error] {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void ValidateParameters(MakeCupArchiveArguments parameters)
        {
            if (parameters.Version != null && !Regex.IsMatch(parameters.Version, "^[0-9]+$"))
            {
                throw new Exception(string.Format("Invalid system meta version: {0}", parameters.Version));
            }

            if (parameters.Id != null && !Regex.IsMatch(parameters.Id, "^0x([0-9a-fA-F]){16}$"))
            {
                throw new Exception(string.Format("Invalid system meta id: {0}", parameters.Id));
            }
            if ((parameters.Version != null && parameters.Id == null) || (parameters.Version == null && parameters.Id != null))
            {
                throw new Exception(string.Format("Both --system-meta-id and --system-meta-version are required if use"));
            }
        }
        private static string GetUpdatePartitionId(IEnumerable<string> contentIds)
        {
            using (var sha = new System.Security.Cryptography.SHA256Managed())
            {
                // 文字を単純連結したものを、バイナリにしてハッシュをとる
                // contentIds が 16 進数であることは使っていないことに注意
                var str = string.Join("", contentIds);
                var hash = sha.ComputeHash(Encoding.UTF8.GetBytes(str));
                return string.Join("", hash.Select(b => string.Format("{0,2:x2}", b)));
            }
        }

        private static void CreateUpdatePartitionXml(FileInfo outputXml, DirectoryInfo updatePartitionDirectory)
        {
            // 中に含まれている nca 群を列挙する
            var nca = updatePartitionDirectory.EnumerateFiles()
                                              .Where(f => f.Extension == ".nca");
            var contentIds = nca.Select(f => f.Name.Substring(0, 32))
                                .OrderBy(s => s, StringComparer.Ordinal);
            var uppId = GetUpdatePartitionId(contentIds);
            var systemUpdateMeta = updatePartitionDirectory.EnumerateFiles().First(f => IsSystemUpdateMetaXml(f));
            var systemUpdateMetaXml = XElement.Load(systemUpdateMeta.FullName);

            XmlUtility.WriteXml(outputXml, new XElement("UpdatePartition",
                                                        new XElement("Hash", new XText(uppId.Substring(0, 16))),
                                                        new XElement("FullHash", new XText(uppId)),
                                                        new XElement("SystemUpdateMetaId", new XText(systemUpdateMetaXml.Element("Id").Value)),
                                                        new XElement("SystemUpdateMetaVersion", new XText(systemUpdateMetaXml.Element("Version").Value)),
                                                        new XElement("Ncas", nca.Select(p => new XElement("Nca", new XText(p.Name))))
                                                        ));

        }

        private static void MakeCupArchive(FileInfo outputNsp, FileInfo inputNfa, FileInfo keyConfig, string id, string version)
        {
            using(var tempHolder = new TemporaryFileHolder("MakeCupArchive"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var updatePartitionDirectory = tempHolder.CreateTemporaryDirectory("UppEntries");
                var systemUpdateMetaXml = tempHolder.CreateTemporaryFilePath("systemUpdateMeta", ".xml");
                var systemUpdateMetaNsp = tempHolder.CreateTemporaryFilePath("systemUpdateMeta", ".nsp");
                var updatePartitionXml = new FileInfo(Path.Combine(updatePartitionDirectory.FullName, "UpdatePartition.xml"));

                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputNfa);

                List<FileInfo> nspInfos = archiveDirectory.GetSystemNspList(extractedNfa);

                if (id != null)
                {
                    MakeSystemUpdateMeta(systemUpdateMetaXml, archiveDirectory.SystemDirectory, id, version);
                    MakeSystemUpdateMetaNsp(systemUpdateMetaNsp, systemUpdateMetaXml, keyConfig);

                    nspInfos.Add(systemUpdateMetaNsp);
                }
                foreach (var nspInfo in nspInfos)
                {
                    ExtractNsp(updatePartitionDirectory, nspInfo);
                }

                CheckSystemUpdateMetaXml(updatePartitionDirectory);
                CreateUpdatePartitionXml(updatePartitionXml, updatePartitionDirectory);

                MakeNsp(outputNsp, updatePartitionDirectory);
            }
        }

        private static void MakeNsp(FileInfo outputNsp, DirectoryInfo updatePartitionDirectory)
        {
            SdkTool.Execute(
                SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "createfs",
                    "-o", outputNsp.FullName,
                    "--format", "partitionfs",
                    updatePartitionDirectory.FullName
                });
        }

        private static void MakeSystemUpdateMetaNsp(FileInfo systemUpdateMetaNsp, FileInfo systemUpdateMetaXml, FileInfo keyConfig)
        {
            SdkTool.Execute(
                SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "creatensp",
                    "-o", systemUpdateMetaNsp.FullName,
                    "--type", "SystemUpdate",
                    "--meta", systemUpdateMetaXml.FullName,
                    "--keyconfig", keyConfig.FullName
                });
        }

        private static void MakeSystemUpdateMeta(FileInfo systemUpdateMetaXml, List<FileInfo> nspInfos, string id, string version)
        {
            var arguments = (
                new string[] {
                    "-o", systemUpdateMetaXml.FullName,
                    "--system-meta-id", id,
                    "--system-meta-version", version,
                }).Concat(
                (from nspInfo in nspInfos select new string[] { "-i", nspInfo.FullName }).SelectMany(x => x)).ToArray();

            SdkTool.Execute("MakeSystemUpdateMeta.exe", arguments);
        }

        private static void MakeSystemUpdateMeta(FileInfo systemUpdateMetaXml, DirectoryInfo nspDirInfo, string id, string version)
        {
            var arguments = (
                new string[] {
                    "-o", systemUpdateMetaXml.FullName,
                    "--system-meta-id", id,
                    "--system-meta-version", version,
                    "--input-directory", nspDirInfo.FullName
                });

            SdkTool.Execute("MakeSystemUpdateMeta.exe", arguments);
        }

        private static void CheckSystemUpdateMetaXml(DirectoryInfo updatePartitionDirectory)
        {
            int systemMetaCnt = 0;
            foreach (var file in updatePartitionDirectory.EnumerateFiles())
            {
                if (IsSystemUpdateMetaXml(file))
                {
                    if (++systemMetaCnt > 1)
                    {
                        throw new Exception("System meta contained more than once");
                    }
                }
            }
        }

        private static bool IsSystemUpdateMetaXml(FileInfo file)
        {
            var isSystemMeta = false;
            if (file.Extension == ".xml")
            {
                var xml = new XmlDocument();
                xml.Load(file.FullName);
                var node = xml.SelectSingleNode("ContentMeta/Type");
                isSystemMeta = node != null ? node.InnerText == "SystemUpdate" : false;
            }
            return isSystemMeta;
        }

        private static void ExtractNsp(DirectoryInfo updatePartitionDirectory, FileInfo nspInfo)
        {
            SdkTool.Execute(
                SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "extractnsp",
                    "-o", updatePartitionDirectory.FullName,
                    nspInfo.FullName
                });
        }
    }
}
