﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text.RegularExpressions;

namespace Nintendo.Log
{
    static class SgrCommandProcessor
    {
        private static void InterpretCommand(
            IEnumerable<int> commands, ref Color foreColor, ref Color backColor, ref FontStyle fontStyle)
        {
            foreach (var command in commands)
            {
                if (command == 0)
                {
                    foreColor = DefaultForeColor;
                    backColor = DefaultBackColor;
                    fontStyle = FontStyle.Regular;
                }
                else if (FontStyleTable.ContainsKey(command))
                {
                    fontStyle |= FontStyleTable[command];
                }
                else if (command == 7)
                {
                    var temporary = foreColor;
                    foreColor = backColor;
                    backColor = temporary;
                }
                else if (30 <= command && command <= 37)
                {
                    foreColor = AnsiColorTable[command - 30];
                }
                else if (command == 38)
                {
                    // 前景色の 8/24 bit カラー指定。（未対応）
                }
                else if (command == 39)
                {
                    foreColor = DefaultForeColor;
                }
                else if (40 <= command && command <= 47)
                {
                    backColor = AnsiColorTable[command - 40];
                }
                else if (command == 48)
                {
                    // 背景色の 8/24 bit カラー指定。（未対応）
                }
                else if (command == 49)
                {
                    backColor = DefaultBackColor;
                }
                else if (90 <= command && command <= 97)
                {
                    foreColor = AnsiiAccentColorTable[command - 90];
                }
                else if (100 <= command && command <= 107)
                {
                    backColor = AnsiiAccentColorTable[command - 100];
                }
            }
        }

        public static Tuple<string, Queue<Tuple<int, Color, Color, FontStyle>>> Interpret(
            string input, Color foreColor, Color backColor, FontStyle fontStyle, int offset = 0)
        {
            var commandQueue = new Queue<Tuple<int, Color, Color, FontStyle>>();

            var minus = 0;
            input = SgrCommandPattern.Replace(input, match =>
            {
                var commandList =
                    (from commandCode in match.Groups["CommandList"].Value.Split(';')
                     select string.IsNullOrEmpty(commandCode)
                         ? 0 // 空のコマンドコードはリセットコマンドとして解釈する。
                         : int.Parse(commandCode));

                InterpretCommand(commandList, ref foreColor, ref backColor, ref fontStyle);

                commandQueue.Enqueue(
                    new Tuple<int, Color, Color, FontStyle>(
                        match.Index + offset + minus, foreColor, backColor, fontStyle));
                minus -= match.Length;

                return string.Empty;
            });

            return new Tuple<string, Queue<Tuple<int, Color, Color, FontStyle>>>(input, commandQueue);
        }

        public static string StripSgrCommand(string input)
        {
            return SgrCommandPattern.Replace(input, string.Empty);
        }

        public static int CalculateDisplayLength(string input)
        {
            return StripSgrCommand(input).Length;
        }

        public static Color DefaultForeColor = Color.Black;
        public static Color DefaultBackColor = Color.White;

        private static readonly Regex SgrCommandPattern = new Regex(@"\u001b\[(?<CommandList>\d+(;\d+)*)?m", RegexOptions.Compiled);
        private static readonly Dictionary<int, Color> AnsiColorTable = new Dictionary<int, Color>()
        {
            { 0, Color.Black },
            { 1, Color.Crimson },
            { 2, Color.LimeGreen },
            { 3, Color.Goldenrod },
            { 4, Color.RoyalBlue },
            { 5, Color.DarkMagenta },
            { 6, Color.DarkCyan },
            { 7, Color.LightGray },
        };
        private static readonly Dictionary<int, Color> AnsiiAccentColorTable = new Dictionary<int, Color>()
        {
            { 0, Color.DarkGray },
            { 1, Color.Red },
            { 2, Color.Lime },
            { 3, Color.Gold },
            { 4, Color.CornflowerBlue },
            { 5, Color.DarkViolet },
            { 6, Color.MediumTurquoise },
            { 7, Color.WhiteSmoke },
        };
        private static readonly Dictionary<int, FontStyle> FontStyleTable = new Dictionary<int, FontStyle>()
        {
            { 1, FontStyle.Bold },
            { 3, FontStyle.Italic },
            { 4, FontStyle.Underline },
            { 9, FontStyle.Strikeout }
        };
    }
}
