﻿using System;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace Nintendo.Log
{
    class RichTextBoxEx : RichTextBox
    {
        private const int WM_GETTEXTLENGTH = 0x000E;
        private const int EM_SETSEL = 0x00B1;
        private const int EM_REPLACESEL = 0x00C2;
        private const int WM_SETREDRAW = 0x000B;
        private const int WM_USER = 0x400;
        private const int EM_GETEVENTMASK = WM_USER + 59;
        private const int EM_SETEVENTMASK = WM_USER + 69;
        private const int EM_GETSCROLLPOS = WM_USER + 221;
        private const int EM_SETSCROLLPOS = WM_USER + 222;

        private const int WM_PAINT = 0x000F;

        [DllImport("User32.dll")]
        private static extern int SendMessage(IntPtr hWnd, int Msg, int wParam, string lParam);

        [DllImport("user32.dll")]
        private static extern IntPtr SendMessage(IntPtr hWnd, int Msg, int wParam, ref Point lParam);

        [DllImport("User32.dll")]
        private static extern int SendMessage(IntPtr hWnd, int Msg, int wParam, int lParam);

        [DllImport("user32.dll")]
        private static extern IntPtr SendMessage(IntPtr hWnd, int Msg, int wParam, IntPtr lParam);

        public void FastAppendText(string text)
        {
            var textLength = SendMessage(this.Handle, WM_GETTEXTLENGTH, 0, 0);
            SendMessage(this.Handle, EM_SETSEL, textLength, textLength);
            SendMessage(this.Handle, EM_REPLACESEL, 0, text);
        }

        public void FastRemoveText(int startIndex, int length)
        {
            var textLength = SendMessage(this.Handle, WM_GETTEXTLENGTH, 0, 0);
            if (startIndex >= textLength)
            {
                return;
            }
            var removeTextLength = Math.Min(length, textLength - startIndex);
            if (removeTextLength <= 0)
            {
                return;
            }
            SendMessage(this.Handle, EM_SETSEL, startIndex, startIndex + removeTextLength);
            SendMessage(this.Handle, EM_REPLACESEL, 0, string.Empty);
        }

        private struct SuspendContext
        {
            public bool isSuspending;
            public int selectionStart;
            public int selectionLength;
            public Point scrollPosition;
            public IntPtr eventMask;
        };
        private SuspendContext suspendContext;

        public void SuspendScroll()
        {
            if (!suspendContext.isSuspending)
            {
                suspendContext.isSuspending = true;
                suspendContext.selectionStart = this.SelectionStart;
                suspendContext.selectionLength = this.SelectionLength;
                suspendContext.eventMask = SendMessage(this.Handle, EM_GETEVENTMASK, 0, IntPtr.Zero);
                SendMessage(this.Handle, EM_GETSCROLLPOS, 0, ref suspendContext.scrollPosition);
                SendMessage(this.Handle, WM_SETREDRAW, 0, IntPtr.Zero);
            }
        }

        public void ResumeScroll()
        {
            if (suspendContext.isSuspending)
            {
                this.Select(suspendContext.selectionStart, suspendContext.selectionLength);
                SendMessage(this.Handle, EM_SETSCROLLPOS, 0, ref suspendContext.scrollPosition);
                SendMessage(this.Handle, EM_SETEVENTMASK, 0, suspendContext.eventMask);
                SendMessage(this.Handle, WM_SETREDRAW, 1, IntPtr.Zero);
                suspendContext.isSuspending = false;
                this.Invalidate();
            }
        }
    }
}
