﻿using System;

namespace Nintendo.Log
{
    internal class FilterKeyNotFoundObject
    {
    }

    public struct LogPacketHeader
    {
        public static LogPacketHeader Parse(byte[] buffer)
        {
            var header = new LogPacketHeader();

            header.PeerName = "";
            header.ProcessId = BitConverter.ToUInt64(buffer, ProcessIdByteIndex);
            header.ThreadId = BitConverter.ToUInt64(buffer, ThreadIdByteIndex);
            header.Severity = buffer[SeverityByteIndex];
            header.Verbosity = buffer[VerbosityByteIndex];
            header.PayloadSize = BitConverter.ToInt32(buffer, PayloadSizeByteIndex);

            var flags = buffer[FlagsByteIndex];
            header.IsHead = (flags & (1 << IsHeadBitOffset)) != 0;
            header.IsTail = (flags & (1 << IsTailBitOffset)) != 0;
            var isLittleEndian = (flags & (1 << IsLittleEndianBitOffset)) != 0;

            if (BitConverter.IsLittleEndian != isLittleEndian)
            {
                Func<byte[], byte[]> Reverse = (x) => { Array.Reverse(x); return x; };
                header.ProcessId = BitConverter.ToUInt64(Reverse(BitConverter.GetBytes(header.ProcessId)), 0);
                header.ThreadId = BitConverter.ToUInt64(Reverse(BitConverter.GetBytes(header.ProcessId)), 0);
                header.PayloadSize = BitConverter.ToInt32(Reverse(BitConverter.GetBytes(header.PayloadSize)), 0);
            }
            return header;
        }

        public const int HeaderSize = 24;

        private const int ProcessIdByteIndex = 0;
        private const int ThreadIdByteIndex = 8;
        private const int FlagsByteIndex = 16;
        private const int SeverityByteIndex = 18;
        private const int VerbosityByteIndex = 19;
        private const int PayloadSizeByteIndex = 20;

        private const int IsHeadBitOffset = 0;
        private const int IsTailBitOffset = 1;
        private const int IsLittleEndianBitOffset = 2;

        public string PeerName;
        public ulong ProcessId;
        public ulong ThreadId;
        public bool IsHead;
        public bool IsTail;
        public int Severity;
        public int Verbosity;
        public int PayloadSize;
    }

    public struct LogPacket
    {
        public LogPacketHeader Header;
        public byte[] Payload;
    }

    public class LogRecordHeader
    {
        public string PeerName { get; set; }
        public ulong ProcessId { get; set; }
        public ulong ThreadId { get; set; }
        public int Severity { get; set; }
        public int Verbosity { get; set; }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (LogRecordHeader)obj;
            return value.PeerName.Equals(PeerName) &&
                value.ProcessId.Equals(ProcessId) &&
                value.ThreadId.Equals(ThreadId) &&
                value.Severity.Equals(Severity) &&
                value.Verbosity.Equals(Verbosity);
        }

        public override int GetHashCode()
        {
            return (PeerName != null ? PeerName.GetHashCode() : 0) ^
                ProcessId.GetHashCode() ^
                ThreadId.GetHashCode() ^
                Severity.GetHashCode() ^
                Verbosity.GetHashCode();
        }
    }

    internal struct LogRecord
    {
        public LogRecordHeader Header;
        public byte[] Payload;
    }

    internal struct LogManagerBinaryHeader
    {
        public static LogManagerBinaryHeader Parse(byte[] buffer)
        {
            var header = new LogManagerBinaryHeader();

            header.MagicNumber = BitConverter.ToUInt32(buffer, MagicNumberdByteIndex);
            header.Version = buffer[VersionByteIndex];

            return header;
        }

        public const int HeaderSize = 8;

        public const uint SupposedMagicNumber = 0x70687068;

        private const int MagicNumberdByteIndex = 0;
        private const int VersionByteIndex = 4;

        public UInt32 MagicNumber;
        public byte Version;
        // public unsafe byte Reserved[3]; // unsafe モードが必要。使わないのでコメントアウトしておく
    }
}
