﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Xml.Linq;

namespace Nintendo.Htcs
{
    public static class HtcsPeerName
    {
        public static readonly string Any = string.Empty;
        public static readonly string DefalutHost = string.Empty;
    }

    public class HtcsInfo
    {
        public HtcsInfo(List<Target> targetList, List<PortMapItem> portMap)
        {
            TargetList = targetList;
            PortMap = portMap;
        }

        public HtcsInfo(XElement element) :
            this(ParseTargetList(element.Element("TargetList")),
                ParsePortMap(element.Element("PortMap")))
        {
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (HtcsInfo)obj;
            return value.TargetList.SequenceEqual(TargetList) &&
                value.PortMap.SequenceEqual(PortMap);
        }

        private static int GetHashCode<T>(IEnumerable<T> enumerable)
        {
            const int fnvOffsetBasis = -2128831035;
            const int fnvPrime = 16777619;

            var hash = fnvOffsetBasis;
            foreach (var item in enumerable)
            {
                hash = (fnvPrime * hash) ^ item.GetHashCode();
            }
            return hash;
        }

        public override int GetHashCode()
        {
            return ((TargetList != null && TargetList.Count > 0) ? GetHashCode(TargetList) : 0) ^
                ((PortMap != null && PortMap.Count > 0) ? GetHashCode(PortMap) : 0);
        }

        public List<Target> TargetList { get; protected set; }
        public List<PortMapItem> PortMap { get; protected set; }

        private static List<Target> ParseTargetList(XElement element)
        {
            return element.Elements("Target").Select(x => new Target(x)).ToList();
        }

        private static List<PortMapItem> ParsePortMap(XElement element)
        {
            return element.Elements("PortMapItem").Select(x => new PortMapItem(x)).ToList();
        }
    }

    public class Target
    {
        public Target(string peerType, string peerName, string id)
        {
            PeerType = peerType;
            HtcsPeerName = peerName;
            Id = id;
        }

        public Target(XElement element) :
            this(element.Element("PeerType").Value,
                element.Element("HtcsPeerName").Value,
                element.Element("Id").Value)
        {
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (Target)obj;
            return PeerType.Equals(value.PeerType) &&
                HtcsPeerName.Equals(value.HtcsPeerName) &&
                Id.Equals(value.Id);
        }

        public override int GetHashCode()
        {
            return (PeerType != null ? PeerType.GetHashCode() : 0) ^
                (HtcsPeerName != null ? HtcsPeerName.GetHashCode() : 0) ^
                (Id != null ? Id.GetHashCode() : 0);
        }

        public string PeerType { get; set; }
        public string HtcsPeerName { get; set; }
        public string Id { get; set; }
    }

    public class PortMapItem
    {
        public PortMapItem(HtcsEndPoint htcsEndPoint, IPEndPoint ipEndPoint)
        {
            HtcsEndPoint = htcsEndPoint;
            IPEndPoint = ipEndPoint;
        }

        public PortMapItem(XElement element) :
            this(new HtcsEndPoint(
                    element.Element("HtcsPeerName").Value,
                    element.Element("HtcsPortName").Value),
                new IPEndPoint(
                    IPAddress.Parse(element.Element("IPAddress").Value),
                    int.Parse(element.Element("TcpPortNumber").Value)))
        {
        }

        public XElement ToXElement()
        {
            return
                new XElement("PortMapItem",
                    new XElement("HtcsPeerName", HtcsEndPoint.HtcsPeerName),
                    new XElement("HtcsPortName", HtcsEndPoint.HtcsPortName),
                    new XElement("IPAddress", IPEndPoint.Address),
                    new XElement("TcpPortNumber", IPEndPoint.Port));
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (PortMapItem)obj;
            return value.HtcsEndPoint.Equals(HtcsEndPoint) &&
                value.IPEndPoint.Equals(IPEndPoint);
        }

        public override int GetHashCode()
        {
            return (HtcsEndPoint != null ? HtcsEndPoint.GetHashCode() : 0) ^
                (IPEndPoint != null ? IPEndPoint.GetHashCode() : 0);
        }

        public override string ToString()
        {
            return "{" + HtcsEndPoint + "}, {" + IPEndPoint + "}";
        }

        public HtcsEndPoint HtcsEndPoint { get; set; }
        public IPEndPoint IPEndPoint { get; set; }
    }

    public class HtcsEndPoint
    {
        public HtcsEndPoint(string peerName, string portName)
        {
            HtcsPeerName = peerName;
            HtcsPortName = portName;
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }
            var value = (HtcsEndPoint)obj;
            return value.HtcsPeerName.Equals(HtcsPeerName) &&
                value.HtcsPortName.Equals(HtcsPortName);
        }

        public override int GetHashCode()
        {
            return (HtcsPeerName != null ? HtcsPeerName.GetHashCode() : 0) ^
                (HtcsPortName != null ? HtcsPortName.GetHashCode() : 0);
        }

        public override string ToString()
        {
            return HtcsPeerName + ":" + HtcsPortName;
        }

        public string HtcsPeerName { get; set; }
        public string HtcsPortName { get; set; }
    }

    public class CommandResult
    {
        public CommandResult(string requestName, int value)
        {
            RequestName = requestName;
            Value = value;
        }

        public CommandResult(XElement element) :
            this(element.Element("RequestName").Value,
                int.Parse(element.Element("Value").Value))
        {
        }

        public string RequestName { get; set; }
        public int Value { get; set; }
    }

    public class RegisterPortMapCommand : IHtcsCommand
    {
        public RegisterPortMapCommand(PortMapItem portMapItem, string requestName)
        {
            PortMapItem = portMapItem;
            RequestName = requestName;
        }

        public RegisterPortMapCommand(PortMapItem portMapItem) :
            this(portMapItem, Guid.NewGuid().ToString())
        {
        }

        public RegisterPortMapCommand(XElement element) :
            this(new PortMapItem(element),
                element.Element("RequestName").Value)
        {
        }

        public XElement ToXElement()
        {
            return new XElement("RegisterPortMapCommand",
                PortMapItem.ToXElement(),
                new XElement("RequestName", RequestName));
        }

        public PortMapItem PortMapItem { get; set; }
        public string RequestName { get; set; }
    }

    public class UnregisterPortMapCommand : IHtcsCommand
    {
        public UnregisterPortMapCommand(HtcsEndPoint endPoint, string requestName)
        {
            HtcsEndPoint = endPoint;
            RequestName = requestName;
        }

        public UnregisterPortMapCommand(HtcsEndPoint endPoint) :
            this(endPoint, Guid.NewGuid().ToString())
        {
        }

        public UnregisterPortMapCommand(XElement element) :
            this(new HtcsEndPoint(
                    element.Element("HtcsPeerName").Value,
                    element.Element("HtcsPeerName").Value),
                element.Element("RequestName").Value)
        {
        }

        public XElement ToXElement()
        {
            return new XElement("UnregisterPortMapCommand",
                new XElement("HtcsPeerName", HtcsEndPoint.HtcsPeerName),
                new XElement("HtcsPortName", HtcsEndPoint.HtcsPortName),
                new XElement("RequestName", RequestName));
        }

        public HtcsEndPoint HtcsEndPoint { get; set; }
        public string RequestName { get; set; }
    }

    public interface IHtcsCommand
    {
        string RequestName { get; set; }
        XElement ToXElement();
    }
}
