﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks.Dataflow;
using System.Xml.Linq;

namespace Nintendo.Htcs
{
    internal sealed class HtcsProxyImpl : IDisposable
    {
        private static HtcsProxyImpl instance = new HtcsProxyImpl();

        public static HtcsProxyImpl  Instance { get { return instance; } }

        public event Action OnTargetManagerConnect;
        public event Action OnTargetManagerDisconnect;
        public event Action<List<Target>> OnTargetListUpdate;
        public event Action<List<PortMapItem>> OnPortMapUpdate;

        private HtcsProxyImpl()
        {
            HtcsInfo = new HtcsInfo(new List<Target>(), new List<PortMapItem>());
            AckTable = new Dictionary<string, TargetManagerAckEvent>();
            ReceiveBuffer = new BufferBlock<string>();
            CancellationTokenSource = new CancellationTokenSource();
        }

        public void Dispose()
        {
            Stop();
        }

        public void Start()
        {
            lock (StartedLock)
            {
                Debug.Assert(!Started);
                ConnectionThread = new Thread(ConnectFunc);
                ConnectionThread.Start();
                ProcessThread = new Thread(ProcessFunc);
                ProcessThread.Start();
                Started = true;
            }
        }

        public void Stop()
        {
            lock (StartedLock)
            {
                Debug.Assert(Started);
                CancellationTokenSource.Cancel();
                ReceiveBuffer.Complete();
                ReceiveBuffer.Completion.Wait();
                ConnectionThread.Join();
                ProcessThread.Join();
            }
        }

        public bool Connected { get { return (TcpClient != null ? TcpClient.Connected : false); } }

        public int RegisterPortMap(PortMapItem item)
        {
            AssertStarted();
            return SendCommand(new RegisterPortMapCommand(item));
        }

        public int UnregisterPortMap(HtcsEndPoint endPoint)
        {
            AssertStarted();
            return SendCommand(new UnregisterPortMapCommand(endPoint));
        }

        public IPEndPoint QueryIPEndPoint(HtcsEndPoint endPoint)
        {
            AssertStarted();
            return
                (from portMapItem in HtcsInfo.PortMap
                where portMapItem.HtcsEndPoint.Equals(endPoint)
                select portMapItem.IPEndPoint).FirstOrDefault();
        }

        public IPEndPoint QueryIPEndPoint(string PortName)
        {
            AssertStarted();
            return
                (from portMapItem in HtcsInfo.PortMap
                 where portMapItem.HtcsEndPoint.HtcsPortName.Equals(PortName)
                 select portMapItem.IPEndPoint).FirstOrDefault();
        }

        public HtcsEndPoint QueryHtcsEndPoint(IPEndPoint endPoint)
        {
            AssertStarted();
            return
                (from portMapItem in HtcsInfo.PortMap
                where portMapItem.IPEndPoint.Equals(endPoint)
                select portMapItem.HtcsEndPoint).FirstOrDefault();
        }

        private void AssertStarted()
        {
            lock (StartedLock)
            {
                if (!Started)
                {
                    throw new InvalidOperationException(
                        "A operation was isssued to HTCS proxy before it is not started.");
                }
            }
        }

        private void ConnectFunc()
        {
            while (!CancellationTokenSource.IsCancellationRequested)
            {
                while (!TryConnect())
                {
                    if (CancellationTokenSource.IsCancellationRequested)
                    {
                        if (TcpClient != null)
                        {
                            TcpClient.Close();
                            TcpClient = null;
                        }
                        return;
                    }
                    Thread.Sleep(1000);
                }

                ReceiveFunc();
            }
        }

        private bool TryConnect()
        {
            if (TcpClient == null)
            {
                TcpClient = new TcpClient();
            }
            else if (TcpClient.Connected)
            {
                return true;
            }

            try
            {
                TcpClient.Connect(IPAddress.Loopback, ControlPort);
            }
            catch (SocketException)
            {
                // Target Manager が起動していない
            }

            if (TcpClient.Connected)
            {
                Trace.TraceInformation("Connected to HTCS control port: {0}", TcpClient.Client.RemoteEndPoint);

                RequestSystemPortMapping();

                OnTargetManagerConnect?.Invoke();
            }

            return TcpClient.Connected;
        }

        private void ReceiveFunc()
        {
            try
            {
                var sb = new StringBuilder();
                var buffer = new byte[4096];
                using (var bs = new BufferedStream(TcpClient.GetStream()))
                using (CancellationTokenSource.Token.Register(() => bs.Close()))
                while (true)
                {
                    try
                    {
                        var readCount = bs.Read(buffer, 0, buffer.Length);
                        if (readCount <= 0)
                        {
                            break; // Target Manager との切断。
                        }
                        sb.Append(Encoding.UTF8.GetString(buffer, 0, readCount));
                    }
                    catch (Exception exception) when (
                        exception is IOException ||
                        exception is ObjectDisposedException)
                    {
                        if (CancellationTokenSource.IsCancellationRequested)
                        {
                            return; // Stop() の呼び出し。
                        }
                        else
                        {
                            break; // Target Manager との切断。
                        }
                    }

                    var index = sb.ToString().LastIndexOf('\n');
                    if (index >= 0)
                    {
                        foreach (var line in sb.ToString().Substring(0, index).Split('\n'))
                        {
                            //Trace.TraceInformation(
                            //    XElement.Parse(line).ToString());
                            ReceiveBuffer.Post(line);
                        }
                        sb.Remove(0, index + 1);
                    }
                }
            }
            finally
            {
                Debug.Assert(TcpClient != null);
                Trace.TraceInformation("Disconnected to HTCS control port");
                TcpClient.Close();
                TcpClient = null;
            }
            OnTargetManagerDisconnect?.Invoke();
        }

        private void ProcessFunc()
        {
            while (true)
            {
                var line = string.Empty;
                try
                {
                    line = ReceiveBuffer.Receive(CancellationTokenSource.Token);
                }
                catch (OperationCanceledException)
                {
                    break;
                }

                var element = XElement.Parse(line);
                if (element.Name == "HtcsInfo")
                {
                    UpdateHtcsInfo(new HtcsInfo(element));
                }
                else
                {
                    Debug.Assert(element.Name == "CommandResult");
                    NotifyAck(new CommandResult(element));
                }
            }
        }

        private int SendCommand(IHtcsCommand command)
        {
            var ackEvent = new TargetManagerAckEvent();
            lock (AckTable)
            {
                AckTable[command.RequestName] = ackEvent;
            }
            if (!TryConnect()) // ポーリングの隙間で実行されたときのために接続を試行する
            {
                throw new TargetManagerNotFoundException();
            }
            try
            {
                using (var sw = new StreamWriter(TcpClient.GetStream(), new UTF8Encoding(false), 1024, true))
                {
                    sw.WriteLine(command.ToXElement().ToString(SaveOptions.DisableFormatting));
                }
            }
            catch (InvalidOperationException) // Target Manager との接続が切れている場合は不正な操作になる
            {
                throw new TargetManagerNotFoundException();
            }
            ackEvent.Wait();
            return ackEvent.CommandResult.Value;
        }

        private void UpdateHtcsInfo(HtcsInfo htcsInfo)
        {
            lock (HtcsInfo)
            {
                if (OnTargetListUpdate != null &&
                    !HtcsInfo.TargetList.SequenceEqual(htcsInfo.TargetList))
                {
                    OnTargetListUpdate.Invoke(htcsInfo.TargetList);
                }
                if (OnPortMapUpdate != null &&
                    !HtcsInfo.PortMap.SequenceEqual(htcsInfo.PortMap))
                {
                    OnPortMapUpdate.Invoke(htcsInfo.PortMap);
                }
                HtcsInfo = htcsInfo;
            }
        }

        private void NotifyAck(CommandResult commandResult)
        {
            TargetManagerAckEvent ackEvent;
            lock (AckTable)
            {
                ackEvent = AckTable[commandResult.RequestName];
                AckTable.Remove(commandResult.RequestName);
            }
            ackEvent.CommandResult = commandResult;
            ackEvent.Notify();
        }

        private void RequestSystemPortMapping()
        {
            try
            {
                using (var sw = new StreamWriter(TcpClient.GetStream(), new UTF8Encoding(false), 1024, true))
                {
                    sw.WriteLine(new XElement("RequestSystemPortMapping").ToString(SaveOptions.DisableFormatting));
                }
            }
            catch (Exception exception) when (
                exception is InvalidOperationException || // Target Manager との接続が切れている場合は不正な操作になる
                exception is IOException)
            {
                throw new TargetManagerNotFoundException();
            }
        }

        internal class TargetManagerAckEvent
        {
            private AutoResetEvent AckEvent = new AutoResetEvent(false);
            public CommandResult CommandResult { get; set; }

            public void Wait()
            {
                // TORIAEZU: 接続台数が増えたときの TM の応答時間が読めないため、タイムアウト時間を設定していない
                if (!AckEvent.WaitOne())
                {
                    throw new CommandTimeoutException();
                }
            }

            public void Notify()
            {
                AckEvent.Set();
            }
        }

        private object StartedLock = new object();
        private bool Started = false;
        private const int ControlPort = 8003;
        private TcpClient TcpClient;
        public HtcsInfo HtcsInfo { get; private set; }
        private Dictionary<string, TargetManagerAckEvent> AckTable;
        private Thread ConnectionThread;
        private Thread ProcessThread;
        private BufferBlock<string> ReceiveBuffer;
        private CancellationTokenSource CancellationTokenSource;
    }
}
