﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;
using System.IO;

namespace Nintendo.HtcTools.Htclow
{
    internal sealed class Worker : IDisposable
    {
        // 下位通信路の切断を示す CancellationToken
        private CancellationTokenSource m_ConnectionCancel;

        private Mux m_Mux;
        private Stream m_Stream;
        private BinaryReader m_Reader;
        private BinaryWriter m_Writer;

        private Task m_ReceiveTask;
        private Task m_SendTask;

        public Worker(CancellationTokenSource connectionCancel, Mux mux, Stream stream)
        {
            m_ConnectionCancel = connectionCancel;

            m_Mux = mux;
            m_Stream = stream;
            m_Reader = new BinaryReader(m_Stream);
            m_Writer = new BinaryWriter(m_Stream);

            m_ReceiveTask = new Task(ReceiveTaskBody, TaskCreationOptions.LongRunning);
            m_SendTask = new Task(SendTaskBody, TaskCreationOptions.LongRunning);

            m_ReceiveTask.Start();
            m_SendTask.Start();
        }

        private void ReceiveTaskBody()
        {
            try
            {
                while (true)
                {
                    // ヘッダ受信
                    var buffer = m_Reader.ReadBytes(Packet.HeaderSize);
                    if (buffer.Length < Packet.HeaderSize)
                    {
                        throw new HtclowException("Failed to receive header.");
                    }

                    var header = new Packet(buffer);

                    // 通信続行不可能なヘッダのエラーを検査
                    if (header.Protocol != Packet.HtclowProtocol)
                    {
                        throw new HtclowException($"Received packet protocol is unsupported. ({header.Protocol})");
                    }

                    if (header.PacketType != PacketType.Syn)
                    {
                        if (header.Version > PacketFactory.MaxVersion)
                        {
                            throw new HtclowException($"Received packet version is unsupported. ({header.PacketType}, {header.Version})");
                        }
                    }

                    if (header.PacketType == PacketType.Data)
                    {
                        if (header.BodySize < 0 || header.BodySize > Packet.MaxBodySize)
                        {
                            throw new HtclowException($"Received packet body size is invalid. ({header.PacketType}, {header.BodySize}");
                        }
                    }
                    else
                    {
                        if (header.BodySize != 0)
                        {
                            throw new HtclowException($"Received packet body size is invalid. ({header.PacketType}, {header.BodySize}");
                        }
                    }

                    var packet = new Packet(header, header.BodySize);

                    // ボディ受信
                    if (header.BodySize != 0)
                    {
                        var body = m_Reader.ReadBytes(header.BodySize);
                        packet.SetBody(body);
                    }

                    Log.ReceivePacket(packet);

                    m_Mux.ProcessReceive(packet);
                }
            }
            catch (ObjectDisposedException e)
            {
                Log.Exception(e);
            }
            catch (IOException e)
            {
                Log.Exception(e);
            }
            catch (HtclowException e)
            {
                Log.Exception(e);
            }
            finally
            {
                Log.Info("Receive task finished.");
                m_ConnectionCancel.Cancel();
                m_Reader.Dispose();
                m_Stream.Dispose();
            }
        }

        private void SendTaskBody()
        {
            try
            {
                while (true)
                {
                    var packet = m_Mux.QuerySendPacket();

                    Log.SendPacket(packet);
                    m_Writer.Write(packet.GetBytes());
                }
            }
            catch (ObjectDisposedException e)
            {
                Log.Exception(e);
            }
            catch (OperationCanceledException e)
            {
                Log.Exception(e);
            }
            catch (IOException e)
            {
                Log.Exception(e);
            }
            catch (HtclowException e)
            {
                Log.Exception(e);
            }
            finally
            {
                Log.Info("Send task finished.");
                m_ConnectionCancel.Cancel();
                m_Writer.Dispose();
                m_Stream.Dispose();
            }
        }

        public void Dispose()
        {
            m_ConnectionCancel.Cancel();

            m_Writer.Dispose();
            m_Reader.Dispose();
            m_Stream.Dispose();

            m_SendTask.Wait();
            m_ReceiveTask.Wait();

            m_SendTask.Dispose();
            m_ReceiveTask.Dispose();
        }
    }
}
