﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

using System.Diagnostics;

using System.IO;

namespace Nintendo.HtcTools.Htclow
{
    /// <summary>
    /// チャネルごとの送信バッファです。
    /// パケットのシーケンス番号はこのクラス内で割り振られて管理されます。
    /// </summary>
    internal class SendBuffer : IDisposable
    {
        // スライディングウィンドウサイズ
        internal const int MaxSendingPacketCount = 32;

        private class PacketInfo
        {
            public Packet Packet { get; set; }
            public bool SendFlag { get; set; }
            public DateTime SendTime { get; set; }
        }

        // 送信データのパケットのリスト
        // ack が届いて不要になったらパケットが削除される
        private List<Packet> m_AckPackets = new List<Packet>();

        // ack パケットのリスト
        // 送信したら即座に削除される
        private List<PacketInfo> m_Packets = new List<PacketInfo>();

        // 追加された送信データのパケットに付与するシーケンス番号
        private long m_NextSequenceId = 1;

        // 送信バッファが空になったことを示すイベント
        private ManualResetEventSlim m_EmptyEvent = new ManualResetEventSlim(false);
        public ManualResetEventSlim EmptyEvent
        {
            get
            {
                return m_EmptyEvent;
            }
        }

        public bool RetransmitEnable { get; set; }

        public void AddPacket(Packet packet)
        {
            if (packet.PacketType != PacketType.Syn &&
                packet.PacketType != PacketType.Data &&
                packet.PacketType != PacketType.Fin)
            {
                throw new ArgumentException();
            }

            if (packet.PacketType == PacketType.Syn && m_NextSequenceId != 0)
            {
                throw new ArgumentException();
            }

            packet.SequenceId = m_NextSequenceId;
            m_NextSequenceId++;

            m_EmptyEvent.Reset();

            var info = new PacketInfo()
            {
                Packet = packet,
                SendFlag = false,
                SendTime = DateTime.MinValue,
            };

            m_Packets.Add(info);
        }

        public void AddAckPacket(Packet packet)
        {
            if (packet.PacketType != PacketType.SynAck &&
                packet.PacketType != PacketType.DataAck &&
                packet.PacketType != PacketType.FinAck)
            {
                throw new ArgumentException();
            }

            m_EmptyEvent.Reset();

            m_AckPackets.Add(packet);
        }

        public void RemovePacket(long sequenceId)
        {
            lock (m_Packets)
            {
                m_Packets.RemoveAll(info => info.Packet.SequenceId <= sequenceId);

                if (m_Packets.Count == 0 && m_AckPackets.Count == 0)
                {
                    m_EmptyEvent.Set();
                }
            }
        }

        public void RemoveAckPacket(long sequenceId)
        {
            lock (m_AckPackets)
            {
                m_AckPackets.RemoveAll(packet => packet.SequenceId <= sequenceId);

                if (m_Packets.Count == 0 && m_AckPackets.Count == 0)
                {
                    m_EmptyEvent.Set();
                }
            }
        }

        public Packet GetNextPacket()
        {
            // 送るべき ack があれば、まずそれを送る
            if (m_AckPackets.Count != 0)
            {
                var packet = m_AckPackets.First();

                // ack は取り出した際に削除
                RemoveAckPacket(packet.SequenceId);

                return packet;
            }

            // ack 以外のパケットのリストが空なら、以後の処理はしない
            if (m_Packets.Count == 0)
            {
                return null;
            }

            // リストの先頭のパケットの再送時刻になったら、すべてのパケットの送信履歴を破棄
            UpdateForRetransmit();

            // 未送信パケットのうち最初のものを取得
            var packetInfoToSend = m_Packets.Find(info => !info.SendFlag);
            if (packetInfoToSend == null)
            {
                return null;
            }

            // 送信中のパケット数が既定値を超えてしまうなら送信しない
            var sendingPacketCount = m_Packets.Sum(info => info.SendFlag ? 1 : 0);
            if (sendingPacketCount > MaxSendingPacketCount)
            {
                return null;
            }

            // パケットの送信が決定
            packetInfoToSend.SendFlag = true;
            packetInfoToSend.SendTime = DateTime.Now;

            return packetInfoToSend.Packet;
        }

        private void UpdateForRetransmit()
        {
            if (!RetransmitEnable)
            {
                return;
            }

            var head = m_Packets.First();
            if (head.SendFlag && (DateTime.Now - head.SendTime) > TimeSpan.FromMilliseconds(Packet.RetransmitTimeoutMs))
            {
                m_Packets.ForEach(info =>
                {
                    info.SendFlag = false;
                    info.SendTime = DateTime.MinValue;
                });
            }
        }

        #region Dispose
        public void Dispose()
        {
            m_EmptyEvent.Dispose();
        }
        #endregion
    }
}
