﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;

namespace Nintendo.HtcTools.Htclow
{
    internal enum PacketType : ushort
    {
        Data = 0x0000,
        DataAck = 0x0001,
        Syn = 0x0002,
        SynAck = 0x0003,
        Fin = 0x0004,
        FinAck = 0x0005,
    }

    internal class Packet
    {
        internal const ushort HtclowProtocol = 256; // htclow protocol
        internal const int MaxBodySize = 65536;

        // パケット再送信タイムアウト
        internal const int RetransmitTimeoutMs = 100;

        internal const int HeaderSize = 24;

        private const int OffsetProtocol = 0;
        private const int OffsetVersion = 2;
        private const int OffsetReserved = 4;
        private const int OffsetPacketType = 6;
        private const int OffsetChannelId = 8;
        private const int OffsetModuleId = 11;
        private const int OffsetBodySize = 12;
        private const int OffsetSequenceId = 16;

        private byte[] m_Buffer;

        private short GetInt16(int byteOffset)
        {
            var span = new Span<byte>(m_Buffer);
            var shortSpan = span.Slice(0).NonPortableCast<byte, short>();
            return shortSpan[byteOffset / sizeof(short)];
        }

        private void SetInt16(int byteOffset, short value)
        {
            var span = new Span<byte>(m_Buffer);
            var shortSpan = span.Slice(0).NonPortableCast<byte, short>();
            shortSpan[byteOffset / sizeof(short)] = value;
        }

        private int GetInt32(int byteOffset)
        {
            var span = new Span<byte>(m_Buffer);
            var intSpan = span.Slice(0).NonPortableCast<byte, int>();
            return intSpan[byteOffset / sizeof(int)];
        }

        private void SetInt32(int byteOffset, int value)
        {
            var span = new Span<byte>(m_Buffer);
            var intSpan = span.Slice(0).NonPortableCast<byte, int>();
            intSpan[byteOffset / sizeof(int)] = value;
        }

        private long GetInt64(int byteOffset)
        {
            var span = new Span<byte>(m_Buffer);
            var longSpan = span.Slice(0).NonPortableCast<byte, long>();
            return longSpan[byteOffset / sizeof(long)];
        }

        private void SetInt64(int byteOffset, long value)
        {
            var span = new Span<byte>(m_Buffer);
            var longSpan = span.Slice(0).NonPortableCast<byte, long>();
            longSpan[byteOffset / sizeof(long)] = value;
        }

        public ushort Protocol
        {
            get
            {
                return (ushort)GetInt16(OffsetProtocol);
            }
        }

        public ushort Version
        {
            get
            {
                return (ushort)GetInt16(OffsetVersion);
            }
            set
            {
                SetInt16(OffsetVersion, (short)value);
            }
        }

        public PacketType PacketType
        {
            get
            {
                return (PacketType)GetInt16(OffsetPacketType);
            }
            set
            {
                SetInt16(OffsetPacketType, (short)value);
            }
        }

        public Channel Channel
        {
            get
            {
                var span = new Span<byte>(m_Buffer);
                return new Channel() {
                    ChannelId = (ushort)GetInt16(OffsetChannelId),
                    ModuleId = span[OffsetModuleId],
                };
            }
            set
            {
                var span = new Span<byte>(m_Buffer);
                SetInt16(OffsetChannelId, (short)value.ChannelId);
                span[OffsetModuleId] = value.ModuleId;
            }
        }

        public int BodySize
        {
            get
            {
                return GetInt32(OffsetBodySize);
            }
        }

        public long SequenceId
        {
            get
            {
                return GetInt64(OffsetSequenceId);
            }
            set
            {
                SetInt64(OffsetSequenceId, value);
            }
        }

        public byte[] GetBytes()
        {
            return m_Buffer;
        }

        public byte[] GetBodyBytes()
        {
            var span = new Span<byte>(m_Buffer);
            return span.Slice(HeaderSize).ToArray();
        }

        /// <summary>
        /// ボディが無いパケット (ヘッダ) を作成する。
        /// </summary>
        public Packet()
            : this(0)
        {
        }

        /// <summary>
        /// 指定サイズのボディを持つパケットを作成する。
        /// </summary>
        public Packet(int bodySize)
        {
            if (bodySize < 0)
            {
                throw new ArgumentOutOfRangeException();
            }

            m_Buffer = new byte[HeaderSize + bodySize];

            SetInt16(OffsetProtocol, (short)HtclowProtocol);
            SetInt16(OffsetVersion, 0);
            SetInt16(OffsetReserved, 0);
            SetInt32(OffsetBodySize, bodySize);
        }

        /// <summary>
        /// 与えられたバッファを参照するパケットクラスを作成する。
        /// </summary>
        /// <param name="buffer">バッファ</param>
        public Packet(byte[] buffer)
        {
            if (buffer.Length < HeaderSize)
            {
                throw new ArgumentException("header.m_Buffer が十分な大きさを持っていません");
            }

            m_Buffer = buffer;
        }

        /// <summary>
        /// ヘッダとボディサイズを指定してパケットを作成する。
        /// 作成されたパケットには引数に与えたヘッダがコピーされる。
        /// </summary>
        /// <param name="header">ヘッダ</param>
        /// <param name="bodySize">ボディサイズ</param>
        public Packet(Packet header, int bodySize)
        {
            if (bodySize < 0)
            {
                throw new ArgumentOutOfRangeException();
            }

            if (header.m_Buffer.Length < HeaderSize)
            {
                throw new ArgumentException("header.m_Buffer が十分な大きさを持っていません");
            }

            m_Buffer = new byte[HeaderSize + bodySize];
            System.Buffer.BlockCopy(header.m_Buffer, 0, m_Buffer, 0, HeaderSize);
        }

        /// <summary>
        /// 引数で与えたボディをパケットにコピーする。
        /// </summary>
        /// <param name="body">ボディ</param>
        public void SetBody(byte[] body)
        {
            SetBody(body, 0, body.Length);
        }

        public void SetBody(byte[] body, int bodyOffset, int bodySize)
        {
            if (m_Buffer.Length != HeaderSize + bodySize)
            {
                throw new ArgumentException();
            }

            System.Buffer.BlockCopy(body, bodyOffset, m_Buffer, HeaderSize, bodySize);
        }

        public override bool Equals(object other)
        {
            var otherAsPacket = other as Packet;
            if (otherAsPacket == null)
            {
                return false;
            }

            return Enumerable.SequenceEqual(m_Buffer, otherAsPacket.m_Buffer);
        }

        public override int GetHashCode()
        {
            if (m_Buffer.Length < HeaderSize)
            {
                throw new ArgumentException("m_Buffer が十分な大きさを持っていません");
            }

            var hash = 0;
            for (int i = 0; i < HeaderSize; i++)
            {
                hash ^= m_Buffer[i];
            }
            return hash;
        }

        public override string ToString()
        {
            return $"packetType:{PacketType}, channel:({Channel}), sequenceId:{SequenceId}";
        }
    }
}
