﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;
using System.IO;

namespace Nintendo.HtcTools
{
    public class HtclowClient : IDisposable
    {
        private readonly CancellationTokenSource m_ConnectionCancel;
        private readonly Htclow.Mux m_Mux;
        private readonly Htclow.Worker m_Worker;

        internal bool RetransmitEnable
        {
            set
            {
                m_Mux.RetransmitEnable = value;
            }
        }

        public HtclowClient(Stream stream)
        {
            m_ConnectionCancel = new CancellationTokenSource();
            m_Mux = new Htclow.Mux(m_ConnectionCancel.Token);
            m_Worker = new Htclow.Worker(m_ConnectionCancel, m_Mux, stream);
        }

        /// <summary>
        /// 指定したモジュール ID に対応する HtclowModule オブジェクトを作成します。
        /// </summary>
        /// <param name="moduleId">モジュール ID</param>
        /// <returns>HtclowModule オブジェクト</returns>
        public HtclowModule OpenModule(byte moduleId)
        {
            return new HtclowModule(m_Mux, moduleId);
        }

        /// <summary>
        /// 指定したモジュール ID とチャネル ID に対応する HtclowChannel オブジェクトを作成します。
        /// </summary>
        /// <param name="moduleId">モジュール ID</param>
        /// <param name="channelId">チャンネル ID</param>
        /// <returns>HtclowModule オブジェクト</returns>
        public HtclowChannel OpenChannel(byte moduleId, ushort channelId)
        {
            return new HtclowChannel(m_Mux, moduleId, channelId);
        }

        /// <summary>
        /// 指定したモジュール ID とチャネル ID に対応する HtclowChannelStream オブジェクトを作成します。
        /// </summary>
        /// <param name="moduleId">モジュール ID</param>
        /// <param name="channelId">チャンネル ID</param>
        /// <returns>HtclowModule オブジェクト</returns>
        public HtclowChannelStream OpenChannelStream(byte moduleId, ushort channelId)
        {
            var htclowChannel = new HtclowChannel(m_Mux, moduleId, channelId);
            return new HtclowChannelStream(htclowChannel);
        }

        #region Dispose
        private bool disposedValue = false; // 重複する呼び出しを検出するには

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    m_Worker.Dispose();
                    m_Mux.Dispose();
                    m_ConnectionCancel.Dispose();
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion
    }
}
