﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;

namespace Nintendo.HtcTools
{
    public sealed class HtclowChannel : IDisposable
    {
        private readonly Htclow.Mux m_Mux;
        private readonly Htclow.Channel m_Channel;

        internal HtclowChannel(Htclow.Mux mux, byte moduleId, ushort channelId)
        {
            m_Mux = mux;
            m_Channel = new Htclow.Channel() { ModuleId = moduleId, ChannelId = channelId };

            m_Mux.Open(m_Channel);
        }

        /// <summary>
        /// ターゲットのチャネルと通信を確立します。
        /// この API は、通信が確立するまでブロックします。
        /// </summary>
        public void Connect()
        {
            m_Mux.Connect(m_Channel);
        }

        /// <summary>
        /// ターゲットにデータを送信します。
        /// 呼び出し後に送信バッファの内容を変更しないでください。
        /// </summary>
        /// <param name="buffer">送信バッファ</param>
        public void Send(byte[] buffer)
        {
            m_Mux.Send(m_Channel, buffer, 0, buffer.Length);
        }

        /// <summary>
        /// ターゲットにデータを送信します。
        /// 呼び出し後に送信バッファの内容を変更しないでください。
        /// </summary>
        /// <param name="buffer">送信バッファ</param>
        /// <param name="offset">送信データの buffer におけるオフセット</param>
        /// <param name="count">送信するバイト数</param>
        public void Send(byte[] buffer, int offset, int count)
        {
            m_Mux.Send(m_Channel, buffer, offset, count);
        }

        /// <summary>
        /// ターゲットからデータを受信します。
        /// </summary>
        /// <param name="count">受信するバイト数</param>
        /// <returns>受信データ</returns>
        public byte[] Receive(int count)
        {
            var buffer = new byte[count];
            var receiveCount = m_Mux.Receive(m_Channel, buffer, 0, count);

            if (receiveCount == count)
            {
                return buffer;
            }
            else
            {
                // 当初の予定から API 仕様を変えたため、場合によっては何度もコピーが発生する実装に
                // TODO: API の deprecate を検討

                var ret = new byte[receiveCount];
                System.Buffer.BlockCopy(buffer, 0, ret, 0, receiveCount);
                return ret;
            }
        }

        /// <summary>
        /// ターゲットからデータを受信します。
        /// </summary>
        /// <param name="buffer">受信バッファ</param>
        /// <param name="offset">受信バッファにおいてデータの格納する開始オフセット</param>
        /// <param name="count">受信するサイズ</param>
        /// <returns>受信したバイト数</returns>
        public int Receive(byte[] buffer, int offset, int count)
        {
            return m_Mux.Receive(m_Channel, buffer, offset, count);
        }

        /// <summary>
        /// ハンドシェークを実行してターゲットとの通信を終了します。
        /// </summary>
        public void Shutdown()
        {
            m_Mux.Shutdown(m_Channel);
        }

        public void Dispose()
        {
            m_Mux.Close(m_Channel);
        }
    }
}
