﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using Nintendo.FsAccessLogAnalysis;

namespace Nintendo.FsAccessLogChecker
{
    internal class FsAccessLogChecker
    {
        private class FsAccessLogParserPreListener : IFsAccessLogParserListener
        {
            public void SetUp(FsAccessLogAnalyzer analyzer) { }
            public void TearDown() { }
            public void OnParseLine(string line, FsAccessLog log)
            {
                if (log?.Index != null
                    && (string.Equals(log.GetFunctionFullName(), "MountCacheStorage") || string.Equals(log.GetFunctionFullName(), "GetCacheStorageSize")))
                {
                    IsCacheStorageIndexUsed = true;
                }
            }
            public bool IsCacheStorageIndexUsed { get; private set; } = false;
        }

        private class FsAccessLogParserListener : IFsAccessLogParserListener
        {
            public bool IsValid { get; private set; }

            public FsAccessLogParserListener()
            {
                IsValid = true;
            }

            public void SetUp(FsAccessLogAnalyzer analyzer)
            {
                GuidelineChecker = new FsGuidelineChecker();
                GuidelineChecker.SetUp(analyzer);
            }

            public void OnParseLine(string line, FsAccessLog log)
            {
                GuidelineChecker.OnParseLine(line, log, Settings.Verbose, Settings.Quiet);
            }

            public void TearDown()
            {
                GuidelineChecker.TearDown();
                if (!GuidelineChecker.IsValid)
                {
                    IsValid = false;
                }
                GuidelineChecker = null;
            }

            private FsGuidelineChecker GuidelineChecker { get; set; }
        }

        internal static int Main(string[] args)
        {
            CommandLineOption commandLineOption = new CommandLineOption();
            try
            {
                if (!commandLineOption.Parse(args))
                {
                    return 0;
                }
            }
            catch (CommandLineParseErrorException e)
            {
                Console.WriteLine(e.Message);
                commandLineOption.ShowHelp();
                return 1;
            }

            bool isValidSettings = true;
            Settings = commandLineOption.Settings;
            if (Settings.Output != null)
            {
                while (File.Exists(Settings.Output) && Settings.Force == false)
                {
                    Console.Write("File \"" + Settings.Output + "\" already exists. Do you wish to overwrite? [Y/n] ");
                    var input = Console.ReadLine();
                    if (input == "Y" || input == "y" || input == string.Empty)
                    {
                        break;
                    }
                    else if (input == "N" || input == "n")
                    {
                        return 1;
                    }
                    Console.WriteLine(string.Empty);
                }
                try
                {
                    Logger.FileOutputStreamWriter = new StreamWriter(Settings.Output);
                    Logger.FileOutputStreamWriter.AutoFlush = true;
                }
                catch (Exception e)
                {
                    Console.WriteLine(Settings.Output + ": " + e.Message);
                    isValidSettings = false;
                }
            }

            CheckMountTargets = FsGuidelineChecker.CheckMountTargets.ToList();

            if (Settings.AlsoCheckHost)
            {
                CheckMountTargets.Add(FsMountTarget.Host);
            }

            FsAccessLogParser parser = OpenFile(Settings.LogFile);
            if (parser == null)
            {
                isValidSettings = false;
            }

            if (Settings.TimeRange.HasValue)
            {
                if (!FsGuideline.SetCheckStep(Settings.TimeRange.Value))
                {
                    isValidSettings = false;
                }
            }

            if (!FsGuideline.SetWarningThreshold(((double)Settings.WarningRatio) / 100.0))
            {
                isValidSettings = false;
            }

            if (!isValidSettings)
            {
                return 1;
            }

            try
            {
                // ツールバージョンを出力する（ファイルにも含める）
#if FSACCESSLOGCHECKER_FOR_SYSTEM
                Logger.WriteLineWithFile(commandLineOption.GetVersionString() + " (For only System)");
#else
                Logger.WriteLineWithFile(commandLineOption.GetVersionString());
#endif
                {
                    FsAccessLogParserPreListener preListener = new FsAccessLogParserPreListener();
                    parser.Listener = preListener;
                    List<FsAccessLogAnalyzer> resultList = parser.Parse();
                    FsGuideline.IsCacheStorageIndexUsed = preListener.IsCacheStorageIndexUsed;
                }

                FsAccessLogParserListener listener = new FsAccessLogParserListener();
                parser.Listener = listener;
                List<FsAccessLogAnalyzer> analyzerList = parser.Parse();
                if (analyzerList == null)
                {
                    return 1;
                }

                bool isValid = listener.IsValid;

                if (analyzerList.Count > 1)
                {
                    Logger.OutputWarning(" There are multiple fs access logs");
                }

                if (FsGuideline.IsChangeCheckStep())
                {
                    // アクセスチェック幅を変更した場合は、FAIL とする
                    Logger.OutputFail(string.Format("Time range set to {0}, this analysis result requires additional check the usecase.",
                        MessageFormat.FormatTotalMinutes(FsGuideline.CheckMillisecondsStep)));
                    isValid = false;
                }

                {
                    bool isComplete = true;
                    foreach (var log in analyzerList)
                    {
                        isComplete = isComplete && log.IsCompleteLog;
                    }
                    if (!isComplete)
                    {
                        Logger.OutputInfo("Access log is incomplete.");
                        Logger.OutputInfo("Access log is written after a while after the system is put to sleep or turned off from Power Menu.");
                        Logger.OutputInfo("Therefore, if you do the following operation before the log is written, the log may be incomplete.");
                        Logger.OutputInfo("- Remove the microSD card");
                        Logger.OutputInfo("- Remove adapter plug");
                        Logger.OutputInfo("- Turn the system off from TargetManager or ControlTarget");
                        Logger.OutputInfo("Also, please delete the incomplete log before logging to the same file.");
                    }
                }

                FsAccessLogAnalyzer analyzer = analyzerList.Last();

                if (analyzer.IsEmpty())
                {
                    Logger.WriteLineWithFile("Fs access log was not found");
                    return 1;
                }

                if (!isValid)
                {
                    return 1;
                }
            }
            catch (IOException e)
            {
                Console.WriteLine(e.Message);
                // StreamWriter は Dispose 時に Close しますが、
                // その際に Flush を行うため再度例外が発生します。
                // よって、ここで明示的に Close し例外を握りつぶします。
                try
                {
                    Logger.FileOutputStreamWriter?.Close();
                }
                catch
                {
                }
            }
            return 0;
        }

        private static FsAccessLogParser OpenFile(string path)
        {
            try
            {
                return FsAccessLogParser.Open(path);
            }
            catch (Exception e)
            {
                Console.WriteLine(path + ": " + e.Message);
            }
            return null;
        }

        internal static List<FsMountTarget> CheckMountTargets { get; private set; }

        private static CommandLineOption.OptionSettings Settings = null;
    }
}
