﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;
using Nintendo.FsAccessLogAnalysis;

namespace Nintendo.FsAccessLogChecker
{
    [SerializableAttribute]
    public class CommandLineParseErrorException : Exception
    {
        public CommandLineParseErrorException(string message)
            : base(message)
        {
        }
    }

    public class CheckMountTargetOption
    {
        public CheckMountTargetOption()
        {
        }
        public CheckMountTargetOption(string value)
        {
            HashSet<FsMountTarget> targetSet = new HashSet<FsMountTarget>();
            foreach (var targetString in value.Split(','))
            {
                if (targetString == "All")
                {
                    foreach (FsMountTarget target in Enum.GetValues(typeof(FsMountTarget)))
                    {
                        targetSet.Add(target);
                    }
                }
                else
                {
                    FsMountTarget target = FsMountTarget.Unknown;
                    if (Enum.TryParse<FsMountTarget>(targetString, out target))
                    {
                        targetSet.Add(target);
                    }
                    else
                    {
                        throw new CommandLineParseErrorException("invalid MountTarget:" + targetString);
                    }
                }
            }
            CheckMountTargets = targetSet.ToList();
        }
        public static implicit operator CheckMountTargetOption(string value)
        {
            CheckMountTargetOption option = new CheckMountTargetOption(value);
            return option;
        }
        public static CheckMountTargetOption Parse(string value)
        {
            return new CheckMountTargetOption(value);
        }
        public List<FsMountTarget> CheckMountTargets { get; private set; } = new List<FsMountTarget>();

        public static string Help
        {
            get
            {
                string help = "specify the targets to be checked (";
                foreach (FsMountTarget target in Enum.GetValues(typeof(FsMountTarget)))
                {
                    help += target.ToString() + ",";
                }
                help += "All)";
                return help;
            }
        }
    }

    public class CommandLineOption
    {
        public class OptionSettings
        {
            [PositionOption(0), RequiredOption]
            public string LogFile { get; set; } = null;
            [ShortNameOption("o"), HelpOption("output to a file"), MetaVarOption("FILE")]
            public string Output { get; set; } = null;
            [RangeValue("1", "240"), HelpOption("set check time range"), MetaVarOption("minutes")]
            public long? TimeRange { get; set; } = null;
            [RangeValue("0", "100"), HelpOption("set warning ratio"), MetaVarOption("%")]
            public int WarningRatio { get; set; } = 100;
            [SingleOption, HelpOption("verbose output")]
            public bool Verbose { get; set; } = false;
            [SingleOption, HelpOption("minimum output")]
            public bool Quiet { get; set; } = false;
            [SingleOption, ShortNameOption("f"), HelpOption("overwite output file forcely")]
            public bool Force { get; set; } = false;
            [SingleOption, HelpOption("also check host access")]
            public bool AlsoCheckHost { get; set; } = false;
            [SingleOption, ShortNameOption("h")]
            public bool Help { get; set; } = false;
            [SingleOption]
            public bool Version { get; set; } = false;
        }
        public OptionSettings Settings { get; set; } = new OptionSettings();

        public bool Parse(string[] args)
        {
            var properties = Settings.GetType().GetProperties();
            Dictionary<string, OptionProperty> mapOptions = new Dictionary<string, OptionProperty>();
            Dictionary<int, OptionProperty> mapPositionOptions = new Dictionary<int, OptionProperty>();
            foreach (var property in properties)
            {
                string memberName = property.Name;
                Type type = property.PropertyType;
                string longName = "--" + ToSnakeCase(memberName);
                string shortName = string.Empty;
                bool isSingle = false;
                bool isRequired = false;
                int position = -1;
                ValueCheckOption checker = null;

                foreach (Attribute attribute in property.GetCustomAttributes())
                {
                    if (attribute is ShortNameOption)
                    {
                        shortName = "-" + ((ShortNameOption)attribute).Name;
                    }
                    else if (attribute is SingleOption)
                    {
                        isSingle = true;
                    }
                    else if (attribute is RequiredOption)
                    {
                        isRequired = true;
                    }
                    else if (attribute is PositionOption)
                    {
                        isSingle = true;
                        position = ((PositionOption)attribute).Position;
                    }
                    else if (attribute is ValueCheckOption)
                    {
                        checker = (ValueCheckOption)attribute;
                    }
                }
                if (position >= 0)
                {
                    mapPositionOptions.Add(position, new OptionProperty(memberName, type, isSingle, isRequired, checker));
                }
                else
                {
                    mapOptions.Add(longName, new OptionProperty(memberName, type, isSingle, isRequired, checker));
                    if (shortName.Length > 0)
                    {
                        // ショートネームの方はロングネームの登録があるため isRequired は false
                        mapOptions.Add(shortName, new OptionProperty(memberName, type, isSingle, false, checker));
                    }
                }
            }

            List<string> listIsArranged = new List<string>();
            int currentPosition = 0;
            for (int i = 0; i < args.Length; ++i)
            {
                if (args[i].StartsWith("-"))
                {
                    bool isFindOption = false;
                    foreach (var option in mapOptions)
                    {
                        if (args[i] == option.Key)
                        {
                            if (option.Value.IsSingle)
                            {
                                isFindOption = true;
                                Settings.GetType().InvokeMember(option.Value.PropertyName,
                                    BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.Public,
                                    null, Settings, new object[] { true });
                                if (!listIsArranged.Contains(option.Value.PropertyName))
                                {
                                    listIsArranged.Add(option.Value.PropertyName);
                                }
                                break;
                            }
                            else if (i + 1 < args.Length)
                            {
                                isFindOption = true;
                                if (option.Value.PropertyType == typeof(string))
                                {
                                    Settings.GetType().InvokeMember(option.Value.PropertyName,
                                        BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.Public,
                                        null, Settings, new object[] { args[i + 1] });
                                }
                                else
                                {
                                    var propertyType = option.Value.PropertyType;
                                    if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition() == typeof(Nullable<>))
                                    {
                                        propertyType = Nullable.GetUnderlyingType(propertyType);
                                    }
                                    object value = 0;
                                    try
                                    {
                                        value = propertyType.InvokeMember("Parse",
                                            BindingFlags.InvokeMethod | BindingFlags.Static | BindingFlags.Public,
                                            null, null, new object[] { args[i + 1] });
                                    }
                                    catch (TargetInvocationException)
                                    {
                                        throw new CommandLineParseErrorException(string.Format("option <{0}> is invalid value: ", option.Key) + args[i + 1]);
                                    }
                                    if (option.Value.Checker != null)
                                    {
                                        if (!option.Value.Checker.Check(propertyType, value))
                                        {
                                            throw new CommandLineParseErrorException(string.Format("option <{0}> is invalid value: ", option.Key) + args[i + 1]);
                                        }
                                    }
                                    Settings.GetType().InvokeMember(option.Value.PropertyName,
                                        BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.Public,
                                        null, Settings, new object[] { value });
                                }
                                if (!listIsArranged.Contains(option.Value.PropertyName))
                                {
                                    listIsArranged.Add(option.Value.PropertyName);
                                }
                                ++i;
                                break;
                            }
                        }
                    }
                    if (!isFindOption)
                    {
                        throw new CommandLineParseErrorException("invalid commandline option:" + args[i]);
                    }
                }
                else
                {
                    if (!mapPositionOptions.ContainsKey(currentPosition))
                    {
                        throw new CommandLineParseErrorException("invalid commandline option:" + args[i]);
                    }
                    OptionProperty optionProperty = mapPositionOptions[currentPosition];
                    Settings.GetType().InvokeMember(optionProperty.PropertyName,
                        BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.Public,
                        null, Settings, new object[] { args[i] });
                    if (!listIsArranged.Contains(optionProperty.PropertyName))
                    {
                        listIsArranged.Add(optionProperty.PropertyName);
                    }
                    ++currentPosition;
                }
            }

            if (Settings.Help)
            {
                ShowHelp();
                return false;
            }
            if (Settings.Version)
            {
                ShowVersion();
                return false;
            }

            foreach (var option in mapOptions)
            {
                if (option.Value.IsRequired && !listIsArranged.Contains(option.Value.PropertyName))
                {
                    throw new CommandLineParseErrorException(string.Format("option <{0}> is required", option.Key));
                }
            }

            foreach (var option in mapPositionOptions.Values)
            {
                if (option.IsRequired && !listIsArranged.Contains(option.PropertyName))
                {
                    throw new CommandLineParseErrorException(string.Format("<{0}> is required", option.PropertyName));
                }
            }

            return true;
        }

        public string GetApplicationName()
        {
            return Assembly.GetExecutingAssembly().GetName().Name;
        }
        public string GetVersionString()
        {
            var version = Assembly.GetExecutingAssembly().GetName().Version;
            int major = version.Major;
            int minor = version.Minor;
            string versionString = string.Format("{0}.{1}", major, minor);
            return GetApplicationName() + " version " + versionString;
        }

        public void ShowVersion()
        {
            Console.WriteLine(GetVersionString());
        }

        public void ShowHelp()
        {
            Console.WriteLine(GetApplicationName() + " <LogFile> options...");
            Console.WriteLine("Options:");
            var properties = Settings.GetType().GetProperties();
            foreach (var property in properties)
            {
                string memberName = property.Name;
                string longName = "--" + ToSnakeCase(memberName);
                string shortName = string.Empty;
                string metaVar = memberName;
                string help = string.Empty;
                string valueHelp = null;
                bool isSingle = false;

                var attributes = property.GetCustomAttributes();
                if (attributes.Any(attribute => attribute is PositionOption))
                {
                    continue;
                }
                foreach (Attribute attribute in property.GetCustomAttributes())
                {
                    if (attribute is ShortNameOption)
                    {
                        shortName = "-" + ((ShortNameOption)attribute).Name;
                    }
                    else if (attribute is SingleOption)
                    {
                        isSingle = true;
                    }
                    else if (attribute is HelpOption)
                    {
                        help = ((HelpOption)attribute).Description;
                        if (help == null)
                        {
                            help = property.PropertyType.GetProperty("Help").GetValue(null) as string;
                        }
                    }
                    else if (attribute is MetaVarOption)
                    {
                        metaVar = ((MetaVarOption)attribute).Name;
                    }
                    else if (attribute is ValueCheckOption)
                    {
                        valueHelp = ((ValueCheckOption)attribute).GetValueHelp();
                    }
                }
                string outputText = "  ";
                outputText += longName;
                if (shortName.Length > 0)
                {
                    outputText += "," + shortName;
                }
                if (!isSingle)
                {
                    outputText += " <" + metaVar + ">";
                }
                if (help.Length > 0)
                {
                    outputText += ": " + help;
                }
                if (valueHelp != null)
                {
                    outputText += " " + valueHelp;
                }
                Console.WriteLine(outputText);
            }
        }

        private string ToSnakeCase(string name)
        {
            string result = string.Empty;
            foreach (char c in name)
            {
                if (char.IsUpper(c))
                {
                    if (result.Length > 0)
                    {
                        result += '_';
                    }
                    result += char.ToLower(c);
                }
                else
                {
                    result += c;
                }
            }
            return result;
        }

        private class OptionProperty
        {
            public OptionProperty(string propertyName, Type propertyType, bool isSingle, bool isRequired, ValueCheckOption checker)
            {
                PropertyName = propertyName;
                PropertyType = propertyType;
                IsSingle = isSingle;
                IsRequired = isRequired;
                Checker = checker;
            }

            public string PropertyName { get; private set; }
            public Type PropertyType { get; private set; }
            public bool IsSingle { get; private set; }
            public bool IsRequired { get; private set; }
            public ValueCheckOption Checker { get; private set; }
        }
        private class ShortNameOption : Attribute
        {
            public ShortNameOption(string name)
            {
                Name = name;
            }

            public string Name { get; private set; }
        }
        private class SingleOption : Attribute
        {
        }
        private class RequiredOption : Attribute
        {
        }
        private class PositionOption : Attribute
        {
            public PositionOption(int position)
            {
                Position = position;
            }

            public int Position { get; private set; }
        }
        private class HelpOption : Attribute
        {
            public HelpOption(string description)
            {
                Description = description;
            }

            public string Description { get; private set; }
        }
        private class HasObjectHelpOption : HelpOption
        {
            public HasObjectHelpOption()
                : base(null)
            {
            }
        }
        private class MetaVarOption : Attribute
        {
            public MetaVarOption(string name)
            {
                Name = name;
            }

            public string Name { get; private set; }
        }

        private abstract class ValueCheckOption : Attribute
        {
            public abstract bool Check(Type type, object value);
            public abstract string GetValueHelp();
        }

        private class RangeValue : ValueCheckOption
        {
            public RangeValue(string min, string max)
            {
                MinValue = min;
                MaxValue = max;
            }
            public override bool Check(Type type, object value)
            {
                IComparable comparableValue = (IComparable)value;
                var min = type.InvokeMember("Parse",
                    BindingFlags.InvokeMethod | BindingFlags.Static | BindingFlags.Public,
                    null, null, new object[] { MinValue });
                var max = type.InvokeMember("Parse",
                    BindingFlags.InvokeMethod | BindingFlags.Static | BindingFlags.Public,
                    null, null, new object[] { MaxValue });
                if (comparableValue.CompareTo(min) < 0 || comparableValue.CompareTo(max) > 0)
                {
                    return false;
                }
                return true;
            }
            public string MinValue { get; private set; }
            public string MaxValue { get; private set; }

            public override string GetValueHelp()
            {
                return string.Format("[{0},{1}]", MinValue, MaxValue);
            }
        }
    }
}
