﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

namespace Nintendo.FsAccessLogAnalysis
{
    public class MillisecondsPeriod
    {
        public MillisecondsPeriod()
        {
            Start = 0;
            End = 0;
        }
        public MillisecondsPeriod(long start, long end)
        {
            Start = start;
            End = end;
        }

        public long Start { get; set; }
        public long End { get; set; }

        public bool IsInclusion(MillisecondsPeriod other)
        {
            return (other.Start >= Start) && (other.End <= End);
        }
        public long GetDistance(MillisecondsPeriod after)
        {
            if (after.Start <= End)
            {
                return 0;
            }
            return after.Start - End;
        }

        public int CompareTo(MillisecondsPeriod other)
        {
            long diff = Start - other.Start;
            if (diff > 0)
            {
                return 1;
            }
            if (diff < 0)
            {
                return -1;
            }
            return 0;
        }
        public override string ToString()
        {
            return Start.ToString() + "-" + End.ToString();
        }

        public class OverlapCompare : IComparer<MillisecondsPeriod>
        {
            public int Compare(MillisecondsPeriod target, MillisecondsPeriod find)
            {
                if (target.Start >= find.Start)
                {
                    if (find.End < 0)
                    {
                        return 0;
                    }
                    if (target.Start <= find.End)
                    {
                        return 0;
                    }
                    return 1;
                }
                else
                {
                    if (target.End < 0)
                    {
                        return 0;
                    }
                    if (find.Start <= target.End)
                    {
                        return 0;
                    }
                    return -1;
                }
            }
        }
    }

    public class MillisecondsPeriodEnumerable : IEnumerable<MillisecondsPeriod>
    {
        public MillisecondsPeriodEnumerable(long begin, long end, long step)
        {
            Begin = begin;
            End = end;
            Step = step;
        }

        public IEnumerator<MillisecondsPeriod> GetEnumerator()
        {
            for (long begin = Begin; begin < End; begin += Step)
            {
                yield return new MillisecondsPeriod(begin, begin + Step);
            }
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        public long Begin { get; private set; }
        public long End { get; private set; }
        public long Step { get; private set; }
    }

    public class Utility
    {
        public static string ToReadabilityByteString(ulong bytes)
        {
            return ToReadabilityByteString((double)bytes);
        }
        public static string ToReadabilityByteString(double bytes)
        {
            string[] suffix = { "KB", "MB", "GB", "TB" };
            if (bytes < 1000)
            {
                return string.Format("{0} B", bytes);
            }
            double byteNumber = bytes;
            for (int i = 0; i < suffix.Length; ++i)
            {
                byteNumber /= 1024;
                if (byteNumber < 1000)
                {
                    return string.Format("{0:0.#0} {1}", byteNumber, suffix[i]);
                }
            }
            return string.Format("{0:N} {1}", byteNumber, suffix.Last());
        }
        public static string ToMegaByteString(double bytes)
        {
            double byteNumber = bytes / 1024 / 1024;
            return string.Format("{0:0.0000} MB", byteNumber);
        }
    }

    public static class ListExtensions
    {
        // BinarySearch で見つからなかった場合に１つ右のインデックスを返します
        public static int BinarySearchEqualOrRight<T>(this List<T> list, T item)
            where T : IComparable<T>
        {
            int result = list.BinarySearch(item);
            if (result < 0)
            {
                result = ~result;
            }
            else
            {
                while ((result > 0) && (list[result - 1].CompareTo(item) == 0))
                {
                    --result;
                }
            }
            return result;
        }
    }

    public class MessageFormat
    {
        private static string Enclose(string str)
        {
            int length = str.Length;
            int half = (SeparateLength - length) / 2;
            string enclosed = " " + str;
            enclosed = enclosed.PadLeft(half + length, SeparateMessagePaddingChar);
            enclosed += " ";
            enclosed = enclosed.PadRight(SeparateLength, SeparateMessagePaddingChar);
            return enclosed;
        }
        public static readonly int SeparateLength = 54;
        public static readonly char SeparateMessagePaddingChar = '-';
        public static readonly string SeparateMessageCheckLogStartTag = "FS Access Log Details";
        public static string GetSeparateMessageCheckLogStart(long time)
        {
            return Enclose(string.Format("{0} (the above {1} minutes)",
                SeparateMessageCheckLogStartTag,
                time / (60 * 1000)));
        }
        public static readonly string SeparateMessageCheckLogEnd = new string(SeparateMessagePaddingChar, SeparateLength);
        public static readonly string SeparateMessageCheckOverallResultTag = "FS Access Log Check Result";
        public static readonly string SeparateMessageCheckOverallResultStart = Enclose(SeparateMessageCheckOverallResultTag);
        public static readonly string SeparateMessageCheckOverallResultEnd = new string(SeparateMessagePaddingChar, SeparateLength);
        public static readonly string SeparateMessageAccessSummaryStartTag = "FS Access Summary";
        public static readonly string SeparateMessageAccessSummaryStart = Enclose(SeparateMessageAccessSummaryStartTag);
        public static readonly string SeparateMessageAccessSummaryEnd = new string(SeparateMessagePaddingChar, SeparateLength);
        public static readonly string ForSystemMarker = "(for system)";

        public static string FormatTime(long milliseconds)
        {
            if (milliseconds < 60 * 1000)
            {
                TimeSpan time = TimeSpan.FromMilliseconds(milliseconds);
                return string.Format("{0:0} seconds", Math.Floor(time.TotalSeconds));
            }
            else
            {
                return FormatTotalMinutes(milliseconds);
            }
        }
        public static string FormatTotalMinutes(long milliseconds)
        {
            TimeSpan time = TimeSpan.FromMilliseconds(milliseconds);
            return string.Format("{0:0} minutes", Math.Floor(time.TotalMinutes));
        }

        public static string FormatAverageValue(double value)
        {
            return string.Format("{0:0.0000}", value);
        }
        public static string GetCheckLogTitle(FsGuideline.CheckResult.ErrorType errorType)
        {
            return errorType.ToString().Replace("Count", string.Empty);
        }
        public static string IntToOrdinal(int num)
        {
            switch (num % 100)
            {
                case 11:
                case 12:
                case 13:
                    return num + "th. ";
            }
            switch (num % 10)
            {
                case 1:
                    return num + "st. ";
                case 2:
                    return num + "nd. ";
                case 3:
                    return num + "rd. ";
                default:
                    return num + "th. ";
            }
        }
    }
}
