﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Nintendo.FsAccessLogAnalysis
{
    public class HandleLogInfo
    {
        public class HandleInfo
        {
            public string FilePath { get; set; }
            public ulong Value { get; set; }
            public FsMountInfo MountInfo { get; set; }

            public MillisecondsPeriod Period { get; set; } = new MillisecondsPeriod();

            public override string ToString()
            {
                string fileinfo = FilePath + "(" + GetMountTarget().ToString() + ")" + ":0x" + Value.ToString("X") + ":";
                return fileinfo + Period.ToString();
            }

            public FsMountTarget GetMountTarget()
            {
                if (MountInfo != null)
                {
                    return MountInfo.Target;
                }
                return FsMountTarget.Unknown;
            }

            public int CompareTo(HandleInfo other)
            {
                return Period.CompareTo(other.Period);
            }
            public class OverlapCompare : IComparer<HandleInfo>
            {
                public int Compare(HandleInfo target, HandleInfo find)
                {
                    var compare = new MillisecondsPeriod.OverlapCompare();
                    return compare.Compare(target.Period, find.Period);
                }
            }
        }

        public FsMountTarget GetMountTarget(FsAccessLog log)
        {
            if (log.MountTarget != FsMountTarget.Unknown)
            {
                return log.MountTarget;
            }

            FsMountTarget target = FsMountTarget.Unknown;
            var handleInfo = Find(log);
            if (handleInfo != null)
            {
                target = handleInfo.GetMountTarget();
                if (target != FsMountTarget.Unknown)
                {
                    return target;
                }
            }

            if (log.Path != null)
            {
                var mountInfo = MountInfo.GetMountInfoByPath(log);
                if (mountInfo != null)
                {
                    target = mountInfo.Target;
                }
            }
            return target;
        }

        public MountLogInfo MountInfo
        {
            get;
        } = new MountLogInfo();

        public HandleInfo Find(FsAccessLog log)
        {
            if (HandleList.ContainsKey(log.Handle))
            {
                var list = HandleList[log.Handle];
                HandleInfo target = new HandleInfo();
                target.Period.Start = log.Start;
                target.Period.End = log.End;
                int find = list.BinarySearch(target, new HandleInfo.OverlapCompare());
                if (find < 0)
                {
                    return null;
                }
                return list[find];
            }
            return null;
        }

        internal void RecordLog(FsAccessLog log)
        {
            if (MountInfo.RecordLog(log))
            {
                return;
            }
            {
                string functionFullName = log.GetFunctionFullName();
                if (FsFunction.Names.HandleOpenAccess.Contains(functionFullName))
                {
                    AddOpenHandleInfo(log);
                }
                else if (FsFunction.Names.HandleCloseAccess.Contains(functionFullName))
                {
                    AddCloseHandleInfo(log);
                }
            }

            // マウント関数でない かつ name キーが有効な場合、name からマウント先を解決する
            if (log.Name != null)
            {
                FsMountInfo mountInfo = MountInfo.GetMountInfo(log.Name, -1);
                if (mountInfo != null)
                {
                    MountInfo.RegisterMountInfo(log, mountInfo);
                }
            }

            // ハンドルからパスを解決する
            if ((log.Handle != 0) && (log.Path == null))
            {
                var handleInfo = Find(log);
                log.Path = handleInfo?.FilePath;
                // マウント情報があれば解決する
                if (handleInfo?.MountInfo != null)
                {
                    MountInfo.RegisterMountInfo(log, handleInfo.MountInfo);
                }
            }
            // パスからマウント先を解決する
            if (log.MountTarget == FsMountTarget.Unknown)
            {
                if (log.Path != null)
                {
                    FsMountInfo mountInfo = MountInfo.GetMountInfoByPath(log);
                    if (mountInfo != null)
                    {
                        MountInfo.RegisterMountInfo(log, mountInfo);
                    }
                }
            }
            // グループアクセスのテーブルから解決する
            if (log.MountTarget == FsMountTarget.Unknown)
            {
                MountInfo.RegisterMountInfoByGroupAccess(log);
            }
            if (log.Size > 0)
            {
                log.Type = FsApiType.Access;
            }
        }

        private void AddOpenHandleInfo(FsAccessLog log)
        {
            ulong handle = log.Handle;
            if (handle == 0)
            {
                return;
            }

            if (!HandleList.ContainsKey(handle))
            {
                HandleList.Add(handle, new List<HandleInfo>());
            }
            var list = HandleList[handle];
            HandleInfo handleInfo = new HandleInfo();
            handleInfo.FilePath = log.Path;
            handleInfo.MountInfo = MountInfo.GetMountInfoByPath(log);
            handleInfo.Value = log.Handle;
            handleInfo.Period.Start = log.Start;
            handleInfo.Period.End = -1;
            list.Add(handleInfo);
        }

        private void AddCloseHandleInfo(FsAccessLog log)
        {
            ulong handle = log.Handle;
            if (handle == 0)
            {
                return;
            }

            if (!HandleList.ContainsKey(handle))
            {
                HandleList.Add(handle, new List<HandleInfo>());
            }
            var list = HandleList[handle];
            long last_end = 0;
            if (list.Any())
            {
                last_end = list.Last().Period.End;
            }
            if (last_end >= 0)
            {
                Debug.Print("OpenFile is not found. handle:0x" + handle.ToString("X"));
                HandleInfo handleInfo = new HandleInfo();
                handleInfo.FilePath = string.Empty;
                handleInfo.Value = handle;
                handleInfo.Period.Start = last_end;
                handleInfo.Period.End = log.End;
                list.Add(handleInfo);
            }
            else
            {
                list.Last().Period.End = log.End;
            }
        }

        private Dictionary<ulong, List<HandleInfo>> HandleList = new Dictionary<ulong, List<HandleInfo>>();
    }
}
