﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

namespace Nintendo.FsAccessLogAnalysis
{
    public class FsGuideline
    {
        // SDK の最低バージョン
        public static readonly Version RequiredSdkVersion = new Version(0, 16, 17);

        // 同一箇所に対する読み込み回数/秒の上限
        public static readonly double CheckSameAddressReadCountPerSecondsThreshold = 1.0;

        // WriteFile のサイズ/分の上限
        public static readonly ulong CheckWriteSizePerMinutesThreshold = 16 * 1024 * 1024;
        // Write アクセス回数/分の上限
        public static readonly double CheckWriteAccessCountPerMinutesThreshold = 32.0;

        // Default Step
        public static readonly long DefaultStepLogTimeMinutes = 10;
        public static readonly long DefaultStepLogTimeMilliseconds = DefaultStepLogTimeMinutes * 60 * 1000;
        // Step に指定可能な時間の最小
        public static readonly long MinStepLogTimeMinutes = 1;
        public static readonly long MinStepLogTimeMilliseconds = MinStepLogTimeMinutes * 60 * 1000;
        // Step に指定可能な時間の最大
        public static readonly long MaxStepLogTimeMinutes = 4 * 60;
        public static readonly long MaxStepLogTimeMilliseconds = MaxStepLogTimeMinutes * 60 * 1000;

        // 解析結果を Warn にする割合のデフォルト値
        public static readonly double DefaultWarningThreshold = 1.0;
        // 解析結果を Warn にする割合の最小値
        public static readonly double MinWarningThreshold = 0.0;
        // 解析結果を Warn にする割合の最大値
        public static readonly double MaxWarningThreshold = 1.0;

        // Step ミリ秒間ごとにガイドラインチェックを行う
        public static long CheckMillisecondsStep { get; private set; } = DefaultStepLogTimeMilliseconds;

        // 値が特定の割合以下
        public static double WarningThreshold { get; private set; } = DefaultWarningThreshold;

        // 検査に必要な最小ログ時間
        public static long RequiredLogTimeMilliseconds { get { return CheckMillisecondsStep; } }

        // CacheStorage 関連の関数において、Index が指定されているか
        public static bool IsCacheStorageIndexUsed { get; set; } = false;

        // System 向けガイドライン
        private class ForSystem
        {
            // 同一箇所に対する読み込み回数/秒の上限
            public static readonly double CheckSameAddressReadCountPerSecondsThreshold = 0.25;
        }

        public static readonly FsMountTarget[] NoCheckMountTargets =
        {
            FsMountTarget.Unknown,
            FsMountTarget.Host,
        };

        public static readonly FsMountTarget[] CheckMountTargets
            = Enum.GetValues(typeof(FsMountTarget)).Cast<FsMountTarget>().Except(NoCheckMountTargets).ToArray();

        public abstract class SameAddressAccessInfo : System.IComparable
        {
            public int Count { get; internal set; } = 0;
            public double CountPerSeconds { get; internal set; } = 0;
            public SameAddressAccessInfo Clone() { return (SameAddressAccessInfo)MemberwiseClone(); }
            public abstract int CompareTo(object obj);
        }

        public class SameAddressAccessInfoPath : SameAddressAccessInfo
        {
            public SameAddressAccessInfoPath(string path)
            {
                Debug.Assert(!string.IsNullOrEmpty(path), "Invalid path.");
                Path = path;
            }

            public string Path { get; internal set; }
            public AccessInfo Positions { get; internal set; }
            public override string ToString()
            {
                if (Positions.IsGroupAccess)
                {
                    return string.Format("path: {0} - {1} times/sec",
                                Path,
                                MessageFormat.FormatAverageValue(CountPerSeconds));
                }
                return string.Format("path: {0}, offset: {1}, size: {2} - {3} times/sec",
                            Path,
                            Positions.Offset,
                            Positions.Size,
                            MessageFormat.FormatAverageValue(CountPerSeconds));
            }
            public override int CompareTo(object obj)
            {
                Debug.Assert(obj is SameAddressAccessInfo, "Invalid AccessInfo.");
                SameAddressAccessInfo accessInfo = (SameAddressAccessInfo)obj;
                if (accessInfo.CountPerSeconds != CountPerSeconds)
                {
                    return accessInfo.CountPerSeconds.CompareTo(CountPerSeconds);
                }
                if (accessInfo is SameAddressAccessInfoPath)
                {
                    SameAddressAccessInfoPath accessInfoPath = (SameAddressAccessInfoPath)accessInfo;
                    if (Path != accessInfoPath.Path)
                    {
                        return Path.CompareTo(accessInfoPath.Path);
                    }
                    if (Positions.Offset != accessInfoPath.Positions.Offset)
                    {
                        return Positions.Offset.CompareTo(accessInfoPath.Positions.Offset);
                    }
                }
                return ToString().CompareTo(accessInfo.ToString());
            }
        }
        public class SameAddressAccessInfoName : SameAddressAccessInfo
        {
            public SameAddressAccessInfoName(string name)
            {
                Debug.Assert(!string.IsNullOrEmpty(name), "Invalid name.");
                Name = name;
            }

            public string Name { get; internal set; }
            public override string ToString()
            {
                return string.Format("{0}: {1} - {2} times/sec",
                            Name == "nfp" ? "namespace" : "function",
                            Name,
                            MessageFormat.FormatAverageValue(CountPerSeconds));
            }

            public override int CompareTo(object obj)
            {
                Debug.Assert(obj is SameAddressAccessInfo, "Invalid AccessInfo.");
                SameAddressAccessInfo accessInfo = (SameAddressAccessInfo)obj;
                if (accessInfo.CountPerSeconds != CountPerSeconds)
                {
                    return accessInfo.CountPerSeconds.CompareTo(CountPerSeconds);
                }
                if (accessInfo is SameAddressAccessInfoName)
                {
                    SameAddressAccessInfoName accessInfoName = (SameAddressAccessInfoName)accessInfo;
                    if (Name != accessInfoName.Name)
                    {
                        return Name.CompareTo(accessInfoName.Name);
                    }
                }
                return ToString().CompareTo(accessInfo.ToString());
            }
        }
        public class CheckResult
        {
            [Flags]
            public enum ErrorType
            {
                None = 0x00,
                WriteSize = 1 << 0,
                WriteAccessCount = 1 << 2,
                MountCount = 1 << 3,
                SameAddressRead = 1 << 4,
            }
            // 解析で利用した時間
            public long CheckedMilliseconds { get; internal set; } = 0;
            // ログの時間範囲
            public long ActualMilliseconds { get; internal set; } = 0;
            public double WriteSizePerMinutes { get; internal set; } = 0;
            public double WriteAccessCountPerMinutes { get; internal set; } = 0;
            public Dictionary<string, double> FailMountCountPerSecondsList { get; internal set; } = new Dictionary<string, double>();
            public Dictionary<string, double> WarnMountCountPerSecondsList { get; internal set; } = new Dictionary<string, double>();
            public Dictionary<string, double> PassMountCountPerSecondsList { get; internal set; } = new Dictionary<string, double>();
            public List<SameAddressAccessInfo> FailReadAccessList { get; internal set; } = new List<SameAddressAccessInfo>();
            public List<SameAddressAccessInfo> WarnReadAccessList { get; internal set; } = new List<SameAddressAccessInfo>();
            public List<SameAddressAccessInfo> PassReadAccessList { get; internal set; } = new List<SameAddressAccessInfo>();
            public ErrorType WarningFlags { get; internal set; } = ErrorType.None;
            public ErrorType ErrorFlags { get; internal set; } = ErrorType.None;
            public bool IsVerificationError { get { return ErrorFlags != ErrorType.None; } }
            public bool ForSystem { get; internal set; } = false;
            public FsMountTarget MountTarget { get; set; } = FsMountTarget.Unknown;
        }

        public static bool SetCheckStep(long minutes)
        {
            long milliseconds = minutes * 60 * 1000;
            if ((milliseconds < MinStepLogTimeMilliseconds) || (milliseconds > MaxStepLogTimeMilliseconds))
            {
                return false;
            }
            CheckMillisecondsStep = milliseconds;
            return true;
        }
        public static bool SetWarningThreshold(double warningThreshold)
        {
            if ((warningThreshold < MinWarningThreshold) || (warningThreshold > MaxWarningThreshold))
            {
                return false;
            }
            WarningThreshold = warningThreshold;
            return true;
        }

        public static bool IsChangeCheckStep()
        {
            return CheckMillisecondsStep != DefaultStepLogTimeMilliseconds;
        }

        public static bool IsEnoughLogTime(FsAccessLogAnalyzer analyzer)
        {
            return analyzer.GetTotalMilliseconds() >= CheckMillisecondsStep;
        }

        public static bool IsAccumulatedLog(FsAccessLogList list, FsAccessLog end)
        {
            FsAccessLog first = list.Any() ? list.Begin : end;
            return FsAccessLogAnalyzer.GetStartPointElapsedMilliseconds(first, end) >= CheckMillisecondsStep;
        }

        public static CheckResult Check(List<FsAccessLog> list)
        {
            return CheckImpl(list, CheckSameAddressReadCountPerSecondsThreshold);
        }
        public static CheckResult CheckForSystem(List<FsAccessLog> list)
        {
            CheckResult result = CheckImpl(list, ForSystem.CheckSameAddressReadCountPerSecondsThreshold);
            result.ForSystem = true;
            return result;
        }

        private static CheckResult CheckImpl(List<FsAccessLog> list, double SameAddressReadCountPerSecondsThreshold)
        {
            CheckResult result = new CheckResult();
            long totalMilliseconds = FsAccessLogAnalyzer.GetStartPointElapsedMilliseconds(list);
            result.CheckedMilliseconds = GetCheckingMilliseconds(totalMilliseconds);
            result.ActualMilliseconds = totalMilliseconds;
            if (list.Any())
            {
                result.MountTarget = list.First().MountTarget;
#if DEBUG
                foreach (var log in list)
                {
                    Debug.Assert(
                        log.MountTarget == result.MountTarget || log.IsNfpAccess(),
                        "It must be the same MountTarget all");
                }
#endif
                CheckWrite(result, list);
                CheckRead(result, list, SameAddressReadCountPerSecondsThreshold);
            }
            return result;
        }

        private static void CheckWrite(CheckResult result, List<FsAccessLog> list)
        {
            ulong writeSize = FsAccessLogAnalyzer.GetTotalWriteSize(list);
            long totalMilliseconds = result.CheckedMilliseconds;
            double writeSizePerMinutes = CalculatePerMinutes(writeSize, totalMilliseconds);
            result.WriteSizePerMinutes = writeSizePerMinutes;
            if (writeSizePerMinutes > CheckWriteSizePerMinutesThreshold)
            {
                result.ErrorFlags |= CheckResult.ErrorType.WriteSize;
            }
            else if (writeSizePerMinutes > CheckWriteSizePerMinutesThreshold * WarningThreshold)
            {
                result.WarningFlags |= CheckResult.ErrorType.WriteSize;
            }

            int count = list.Sum(log => log.WriteCount);
            double writeAccessCountPerMinutes = CalculatePerMinutes(count, totalMilliseconds);
            result.WriteAccessCountPerMinutes = writeAccessCountPerMinutes;
            if (writeAccessCountPerMinutes > CheckWriteAccessCountPerMinutesThreshold)
            {
                result.ErrorFlags |= CheckResult.ErrorType.WriteAccessCount;
            }
            else if (writeAccessCountPerMinutes > CheckWriteAccessCountPerMinutesThreshold * WarningThreshold)
            {
                result.WarningFlags |= CheckResult.ErrorType.WriteAccessCount;
            }
        }

        private static void CheckRead(CheckResult result, List<FsAccessLog> list, double SameAddressReadCountPerSecondsThreshold)
        {
            long totalMilliseconds = result.CheckedMilliseconds;
            var mountCountList = GetMountCount(list);
            if (mountCountList != null)
            {
                foreach (var group in mountCountList)
                {
                    int mountCount = group.Value;
                    double mountCountPerSeconds = CalculatePerSeconds(mountCount, totalMilliseconds);
                    if (mountCountPerSeconds > SameAddressReadCountPerSecondsThreshold)
                    {
                        result.FailMountCountPerSecondsList.Add(group.Key, mountCountPerSeconds);
                        result.ErrorFlags |= CheckResult.ErrorType.MountCount;
                    }
                    else if (mountCountPerSeconds > SameAddressReadCountPerSecondsThreshold * WarningThreshold)
                    {
                        result.WarnMountCountPerSecondsList.Add(group.Key, mountCountPerSeconds);
                        result.WarningFlags |= CheckResult.ErrorType.MountCount;
                    }
                    else
                    {
                        result.PassMountCountPerSecondsList.Add(group.Key, mountCountPerSeconds);
                    }
                }
            }

            foreach (var info in GetSameAddressAccessCount(list))
            {
                double readAccessCountPerSeconds = CalculatePerSeconds(info.Count, totalMilliseconds);
                info.CountPerSeconds = readAccessCountPerSeconds;
                if (readAccessCountPerSeconds > SameAddressReadCountPerSecondsThreshold)
                {
                    result.FailReadAccessList.Add(info);
                    result.ErrorFlags |= CheckResult.ErrorType.SameAddressRead;
                }
                else if (readAccessCountPerSeconds > SameAddressReadCountPerSecondsThreshold * WarningThreshold)
                {
                    result.WarningFlags |= CheckResult.ErrorType.SameAddressRead;
                    result.WarnReadAccessList.Add(info);
                }
                else
                {
                    result.PassReadAccessList.Add(info);
                }
            }
        }

        private static Dictionary<string, int> GetMountCount(List<FsAccessLog> list)
        {
            var mounts = list.Where(log => log.IsResultSuccess() && FsFunction.IsMountFunction(log.GetFunctionFullName())).ToLookup(log => log.MountHash);
            if (!mounts.Any())
            {
                return null;
            }
            Func<IGrouping<int, FsAccessLog>, string> generateKey = (IGrouping<int, FsAccessLog> group) =>
            {
                var keys = group.Select(log => log.GetFunctionFullName()).Distinct().OrderBy(name => name);
                return string.Join("/", keys) + group.ElementAt(0).GetMountArgumentsString();
            };
            Dictionary<string, int> result = new Dictionary<string, int>();
            foreach (IGrouping<string, IGrouping<int, FsAccessLog>> logs in mounts.GroupBy(group => generateKey(group)))
            {
                result.Add(logs.Key, logs.Sum(group => group.ToList().Count()));
            }
            return result;
        }

        private static AccessInfo GetAccessInfo(FsAccessLog log)
        {
            if (!(log.Offset.Invalid || log.Size.Invalid))
            {
                if (log.Size > 0)
                {
                    return new AccessInfo(log.Offset, log.Size, 1);
                }
                if (log.FileSize != null && log.FileSize > 0)
                {
                    return new AccessInfo(0, (ulong)log.FileSize, 1);
                }
            }
            // グループアクセスを検索
            FsFunction.GroupAccess groupAccess;
            if (FsFunction.TryGetReadGroupAccess(log, out groupAccess))
            {
                return new AccessInfo(0, groupAccess.ReadSize, groupAccess.ReadCount, true);
            }
            return null;
        }

        private static IEnumerable<SameAddressAccessInfo> GetSameAddressAccessCount(List<FsAccessLog> list)
        {
            // キャッシュは現状考慮していない
            var readList = list.Where(log => log.AccessType.HasFlag(FsApiAccessType.Read)).ToList();
            Dictionary<int, List<AccessInfo>> accessMap = new Dictionary<int, List<AccessInfo>>();
            Dictionary<int, SameAddressAccessInfo> countMap = new Dictionary<int, SameAddressAccessInfo>();
            foreach (var log in readList)
            {
                if (!(log.IsResultSuccess() || log.IsNfpAccess()))
                {
                    continue;
                }
                AccessInfo accessInfo = GetAccessInfo(log);
                if (accessInfo != null)
                {
                    int hash = log.GetHashCode();
                    if (accessMap.ContainsKey(hash))
                    {
                        int index = accessMap[hash].BinarySearch(accessInfo);
                        if (index < 0)
                        {
                            // 見つからなかったら挿入する
                            accessMap[hash].Insert(~index, accessInfo);
                        }
                        else
                        {
                            // 見つかった場合はカウントを増やす
                            ++accessMap[hash][index].Count;
                        }
                    }
                    else
                    {
                        accessMap.Add(hash, new List<AccessInfo>() { accessInfo });
                        if (!string.IsNullOrEmpty(log.Path))
                        {
                            countMap.Add(hash, new SameAddressAccessInfoPath(log.Path));
                        }
                        else
                        {
                            countMap.Add(hash, new SameAddressAccessInfoName(log.GetAccessInfoName()));
                        }
                    }
                }
            }

            // ファイル（ハンドル）に紐付いた Read アクセスを総当りで衝突回数を計算
            foreach (var pair in accessMap)
            {
                var accessList = pair.Value;
                List<AccessInfo> hitList = new List<AccessInfo>();
                // 逆順から探索することでヒット率を上げる
                foreach (var accessInfo in Enumerable.Reverse(accessList))
                {
                    hitList.AddRange(CheckHitList(accessInfo, hitList));
                }
                foreach (var hit in hitList.Where(hit => hit.Count >= 1))
                {
                    SameAddressAccessInfo info = countMap[pair.Key].Clone();
                    info.Count = hit.Count;
                    if (info is SameAddressAccessInfoPath)
                    {
                        ((SameAddressAccessInfoPath)info).Positions = hit;
                    }
                    yield return info;
                }
            }
        }

        private static List<AccessInfo> CheckHitList(AccessInfo accessInfo, List<AccessInfo> hitList)
        {
            List<AccessInfo> newHitList = new List<AccessInfo>();
            List<AccessInfo> remainedAccessList = new List<AccessInfo>();
            remainedAccessList.Add(accessInfo);
            while (remainedAccessList.Count > 0)
            {
                var remainedAccess = remainedAccessList.First();
                remainedAccessList.RemoveAt(0);
                foreach (var hitInfo in hitList)
                {
                    if (remainedAccess.IsIntersect(hitInfo))
                    {
                        if (remainedAccess.Offset < hitInfo.Offset)
                        {
                            // remainedAccess の前方で衝突しなかった部分は後々へ積み残す
                            remainedAccessList.Add(new AccessInfo(remainedAccess.Offset, hitInfo.Offset - remainedAccess.Offset, remainedAccess.Count));
                            remainedAccess.Begin = hitInfo.Offset;
                        }
                        else if (hitInfo.Offset < remainedAccess.Offset)
                        {
                            // hitInfo の前方で衝突しなかった部分
                            newHitList.Add(new AccessInfo(hitInfo.Offset, remainedAccess.Offset - hitInfo.Offset, hitInfo.Count));
                            hitInfo.Begin = remainedAccess.Offset;
                        }

                        if (remainedAccess.End < hitInfo.End)
                        {
                            // hitInfo の後方で衝突しなかった部分
                            newHitList.Add(new AccessInfo(remainedAccess.End, hitInfo.End - remainedAccess.End, hitInfo.Count));
                            hitInfo.End = remainedAccess.End;
                        }

                        // hitInfo が確定したのでカウントアップ
                        hitInfo.Count += remainedAccess.Count;

                        if (remainedAccess.End == hitInfo.End)
                        {
                            // remainedAccess の未衝突範囲がなくなるので検索終了
                            remainedAccess = null;
                            break;
                        }
                        else
                        {
                            // remainedAccess の後方で衝突しなかった部分
                            remainedAccess.Begin = hitInfo.End;
                        }
                    }
                }
                if (remainedAccess != null)
                {
                    // 最終的に残った部分
                    newHitList.Add(remainedAccess);
                }
            }
            return newHitList;
        }

        private static long GetCheckingMilliseconds(long totalMilliseconds)
        {
            if (totalMilliseconds < CheckMillisecondsStep)
            {
                return CheckMillisecondsStep;
            }
            return totalMilliseconds;
        }

        private static double CalculatePerSeconds<T>(T value, long milliseconds)
            where T : IConvertible
        {
            return (double)Convert.ChangeType(value, typeof(double)) * 1000 / milliseconds;
        }
        private static double CalculatePerMinutes<T>(T value, long milliseconds)
            where T : IConvertible
        {
            return (double)Convert.ChangeType(value, typeof(double)) * 60 * 1000 / milliseconds;
        }
    }
}
