﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Nintendo.FsAccessLogAnalysis
{
    public class LineInfo
    {
        /// <summary>
        /// アクセスログが示す時間
        /// </summary>
        public MillisecondsPeriod Period { get; set; }

        /// <summary>
        /// 行番号
        /// </summary>
        public long LineNumber { get; set; }

        /// <summary>
        /// 文字列データ
        /// </summary>
        public string Log { get; set; }
    }

    public class AccessLogLineInfo : IComparable<AccessLogLineInfo>
    {
        public LineInfo LineInfo { get; set; }
        public List<LineInfo> PreLines { get; set; } = new List<LineInfo>();

        public int CompareTo(AccessLogLineInfo other)
        {
            return LineInfo.Period.Start.CompareTo(other.LineInfo.Period.Start);
        }
    }

    public class FsAccessLogStreamSorter : IDisposable
    {
        /// <summary>
        /// アクセスログを含んだデータの入力ストリーム
        /// </summary>
        public Stream InputStream { get; private set; }

        /// <summary>
        /// 入力ストリームの文字コード
        /// </summary>
        private Encoding InputStreamEncoding { get; set; }

        /// <summary>
        /// アクセス時間でソートした結果を出力する一時ファイルパス
        /// </summary>
        private string TemporaryFilePath { get; set; }

        /// <summary>
        /// 一時ファイルの文字コード
        /// </summary>
        private Encoding OutputEncoding { get; set; }

        public FsAccessLogStreamSorter(Stream inputStream, string temporaryFilePath)
        {
            InputStream = inputStream;
            TemporaryFilePath = temporaryFilePath;
            InputStreamEncoding = Encoding.Default;
            OutputEncoding = Encoding.Unicode;
        }

        public void Dispose()
        {
            File.Delete(TemporaryFilePath);
        }

        public StreamReader GetReader()
        {
            if (!InputStream.CanSeek)
            {
                return new StreamReader(InputStream);
            }
            CreateSortedTemporaryFile(TemporaryFilePath);
            return new StreamReader(File.OpenRead(TemporaryFilePath));
        }

        private void CreateSortedTemporaryFile(string temporaryFilePath)
        {
            Debug.Assert(InputStream.CanSeek, "InputStream must be seekable.");

            // 入力ファイルの情報をセットアップ
            SetupInputStream(temporaryFilePath);

            using (StreamReader streamReader = new StreamReader(InputStream, InputStreamEncoding, true, 1024, true))
            {
                Lines lines = new Lines();
                int lineNumber = 0;

                // アクセスログとその前方にある通常ログのリストを構築
                while (streamReader.Peek() > 0)
                {
                    string line = streamReader.ReadLine();

                    LineInfo lineInfo = new LineInfo();
                    lineInfo.LineNumber = lineNumber;
                    int markerIndex = line.IndexOf(FsAccessLogParser.FsLogMarker);
                    if (markerIndex >= 0)
                    {
                        lineInfo.Period = GetPeriod(line);
                        if (line.IndexOf("sdk_version") >= 0)
                        {
                            // 開始ログを見つけた場合、ここまでの内容を書き出す
                            Flush(lines, temporaryFilePath);
                        }
                    }
                    lineInfo.Log = line;
                    if (lineInfo.Period != null)
                    {
                        AccessLogLineInfo accessLogLine = new AccessLogLineInfo();
                        accessLogLine.LineInfo = lineInfo;
                        accessLogLine.PreLines.AddRange(lines.RemaingLogs);
                        lines.AccessLogs.Add(accessLogLine);
                        lines.RemaingLogs.Clear();
                    }
                    else
                    {
                        lines.RemaingLogs.Add(lineInfo);
                    }
                    ++lineNumber;
                }
                Flush(lines, temporaryFilePath);
            }
        }

        private void SetupInputStream(string temporaryFilePath)
        {
            if (InputStream.Length >= 2)
            {
                byte[] buf = new byte[4];
                InputStream.Read(buf, 0, Math.Min(buf.Length, (int)InputStream.Length));
                InputStream.Seek(0, SeekOrigin.Begin);

                if ((InputStreamEncoding == Encoding.Default)
                 && (InputStream.Length >= 3))
                {
                    if ((buf[0] == 0xEF) && (buf[1] == 0xBB) && (buf[2] == 0xBF))
                    {
                        InputStreamEncoding = Encoding.UTF8;
                    }
                }
                if (InputStreamEncoding == Encoding.Default)
                {
                    if ((buf[0] == 0xFE) && (buf[1] == 0xFF))
                    {
                        InputStreamEncoding = Encoding.BigEndianUnicode;
                    }
                    else if ((buf[0] == 0xFF) && (buf[1] == 0xFE))
                    {
                        InputStreamEncoding = Encoding.Unicode;
                    }
                    InputStream.Seek(0, SeekOrigin.Begin);
                }
            }

            // 書き出し先ファイルの新規作成、BOM の書き出し
            if (InputStream.Length >= 0)
            {
                using (FileStream fileOut = File.Open(temporaryFilePath, FileMode.Create))
                {
                    if (OutputEncoding == Encoding.Unicode)
                    {
                        byte[] bom = new byte[2];
                        bom[0] = 0xFF;
                        bom[1] = 0xFE;
                        fileOut.Write(bom, 0, 2);
                    }
                    else if (OutputEncoding == Encoding.BigEndianUnicode)
                    {
                        byte[] bom = new byte[2];
                        bom[0] = 0xFE;
                        bom[1] = 0xFF;
                    }
                    else if (OutputEncoding == Encoding.UTF8)
                    {
                        byte[] bom = new byte[3];
                        bom[0] = 0xEF;
                        bom[1] = 0xBB;
                        bom[2] = 0xBF;
                        fileOut.Write(bom, 0, 3);
                    }
                }
            }
        }

        private MillisecondsPeriod GetPeriod(string line)
        {
            MillisecondsPeriod period = new MillisecondsPeriod();
            Func<string, long> getTime = (string marker) =>
            {
                int index = line.IndexOf(marker);
                if (index >= 0)
                {
                    string time = line.Substring(index + marker.Length);
                    time = time.Split(new string[] { " ", "," }, 2, StringSplitOptions.RemoveEmptyEntries)[0];
                    return long.Parse(time);
                };
                return -1;
            };

            period.Start = getTime("start:");
            if (period.Start < 0)
            {
                return null;
            }
            period.End = getTime("end:");
            if (period.End < 0)
            {
                return null;
            }
            return period;
        }

        private void Flush(Lines lines, string temporaryFilePath)
        {
            // ソートする
            lines.Sort();

            // ソート情報に従って一時ファイルを作成
            using (FileStream fileStream = File.Open(temporaryFilePath, FileMode.Append))
            {
                using (StreamWriter writer = new StreamWriter(fileStream, OutputEncoding))
                {
                    Action<LineInfo> write = (LineInfo info) =>
                    {
                        writer.WriteLine(info.Log);
                    };

                    foreach (var info in lines.AccessLogs)
                    {
                        // 通常ログを書き出し
                        foreach (var line in info.PreLines)
                        {
                            write(line);
                        }
                        write(info.LineInfo);
                    }
                    foreach (var line in lines.RemaingLogs)
                    {
                        write(line);
                    }
                }
            }

            lines.Clear();
        }

        private class Lines
        {
            public List<LineInfo> RemaingLogs { get; set; } = new List<LineInfo>();
            public List<AccessLogLineInfo> AccessLogs { get; set; } = new List<AccessLogLineInfo>();

            public void Sort()
            {
                List<AccessLogLineInfo> work = new List<AccessLogLineInfo>(AccessLogs.Count / 2);
                Sort(0, AccessLogs.Count, work);
            }

            public void Clear()
            {
                RemaingLogs.Clear();
                AccessLogs.Clear();
            }

            private void Sort(int begin, int end, List<AccessLogLineInfo> work)
            {
                if (end - begin < 128)
                {
                    InsertSort(begin, end);
                    return;
                }
                int middle = (begin + end) / 2;
                Sort(begin, middle, work);
                Sort(middle, end, work);
                Merge(begin, middle, end, work);
            }

            private void Merge(int begin, int middle, int end, List<AccessLogLineInfo> left)
            {
                left.Clear();
                left.AddRange(AccessLogs.GetRange(begin, middle - begin));

                int index = begin;
                int left_index = 0;
                int right_index = middle;
                for (; left_index < left.Count && right_index < end; ++index)
                {
                    if (left[left_index].CompareTo(AccessLogs[right_index]) <= 0)
                    {
                        AccessLogs[index] = left[left_index];
                        ++left_index;
                    }
                    else
                    {
                        // 右から取る場合は、PreLine の付け替えをする
                        if (right_index + 1 >= AccessLogs.Count)
                        {
                            // ログがなくなる場合は Remaing に追加
                            RemaingLogs.InsertRange(0, AccessLogs[right_index].PreLines);
                        }
                        else
                        {
                            AccessLogs[right_index + 1].PreLines.InsertRange(0, AccessLogs[right_index].PreLines);
                        }
                        AccessLogs[right_index].PreLines.Clear();
                        AccessLogs[right_index].PreLines.AddRange(left[left_index].PreLines);
                        TrimExcess(AccessLogs[right_index].PreLines);
                        left[left_index].PreLines.Clear();
                        TrimExcess(left[left_index].PreLines);
                        AccessLogs[index] = AccessLogs[right_index];
                        ++right_index;
                    }
                }

                for (; left_index < left.Count; ++left_index, ++index)
                {
                    AccessLogs[index] = left[left_index];
                }
                for (; right_index < end; ++right_index, ++index)
                {
                    AccessLogs[index] = AccessLogs[right_index];
                }
            }

            private void InsertSort(int begin, int end)
            {
                for (int i = begin + 1; i < end; ++i)
                {
                    for (int j = i; j > begin; --j)
                    {
                        if (AccessLogs[j].CompareTo(AccessLogs[j - 1]) < 0)
                        {
                            // 右から取る場合は、PreLine の付け替えをする
                            if (j + 1 >= AccessLogs.Count)
                            {
                                // ログがなくなる場合は Remaing に追加
                                RemaingLogs.InsertRange(0, AccessLogs[j].PreLines);
                            }
                            else
                            {
                                if (AccessLogs[j + 1].PreLines.Any())
                                {
                                    AccessLogs[j + 1].PreLines.InsertRange(0, AccessLogs[j].PreLines);
                                }
                                else
                                {
                                    var temp = AccessLogs[j + 1].PreLines;
                                    AccessLogs[j + 1].PreLines = AccessLogs[j].PreLines;
                                    AccessLogs[j].PreLines = temp;
                                }
                            }
                            AccessLogs[j].PreLines.Clear();
                            AccessLogs[j].PreLines.AddRange(AccessLogs[j - 1].PreLines);
                            TrimExcess(AccessLogs[j].PreLines);
                            AccessLogs[j - 1].PreLines.Clear();
                            TrimExcess(AccessLogs[j - 1].PreLines);
                            {
                                AccessLogLineInfo temp = AccessLogs[j];
                                AccessLogs[j] = AccessLogs[j - 1];
                                AccessLogs[j - 1] = temp;
                            }
                        }
                    }
                }
            }

            private void TrimExcess(List<LineInfo> list)
            {
                if (list.Capacity >= 256)
                {
                    list.TrimExcess();
                }
            }
        }
    }
}
