﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>

#include "CrashReport.h"
#include "CrashReportRaw.h"

namespace nn { namespace creport {

namespace {

    const size_t RsaKeyLength = 256;
    nn::Bit8     g_Modulus[RsaKeyLength];
    nn::Bit8     g_PrivateExponent[RsaKeyLength];

}

size_t ReceiveEncryptedData(Bit8* buffer, size_t bufferSize)
{
    int value = 0;
    int base = 16;
    size_t dataCount = 0;
    bool inputFlag = false;
    bool crlfFlag = false;
    while (dataCount < bufferSize)
    {
        int c = ::getc(stdin);

        if (c == '\r' || c == '\n')
        {
            if (crlfFlag)
            {
                break;
            }
            crlfFlag = true;
        }
        else
        {
            crlfFlag = false;
        }

        if (c >= '0' && c <= '9')
        {
            value = value * base + c - '0';
            inputFlag = true;
        }
        else if (c >= 'A' && c <= 'F')
        {
            value = value * base + c - 'A' + 10;
            inputFlag = true;
        }
        else if (c >= 'a' && c <= 'f')
        {
            value = value * base + c - 'a' + 10;
            inputFlag = true;
        }
        else if (c == '[')
        {
            base = 10;
        }
        else if (c == ']' || c == '/')
        {
            break;
        }
        else
        {
            if (inputFlag)
            {
                buffer[dataCount++] = static_cast<Bit8>(value);
                value = 0;
                inputFlag = false;
            }
        }
    }

    if (inputFlag)
    {
        buffer[dataCount++] = static_cast<Bit8>(value);
    }

    return dataCount;
}

void DecryptData(void* buffer, size_t bufferSize, Bit8* encKey, size_t encKeySize, Bit8* encData, size_t encDataSize)
{
    nn::Bit8 keyIv[32];

    nn::crypto::RsaOaepDecryptor<RsaKeyLength, nn::crypto::Sha256Generator> rsaDecryptor;
    rsaDecryptor.Initialize(g_Modulus, sizeof(g_Modulus), g_PrivateExponent, sizeof(g_PrivateExponent));
    rsaDecryptor.Decrypt(keyIv, sizeof(keyIv), encKey, encKeySize);

    nn::crypto::DecryptAes128Ctr(buffer, bufferSize, keyIv, 16, keyIv + 16, 16, encData, encDataSize);
}

void CrashReportDecryptor()
{
    Bit8 encKey[256];
    Bit8 encData[1024];
    CrashReportRaw creportRaw;
    size_t keySize;
    size_t dataSize;

    NN_LOG("Input encryption_key:\n");
    keySize = ReceiveEncryptedData(encKey, sizeof(encKey));
    NN_ABORT_UNLESS(keySize == sizeof(encKey));

    NN_LOG("Input encrypted_exception_info:\n");
    dataSize = ReceiveEncryptedData(encData, sizeof(encData));
    //NN_ABORT_UNLESS(dataSize == sizeof(encData));

    DecryptData(&creportRaw, sizeof(creportRaw), encKey, sizeof(encKey), encData, dataSize);

    // Check signature and version
    NN_ABORT_UNLESS(creportRaw.header.signature == 0x50455243);
    NN_ABORT_UNLESS(creportRaw.header.version <= 0x3);

    CrashReport creport;
    creport.ParseRawData(creportRaw);
    creport.Dump();
    creport.ShowStackTraceHash();
}

bool LoadPrivateKey(const char* path)
{
    FILE* fp = fopen(path, "rb");
    if (fp == nullptr)
    {
        NN_LOG("[%s] not found.\n", path);
        return false;
    }

    NN_ABORT_UNLESS(fread(g_Modulus, sizeof(Bit8), RsaKeyLength, fp)==RsaKeyLength);
    NN_ABORT_UNLESS(fread(g_PrivateExponent, sizeof(Bit8), RsaKeyLength, fp)==RsaKeyLength);

    fclose(fp);
    return true;
}

}} // nn::creport

extern "C" int nnMain()
{
    // 引数
    int numArgs = nn::os::GetHostArgc();
    char** args = nn::os::GetHostArgv();

    if (numArgs < 2)
    {
        NN_LOG("usage: CrashReportDecryptor.exe <private key path>\n");
        return 1;
    }

    // 鍵を読み込む
    if (nn::creport::LoadPrivateKey(args[1]) == false)
    {
        return 1;
    }

    // 復号化
    nn::creport::CrashReportDecryptor();

    return 0;
}
