﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Net.Http;
using System.Collections;

using System.Diagnostics;

// for Json
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.IO;

using System.Text.RegularExpressions;

namespace CrashReport
{
    public static class SymbolResolver
    {
        static private Process m_Process = null;

        static public void Prepare(string addr2LinePath, string nssPath)
        {
            if (!(addr2LinePath != null && nssPath != null))
            {
                return;
            }

            if (!(System.IO.File.Exists(addr2LinePath) == true && System.IO.File.Exists(nssPath) == true))
            {
                return;
            }

            if (m_Process == null)
            {
                m_Process = new Process();
                m_Process.StartInfo.UseShellExecute = false;
                m_Process.StartInfo.RedirectStandardOutput = true;
                m_Process.StartInfo.RedirectStandardInput = true;
                m_Process.StartInfo.FileName = addr2LinePath;
                m_Process.StartInfo.Arguments = string.Format("-p -C -fe {0}", nssPath);
                m_Process.Start();
            }
        }

        static public string GetSymbol(UInt64 address)
        {
            m_Process.StandardInput.WriteLine("0x{0:X}", address);
            return m_Process.StandardOutput.ReadLine().TrimEnd('\n', '\r');
        }
    }

    [DataContract]
    public class CrashInformation
    {
        public struct ShowResult
        {
            public ulong serialCount;
            public ulong totalCount;
            public bool isShown;
        }

        [DataContract]
        private struct ModuleInfo
        {
            [DataMember(Name = "module_name", Order = 0)]
            public string moduleName;
            public byte[] moduleId;
            [DataMember(Name = "module_begin_address", Order = 1)]
            public ulong codeBase;
            [DataMember(Name = "module_end_address", Order = 2)]
            public ulong codeEnd;
        }

        [DataContract]
        private class ThreadInfo
        {
            [DataMember(Name = "thread_id", Order = 0)]
            public ulong threadId;
            [DataMember(Name = "registers", Order = 1)]
            public ulong[] reg;
            [DataMember(Name = "fp", Order = 2)]
            public ulong fp;
            [DataMember(Name = "lr", Order = 3)]
            public ulong lr;
            [DataMember(Name = "sp", Order = 4)]
            public ulong sp;
            [DataMember(Name = "pc", Order = 5)]
            public ulong pc;
            [DataMember(Name = "pstate", Order = 6)]
            public uint pstate;
            [DataMember(Name = "stack_top", Order = 7)]
            public ulong stackTop;
            [DataMember(Name = "stack_bottom", Order = 8)]
            public ulong stackBottom;
            [DataMember(Name = "stack_trace", Order = 9)]
            public ulong[] stackTrace;
            [DataMember(Name = "lr_symbol", Order = 10)]
            public string lr_symbol;
            [DataMember(Name = "pc_symbol", Order = 11)]
            public string pc_symbol;
            [DataMember(Name = "stack_trace_symbol", Order = 12)]
            public List<string> stackTraceSymbol;
        };

        [DataMember(Name = "application_version")]
        private string m_ApplicationVersion;
        [DataMember(Name = "serial_number_count")]
        private ulong m_SerialNumberCount;
        [DataMember(Name = "total_count")]
        private ulong m_TotalCount;
        [DataMember(Name = "crash_report_hash")]
        private string m_CrashHash;

        private uint m_Version;
        [DataMember(Name = "error_code")]
        private string m_ErrorCode;
        [DataMember(Name = "is_64bit")]
        private bool m_Bit64;
        private ulong m_ExceptionCode;
        [DataMember(Name = "exception_code")]
        private string m_ExceptionCodeString;
        [DataMember(Name = "exception_address")]
        private ulong m_ExceptionAddress;
        private ulong m_CodeSpecificData;
        [DataMember(Name = "exception_thread_id")]
        private ulong m_ExceptionThreadId;
        [DataMember(Name = "modules")]
        private List<ModuleInfo> m_ModuleInfo = new List<ModuleInfo>();
        [DataMember(Name = "threads")]
        private List<ThreadInfo> m_ThreadInfo = new List<ThreadInfo>();

        // For detail information (detail mode)
        [DataMember(Name = "occurrence_time")]
        private string m_OccurrenceTimeStamp;
        [DataMember(Name = "os_version")]
        private string m_OsVersion;
        [DataMember(Name = "dying_message_base64")]
        private string m_DyingMessage;

        private static readonly string[] ExceptionType = new string[]
        {
            "Undefined Instruction",
            "Access Violation Instruction",
            "Access Violation Data",
            "Data Type Missaligned",
            "Attach Break",
            "Break Point",
            "User Break",
            "Debugger Break",
            "Undefined System Call",
            "Memory System Error",
        };

        public void SetApplicationVersion(string version)
        {
            m_ApplicationVersion = version;
        }

        public void SetErrorCode(string errorCode)
        {
            m_ErrorCode = errorCode;
        }

        public void SetSerialNumberCount(ulong serialcount)
        {
            m_SerialNumberCount = serialcount;
        }

        public void SetTotalCount(ulong totalCount)
        {
            m_TotalCount = totalCount;
        }

        public void SetCrashHash(byte[] hash)
        {
            m_CrashHash = BitConverter.ToString(hash).Replace("-", string.Empty);
        }

        public void SetOccurrenceTimeStamp(string timeStamp)
        {
            m_OccurrenceTimeStamp = timeStamp;
        }

        public void SetOsVersion(string version)
        {
            m_OsVersion = version;
        }

        public void SetDyingMessage(string message)
        {
            m_DyingMessage = message;
        }

        private bool ParseCrashInfoV3(BinaryReader raw)
        {
            m_Bit64 = true;
            m_ExceptionCode = raw.ReadByte();
            m_ExceptionAddress = raw.ReadUInt64();
            m_CodeSpecificData = raw.ReadUInt64();

            var moduleCount = raw.ReadByte();
            for (int i = 0; i < moduleCount; i++)
            {
                var mi = new ModuleInfo
                {
                    moduleName = Encoding.ASCII.GetString(raw.ReadBytes(32)).TrimEnd('\0'),
                    codeBase = raw.ReadUInt64(),
                    codeEnd = raw.ReadUInt64()
                };

                if (mi.moduleName == "")
                {
                    mi.moduleName = "noname";
                }
                m_ModuleInfo.Add(mi);
            }

            var ti = new ThreadInfo();
            ti.threadId = raw.ReadUInt64();
            ti.reg = new ulong[29];
            for (int i = 0; i < 29; i++)
            {
                ti.reg[i] = raw.ReadUInt64();
            }
            ti.fp = raw.ReadUInt64();
            ti.lr = raw.ReadUInt64();
            ti.sp = raw.ReadUInt64();
            ti.pc = raw.ReadUInt64();
            ti.pstate = raw.ReadUInt32();
            ti.stackTop = raw.ReadUInt64();
            ti.stackBottom = raw.ReadUInt64();
            var stackTraceCount = raw.ReadByte();
            ti.stackTrace = new ulong[stackTraceCount];
            for (int i = 0; i < stackTraceCount; i++)
            {
                ti.stackTrace[i] = raw.ReadUInt64();
            }
            m_ThreadInfo.Add(ti);

            m_ExceptionThreadId = ti.threadId;

            return true;
        }

        private bool ParseCrashInfoV4(BinaryReader raw)
        {
            m_Bit64 = raw.ReadByte() == 0 ? false : true;
            m_ExceptionCode = raw.ReadByte();
            m_ExceptionAddress = raw.ReadUInt64();
            m_CodeSpecificData = raw.ReadUInt64();
            m_ExceptionThreadId = raw.ReadUInt64();

            var moduleCount = raw.ReadByte();
            for (int i = 0; i < moduleCount; i++)
            {
                var mi = new ModuleInfo
                {
                    moduleName = Encoding.ASCII.GetString(raw.ReadBytes(32)).TrimEnd('\0'),
                    moduleId = raw.ReadBytes(32),
                    codeBase = raw.ReadUInt64(),
                    codeEnd = raw.ReadUInt64()
                };

                if (mi.moduleName == "")
                {
                    mi.moduleName = "noname";
                }
                m_ModuleInfo.Add(mi);
            }

            var threadCount = raw.ReadUInt32();

            for (int t = 0; t < threadCount; t++)
            {
                var ti = new ThreadInfo();
                ti.threadId = raw.ReadUInt64();
                ti.reg = new ulong[29];
                for (int i = 0; i < 29; i++)
                {
                    ti.reg[i] = raw.ReadUInt64();
                }
                ti.fp = raw.ReadUInt64();
                ti.lr = raw.ReadUInt64();
                ti.sp = raw.ReadUInt64();
                ti.pc = raw.ReadUInt64();
                ti.pstate = raw.ReadUInt32();
                ti.stackTop = raw.ReadUInt64();
                ti.stackBottom = raw.ReadUInt64();
                var stackTraceCount = raw.ReadByte();
                ti.stackTrace = new ulong[stackTraceCount];
                for (int i = 0; i < stackTraceCount; i++)
                {
                    ti.stackTrace[i] = raw.ReadUInt64();
                }
                m_ThreadInfo.Add(ti);
            }

            return true;
        }

        public bool ParseCrashInfo(byte[] data, CommandLineOption option)
        {
            var signature = new byte[5];
            var result = false;

            Stream testStream = new MemoryStream(data);
            var raw = new BinaryReader(testStream);

            raw.Read(signature, 0, 4);
            m_Version = raw.ReadUInt32();

            if (Encoding.ASCII.GetString(signature).TrimEnd('\0') != "CREP")
            {
                // 不正なシグネチャなら返る
                return false;
            }

            if (m_Version > 4)
            {
                // 知らない新しいバージョンのデータだった場合
                m_ExceptionCodeString = "Update CrashReport.exe to see this crash report.";
                return true;
            }

            if (m_Version != 3 && m_Version != 4)
            {
                return false;
            }

            if (m_Version == 3)
            {
                result = ParseCrashInfoV3(raw);
            }
            else if (m_Version == 4)
            {
                result = ParseCrashInfoV4(raw);
            }

            if (result)
            {
                if (option.OutputInJson && option.SummaryOnlyFlag == false && option.CountOnlyFlag == false)
                {
                    Regex rgx = new Regex("^: ");

                    // 生の値をシンボル等に変換
                    foreach (ThreadInfo ti in m_ThreadInfo)
                    {
                        ti.lr_symbol = rgx.Replace(string.Format("{0}", GetCodeAddress(ti.lr, option)), "");
                        ti.pc_symbol = rgx.Replace(string.Format("{0}", GetCodeAddress(ti.pc, option)), "");
                        ti.stackTraceSymbol = new List<string>();
                        foreach (ulong addr in ti.stackTrace)
                        {
                            ti.stackTraceSymbol.Add(rgx.Replace(string.Format("{0}", GetCodeAddress(addr, option)), ""));
                        }
                    }
                }
                m_ExceptionCodeString = ExceptionType[m_ExceptionCode];
            }

            return result;
        }

        private string GetCodeAddress(ulong addr, CommandLineOption option)
        {
            foreach (var mi in m_ModuleInfo)
            {
                if (addr >= mi.codeBase && addr < mi.codeEnd)
                {
                    if (option.ApplicationNssPath != null)
                    {
                        string appName = Path.GetFileNameWithoutExtension(option.ApplicationNssPath);
                        string moduleName = Path.GetFileNameWithoutExtension(mi.moduleName);
                        string nssPath = "";

                        if (appName == moduleName)
                        {
                            // アプリの nss
                            nssPath = option.ApplicationNssPath;
                        }
                        else
                        {
                            //nssPath = option.NintendoSdkRoot + "\\Libraries\\NX-NXFP2-a64\\" + option.BuildType + "\\" + moduleName + ".nss";
                            //Console.WriteLine("{0} 0x{1:X}", nssPath, addr - mi.codeBase);
                            //
                            // 上記２行を有効にして、下記１行を消せば nnSdk, nnrtld 等のシンボルも表示できる
                            // moduleId によって適切な nss かどうか判定できないうちは事故の危険性が高いので無効にしておく
                            return string.Format(": 0x{0:X} + 0x{1:X} ({2})", mi.codeBase, addr - mi.codeBase, mi.moduleName);
                        }

                        string output = SymbolResolver.GetSymbol(addr - mi.codeBase - 4);

                        return string.Format(": {0} ({1})", output, mi.moduleName);
                    }
                    return string.Format(": 0x{0:X} + 0x{1:X} ({2})", mi.codeBase, addr - mi.codeBase, mi.moduleName);
                }
            }
            return "";
        }

        public bool IsShown(CommandLineOption option)
        {
            if (m_SerialNumberCount < option.FilterMinimumCount)
            {
                return false;
            }

            if (option.FilterHash != null && m_CrashHash != option.FilterHash)
            {
                return false;
            }

            return true;
        }

        public ShowResult ShowSummary(CommandLineOption option)
        {
            var result = new ShowResult();
            if (IsShown(option) == false)
            {
                result.isShown = false;
                return result;
            }

            if (option.CountOnlyFlag == false)
            {
                Console.WriteLine("{0:X64} {1} {2}/{3}", m_CrashHash, m_ErrorCode, m_SerialNumberCount, m_TotalCount);
            }

            result.serialCount = m_SerialNumberCount;
            result.totalCount = m_TotalCount;
            result.isShown = true;
            return result;
        }

        public ShowResult ShowCrashDump(CommandLineOption option, bool detailMode)
        {
            var result = new ShowResult();
            if (detailMode == false && IsShown(option) == false)
            {
                result.isShown = false;
                return result;
            }

            if (!option.BinaryInput)
            {
                Console.WriteLine("---------------------------------------------------------------------------------------");
                if (!detailMode)
                {
                    Console.WriteLine("{0:X64} {1}/{2}", m_CrashHash, m_SerialNumberCount, m_TotalCount);
                    Console.WriteLine("");
                }

                Console.WriteLine("Error code:          {0}", m_ErrorCode);
            }
            Console.WriteLine("Exception code:      {0}", m_ExceptionCodeString);
            Console.WriteLine("Exception address:   0x{0:X16}", m_ExceptionAddress);
            Console.WriteLine("Exception Thread ID: 0x{0:X16}", m_ExceptionThreadId);
            if (detailMode)
            {
                Console.WriteLine("Occurrence Time:     {0}", m_OccurrenceTimeStamp);
                Console.WriteLine("Os version:          {0}", m_OsVersion);
                Console.WriteLine("Crash Hash:          {0:X64}", m_CrashHash);
            }
            Console.WriteLine("");

            Console.WriteLine("Module Information:");
            foreach (ModuleInfo mi in m_ModuleInfo)
            {
                Console.WriteLine("  0x{0:X16} - 0x{1:X16} {2}", mi.codeBase, mi.codeEnd, mi.moduleName);
            }
            Console.WriteLine("");

            foreach (ThreadInfo ti in m_ThreadInfo)
            {
                if (detailMode == false && ti.threadId != m_ExceptionThreadId)
                {
                    continue;
                }
                Console.WriteLine("----");
                Console.WriteLine("ThreadId :  0x{0:X16} {1}", ti.threadId, (ti.threadId== m_ExceptionThreadId) ? "<Exception Thread>" : "");
                Console.WriteLine("");
                for (int i = 0; i < 29; i++)
                {
                    Console.WriteLine("X{0:D2}:  0x{1:X16}", i, ti.reg[i]);
                }
                Console.WriteLine("FP :  0x{0:X16}", ti.fp);
                Console.WriteLine("LR :  0x{0:X16}{1}", ti.lr, GetCodeAddress(ti.lr, option));
                Console.WriteLine("SP :  0x{0:X16}", ti.sp);
                Console.WriteLine("PC :  0x{0:X16}{1}", ti.pc, GetCodeAddress(ti.pc, option));
                Console.WriteLine("PSTATE:  0x{0:X8}", ti.pstate);
                Console.WriteLine("");
                Console.WriteLine("Stack: 0x{0:X16} - 0x{1:X16}", ti.stackTop, ti.stackBottom);
                Console.WriteLine("");

                Console.WriteLine("Stack trace:");
                foreach (ulong addr in ti.stackTrace)
                {
                    Console.WriteLine("  0x{0:X16}{1}", addr, GetCodeAddress(addr, option));
                }
                Console.WriteLine("");
            }
            Console.WriteLine("");

            if (detailMode)
            {
                Console.WriteLine("Dying Message (BASE64): \"\"\"");
                Console.WriteLine("{0}\"\"\"", m_DyingMessage);
                Console.WriteLine("");
            }

            result.serialCount = m_SerialNumberCount;
            result.totalCount = m_TotalCount;
            result.isShown = true;
            return result;
        }
    }
}
