﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.IO;
using CommandUtility;
using YamlDotNet;
using Nintendo.Foundation.IO;
using System.IO.Compression;

namespace ConvertPackagedFiles
{
    class ConvertPackagedFilesArguments
    {
        [CommandLineOption("output", Description = "output zip path", IsRequired = true)]
        public string outputZipPath { get; set; }

        [CommandLineOption("input", Description = "input zip path", IsRequired = true)]
        public string inputZipPath { get; set; }

        [CommandLineOption("config", Description = "convert settings file(.yaml)", IsRequired = true)]
        public string configPath { get; set; }

        [CommandLineOption('D', "define", Description = "define variables(ex. -D XX_NAME=xyz)")]
        public string[] definedVariables { get; set; }

        [CommandLineOption("temporary-root", Description = "set temporary root directory")]
        public string temporaryRoot { get; set; }

        [CommandLineOption("verbose", Description = "output many massages.")]
        public bool verbose { get; set; }
    }

    public class ConvertCommand
    {
        public string Program { get; set; }
        public string[] Arguments { get; set; }
    }

    public class ConvertRule
    {
        public string Name { get; set; }
        public string Type { get; set; }
        public string Source { get; set; }
        public string Destination { get; set; }
        public ConvertCommand[] Commands { get; set; }
    }

    public class ConvertConfig
    {
        public ConvertRule[] ConvertRules { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            ConvertPackagedFilesArguments parameters = new ConvertPackagedFilesArguments();
            if (CommandLineParser.Default.ParseArgs<ConvertPackagedFilesArguments>(args, out parameters))
            {
                VaridateParameters(parameters);

                Log.Verbose = parameters.verbose;
                TemporaryFileHolder.GlobalTemporaryRoot = parameters.temporaryRoot;

                Log.WriteLine("Load config: {0}", parameters.configPath);
                var config = ConvertUtility.LoadYamlConfig<ConvertConfig>(
                    new FileInfo(parameters.configPath),
                    ConvertUtility.MakeVariablesDictionary(parameters.definedVariables));

                ConvertPackagedFiles(
                    new FileInfo(parameters.outputZipPath),
                    new FileInfo(parameters.inputZipPath),
                    config);
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[Error] {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void ConvertPackagedFiles(FileInfo outputZipPath, FileInfo inputZipPath, ConvertConfig config)
        {
            using (var tempHolder = new TemporaryFileHolder("ConvertPackagedFiles"))
            {
                var extracted = tempHolder.CreateTemporaryDirectory("extracted");

                Log.WriteLine("Extract zip file: {0}", inputZipPath.FullName);
                Log.WriteLine("Extract to {0}", extracted.FullName);
                ZipFile.ExtractToDirectory(inputZipPath.FullName, extracted.FullName);

                Log.WriteLine("Process rules: num={0}", config.ConvertRules.Count());
                foreach (var rule in config.ConvertRules)
                {
                    Log.WriteLine("Process rule: name={0}", rule.Name);
                    ProcessRule(extracted, rule);
                }

                if (outputZipPath.Exists)
                {
                    outputZipPath.Delete();
                }

                Log.WriteLine("Compress to zip file: {0}", outputZipPath.FullName);
                ZipFile.CreateFromDirectory(extracted.FullName, outputZipPath.FullName);
            }
        }

        private static void ProcessRule(DirectoryInfo targetDirectory, ConvertRule rule)
        {
            foreach (var fileInfo in targetDirectory.EnumerateFiles("*", SearchOption.AllDirectories))
            {
                var filePath = FileUtility.GetRelativePath(fileInfo.FullName, targetDirectory);

                if (rule.Type == "PatternMatch")
                {
                    var matched = Regex.Match(filePath, rule.Source);

                    if (matched.Success)
                    {
                        var groups = (from Group g in matched.Groups select g.Value).ToArray();
                        Log.WriteLine("Target: {0}", filePath);

                        var destinationPath = Path.Combine(targetDirectory.FullName, string.Format(rule.Destination, groups));
                        Log.WriteLine("Output: {0}", destinationPath);

                        foreach( var command in rule.Commands)
                        {
                            ExecuteCommand(command, new string[] { fileInfo.FullName, destinationPath }.Concat(groups).ToArray());
                        }
                    }
                }
            }
        }

        private static void ExecuteCommand(ConvertCommand command, string[] arguments)
        {
            SdkTool.Execute(new FileInfo(command.Program), (from x in command.Arguments select string.Format(x, arguments)).ToArray());
        }

        private static void VaridateParameters(ConvertPackagedFilesArguments parameters)
        {
            if (!File.Exists(parameters.inputZipPath))
            {
                throw new Exception(string.Format("not found input zip file: {0}", parameters.inputZipPath));
            }

            if (!File.Exists(parameters.configPath))
            {
                throw new Exception(string.Format("not found config file: {0}", parameters.configPath));
            }
        }
    }
}
