﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>

namespace Nintendo {
namespace CodecTool {
namespace detail {

enum OpusResult
{
    OpusResult_Success,                     //!< 成功
    OpusResult_InvalidSampleRate,           //!< 無効なサンプルレート
    OpusResult_InvalidChannelCount,         //!< 無効なチャンネル数
    OpusResult_InvalidStreamCount,          //!< 無効なストリーム数
    OpusResult_InvalidPointer,              //!< 無効なポインタ
    OpusResult_InvalidSampleCount,          //!< 無効なサンプル数
    OpusResult_InvalidSize,                 //!< 無効なサイズ
    OpusResult_OutOfMemory,                 //!< リソース不足
    OpusResult_UnexpectedCodingMode,        //!< 指定外の符号化モード
    OpusResult_InternalError                //!< 内部エラー
};

const int ChannelCountMax = 255;

class OpusEncoderImpl
{
public:
    OpusEncoderImpl();
    ~OpusEncoderImpl();

    OpusResult Initialize(int32_t sampleRate, int32_t channelCount, int32_t totalStreamCount, int32_t stereoStreamCount, const uint8_t channelMapping[]);
    void Finalize();

    OpusResult Encode(int32_t* pOutSize, unsigned char* output, int32_t outputSize, const int16_t* input, int32_t sampleCount);

    int32_t GetSampleRate() const;
    int32_t GetChannelCount() const;
    void GetFinalRanges(uint32_t finalRanges[]) const;
    int32_t GetPreSkipSampleCount() const;

    void SetBitRate(int32_t bitRate);
    int32_t GetBitRate() const;

    enum class BitRateControl : int32_t
    {
        Invalid,
        Constant,
        Variable,
        ConstrainedVariable,
    };

    void SetBitRateControl(BitRateControl value);
    BitRateControl GetBitRateControl() const;

    enum class CodingMode : int32_t
    {
        Invalid,
        Celt,
        Silk,
        Hybrid,
        Auto,
    };

    void BindCodingMode(CodingMode value);
    CodingMode GetCodingMode() const;

    int32_t GetTotalStreamCount() const
    {
        return m_TotalStreamCount;
    }
    int32_t GetStereoStreamCount() const
    {
        return m_StereoStreamCount;
    }

    void GetChannelMapping(uint8_t channelMapping[]) const;

private:
    bool m_IsInitialized;
    int32_t m_SampleRate;
    int32_t m_ChannelCount;
    int32_t m_TotalStreamCount;
    int32_t m_StereoStreamCount;
    int32_t m_BitRate;
    int32_t m_ApplicationType;
    BitRateControl m_BitRateControl;
    CodingMode m_CodingMode;
    uint8_t m_ChannelMapping[ChannelCountMax];
    void* m_WorkBuffer;
};

}  // namespace detail
}  // namespace CodecTool
}  // namespace Nintendo
