﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Concurrency;
using System.Reactive.Linq;
using System.Threading.Tasks;
using BezelEditor.Foundation.Types;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel;
using Nintendo.Authoring.AuthoringEditor.MainWindow.SubPanel;
using Nintendo.Authoring.AuthoringEditor.MakeNspPatchWindow;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using ComparisonTargetFiles = Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow.ComparisonTargetFiles;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.StartupPanel
{
    public class StartupPanelVm : ViewModelBase
    {
        public ReactiveCommand CreateProjectCommand { get; }
        public ReactiveCommand CreateApplicationMetaCommand { get; }
        public ReactiveCommand CreateAocMetaCommand { get; }
        public ReactiveCommand CompareFilesCommand { get; }
        public ReactiveCommand OpenDocumentCommand { get; }

        public MainWindowVm Parent { get; set; }

        public ReactiveCommand<MruItemVm> OpenProjectFromProjectCommand { get; }
        public ReactiveCommand<MruItemVm> OpenProjectFromMetaCommand { get; }
        public ReactiveCommand<MruItemVm> OpenProjectFromNspCommand { get; }

        public ReadOnlyReactiveCollection<MruItemVm> MruProjects { get; set; }
        public ReadOnlyReactiveCollection<MruItemVm> MruMetas { get; set; }
        public ReadOnlyReactiveCollection<MruItemVm> MruNsps { get; set; }

        public ReactiveCommand MakeNspPatchCommand { get; set; }

        public ReactiveProperty<CultureSummry> Culture { get; }

        public bool IsSupportMakingPatch { get; }

        private readonly AppProfile _profile;
        private readonly Container _diContainer;

        public StartupPanelVm(Container diContainer, App app, Config config)
        {
            _diContainer = diContainer;
            _profile = diContainer.GetInstance<AppProfile>();

            IsSupportMakingPatch = app.Project.AppCapability.IsSupportMakingPatch;

            CreateProjectCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CreateProjectCommand
                .Subscribe(_ => StartNewProject(diContainer, AppModeType.Project, ContentMetaType.Application))
                .AddTo(CompositeDisposable);

            CreateApplicationMetaCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CreateApplicationMetaCommand
                .Subscribe(_ => StartNewProject(diContainer, AppModeType.ApplicationMeta, ContentMetaType.Application))
                .AddTo(CompositeDisposable);

            CreateAocMetaCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CreateAocMetaCommand
                .Subscribe(_ => StartNewProject(diContainer, AppModeType.AocMeta, ContentMetaType.AddOnContent))
                .AddTo(CompositeDisposable);

            CompareFilesCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CompareFilesCommand
                .Subscribe(_ => CompareFiles(diContainer))
                .AddTo(CompositeDisposable);

            OpenDocumentCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            OpenDocumentCommand.Select(_ => false).Subscribe(_ => { })
                .AddTo(CompositeDisposable);

            OpenProjectFromProjectCommand = new ReactiveCommand<MruItemVm>().AddTo(CompositeDisposable);
            OpenProjectFromProjectCommand
                .Subscribe(mruItem => OpenProject(app, config, mruItem, ImportableFileType.Project))
                .AddTo(CompositeDisposable);

            OpenProjectFromMetaCommand = new ReactiveCommand<MruItemVm>().AddTo(CompositeDisposable);
            OpenProjectFromMetaCommand.Subscribe(mruItem => OpenProject(app, config, mruItem, ImportableFileType.Meta))
                .AddTo(CompositeDisposable);

            OpenProjectFromNspCommand = new ReactiveCommand<MruItemVm>().AddTo(CompositeDisposable);
            OpenProjectFromNspCommand.Subscribe(mruItem => OpenProject(app, config, mruItem, ImportableFileType.Nsp))
                .AddTo(CompositeDisposable);

            MruProjects =
                config.MruProjects.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);

            MruMetas =
                config.MruMetas.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);

            MruNsps =
                config.MruNsps.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);

            Culture = config.ToReactivePropertyAsSynchronized(
                x => x.Culture,
                convert: x => Constants.SupportedCultures.FirstOrDefault(y => y.CultureName == x),
                convertBack: x => x.CultureName
            ).AddTo(CompositeDisposable);
        }

        private void StartNewProject(Container diContainer, AppModeType appMode, ContentMetaType contentMetaType)
        {
            _profile.AppMode = appMode;

            using (new InPreparationBlock())
            {
                ProjectFileHelper.CreateNewProject(diContainer.GetInstance<App>(), contentMetaType);
                Parent.CurrentPanelType = MainWindowVm.PanelType.ProjectEdit;
            }
        }

        private void OpenProject(App app, Config config, MruItemVm mruItem, ImportableFileType type)
        {
            ProjectFileHelper.Open(
                _diContainer, Messenger, type, config,
                mruItem?.FilePath.Value,
                null, // originalNspFilePath
                (p, originalNspFilePath) => app.Open(type, p, originalNspFilePath),
                p =>
                {
                    _profile.AppMode = TypeHelper.ToAppMode(app.Project);
                    Parent.CurrentPanelType = MainWindowVm.PanelType.ProjectEdit;
                }
            );
        }

        private void CompareFiles(Container diContainer)
        {
            /////////////////////////////////////////////////////////////

            var targetFiles = new ComparisonTargetFiles();

            using (var vm = new SelectTwoFilesWindowVm(diContainer, targetFiles))
            {
                Messenger.Raise(new TransitionMessage(vm, "ShowSelectTwoFilesWindow"));

                if (vm.Result != DialogResultType.Ok)
                    return;
            }

            /////////////////////////////////////////////////////////////
            using (new InPreparationBlock())
            {
                try
                {
                    Project project0 = null;
                    Project project1 = null;

                    if (targetFiles.FileType.IsNsp())
                    {
                        Task.WhenAll(
                            Task.Run(() => project0 = Project.Import(diContainer,
                                ImportableFileType.Nsp,
                                targetFiles.File0,
                                targetFiles.FileType0.IsPatch() ? targetFiles.OriginalApplicationNspFile0 : null)),
                            Task.Run(() => project1 = Project.Import(diContainer,
                                ImportableFileType.Nsp,
                                targetFiles.File1,
                                targetFiles.FileType1.IsPatch() ? targetFiles.OriginalApplicationNspFile1 : null))
                        ).Wait();
                    }
                    else if (targetFiles.FileType.IsMeta())
                    {
                        Task.WhenAll(
                            Task.Run(() => project0 = Project.Import(diContainer, ImportableFileType.Meta, targetFiles.File0)),
                            Task.Run(() => project1 = Project.Import(diContainer, ImportableFileType.Meta, targetFiles.File1))
                        ).Wait();
                    }

                    if (project0 == null)
                        throw new Exception(string.Format(Resources.DialogMessage_CannotLoad,targetFiles.File0));

                    if (project1 == null)
                        throw new Exception(string.Format(Resources.DialogMessage_CannotLoad,targetFiles.File1));

                    /////////////////////////////////////////////////
                    _profile.AppMode = AppModeType.Comparison;
                    Parent.CurrentPanelType = MainWindowVm.PanelType.Compare;

                    var compareVm = Parent.SelectedPanel.Value as ComparisonPanelVm;
                    Debug.Assert(compareVm != null);

                    compareVm.Targets.Clear();
                    compareVm.TargetFileType = targetFiles.FileType;

                    compareVm.Targets.Add(
                        new ComparisonPanelVm.Target { Project = project0, Name = targetFiles.File0, File = project0.NspFile });
                    compareVm.Targets.Add(
                        new ComparisonPanelVm.Target { Project = project1, Name = targetFiles.File1, File = project1.NspFile });

                    /////////////////////////////////////////////////
                    {
                        ImportableFileType fileType;
                        if (targetFiles.FileType.IsNsp())
                            fileType = ImportableFileType.Nsp;
                        else if (targetFiles.FileType.IsMeta())
                            fileType = ImportableFileType.Meta;
                        else
                            throw new ArgumentException(nameof(fileType));

                        diContainer.GetInstance<Config>().AddToMruList(fileType, targetFiles.File0);
                        diContainer.GetInstance<Config>().AddToMruList(fileType, targetFiles.File1);
                    }

                    /////////////////////////////////////////////////
                    CheckCompareTargetIsSameApplication(targetFiles, project0, project1);

                    /////////////////////////////////////////////////
                    if (compareVm.IsNoDifference)
                    {
                        Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                        {
                            Icon = TaskDialogStandardIcon.Information,
                            Caption = Resources.Information,
                            Text = Resources.DialogMessage_NoDifference,
                            StandardButtons = DialogMessage.StandardButtonsType.Ok
                        });
                    }
                }
                catch (Exception e)
                {
                    Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                    {
                        Icon = TaskDialogStandardIcon.Error,
                        Caption = Resources.Error,
                        Text = e.Message,
                        StandardButtons = DialogMessage.StandardButtonsType.Ok
                    });
                }
            }
        }

        private void CheckCompareTargetIsSameApplication(ComparisonTargetFiles targetFiles, Project project0, Project project1)
        {
            if (targetFiles.FileType.IsPatch())
            {
                var isEqualOriginalApp = project0.PatchContentMeta?.History.Count > 0
                    ? project0.PatchContentMeta.ApplicationId == project1.Meta.Core.ApplicationId
                    : project1.PatchContentMeta.ApplicationId == project0.Meta.Core.ApplicationId;

                if (isEqualOriginalApp == false)
                {
                    ShowNotSameApplicationNsp(targetFiles, project0, project1);
                }
            }
            else if (targetFiles.FileType == ComparableFileType.AppNsp)
            {
                var isSameApplication = project0.Meta.Core.ApplicationId == project1.Meta.Core.ApplicationId;
                if (isSameApplication == false)
                {
                    ShowNotSameApplicationNsp(targetFiles, project0, project1);
                }
            }
        }

        private void ShowNotSameApplicationNsp(ComparisonTargetFiles targetFiles, Project project0, Project project1)
        {
            Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Warning,
                Caption = Resources.Caution,
                Text = string.Format(
                    Resources.DialogMessge_IsNotSameApplication,
                    Path.GetFileName(targetFiles.File0), project0.Meta.Core.ApplicationId.ToHex(),
                    Path.GetFileName(targetFiles.File1), project1.Meta.Core.ApplicationId.ToHex()
                ),
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });
        }
    }
}
