﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Reactive.Linq;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Convertes;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class SupportedLanguageParamVm : ParamVm
    {
        public ObservableCollection<SupportedLanguageVm> AmericaLanguages { get; set; } =
            new ObservableCollection<SupportedLanguageVm>();

        public ObservableCollection<SupportedLanguageVm> AsiaLanguages { get; set; } =
            new ObservableCollection<SupportedLanguageVm>();

        public ObservableCollection<SupportedLanguageVm> EuropeLanguages { get; set; } =
            new ObservableCollection<SupportedLanguageVm>();

        public ReadOnlyReactiveProperty<string> ValidationMessage { get; }

        private IEnumerable<SupportedLanguageVm> AllLanguages => AmericaLanguages.Concat(AsiaLanguages).Concat(EuropeLanguages);

        private readonly bool _isSupportChineseAndKorean;

        public SupportedLanguageParamVm(Container diContainer, string captionTag, string commentTag,
            ApplicationMeta appMeta, ObservableCollection<SupportedLanguage> model, ReactiveProperty<bool> isReadOnly)
            : base(captionTag, commentTag, null)
        {
            Debug.Assert(appMeta != null);
            Debug.Assert(model != null);

            _isSupportChineseAndKorean = diContainer.GetInstance<ApplicationCapability>()
                .IsSupportEditChineseAndKoreanLanguages;

            IsReadOnly = isReadOnly;

            CompositeDisposable.Add(() => AllLanguages.ForEach(l => l.Dispose()));

            foreach (var m in model)
            {
                if (IsDisplayLanguage(m) == false)
                    continue;

                switch (LanguageHelper.LanguageToArea(m.Language))
                {
                    case AreaType.America:
                        AmericaLanguages.Add(new SupportedLanguageVm(m) { IsReadOnly = IsReadOnly });
                        break;

                    case AreaType.Asia:
                        AsiaLanguages.Add(new SupportedLanguageVm(m) { IsReadOnly = IsReadOnly });
                        break;

                    case AreaType.Europe:
                        EuropeLanguages.Add(new SupportedLanguageVm(m) { IsReadOnly = IsReadOnly });
                        break;

                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            /////////////////////////////////////////////////////////
            ValidationMessage = Observable
                .Merge(appMeta.Application.ObserveProperty(x => x.IsErrorSupportedLanguages).ToUnit())
                .Merge(CultureService.Instance.ObserveProperty(x => x.Resources).ToUnit())
                .Select(x => diContainer.GetInstance<PageValidations>().SupportedLanguagesInMeta(appMeta))
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);
        }

        private bool IsDisplayLanguage(SupportedLanguage m)
        {
            if (_isSupportChineseAndKorean)
                return true;
            // 中韓台が非サポートの環境で、かつ nmeta 上でも中韓台の言語設定が含まれない場合は該当の言語を非表示
            return m.IsNotSupportedChineseOrKoreanLanguage == false;
        }

        protected override bool IsVisibled(string keyword)
        {
            if (base.IsVisibled(keyword))
                return true;

            var c = new EnumToCaptionConverter();

            foreach (var member in  Enum.GetValues(typeof(LanguageType)))
            {
                var caption = c.Convert(member, null, null, null).ToString();

                if (caption.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;
            }

            return false;
        }
    }

    public class SupportedLanguageVm : ViewModelBase
    {
        public ReactiveProperty<LanguageType> Language { get; }
        public ReactiveProperty<bool> IsSupported { get; }
        public SupportedLanguage Model { get; }
        public ReactiveProperty<bool> IsReadOnly { get; set; }

        public ReadOnlyReactiveProperty<bool> HasTitle { get; }

        public SupportedLanguageVm(SupportedLanguage model)
        {
            Debug.Assert(model != null);

            Model = model;

            Language = model.ToReactivePropertyAsSynchronized(m => m.Language).AddTo(CompositeDisposable);
            IsSupported = model.ToReactivePropertyAsSynchronized(m => m.IsSupported).AddTo(CompositeDisposable);
            HasTitle = model.ToReadOnlyReactiveProperty(x => x.HasTitle).AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ => Language.ForceNotify())
                .AddTo(CompositeDisposable);
        }
    }
}
