﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Reactive.Linq;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class SaveDataOwnerIdParamVm : ParamVm
    {
        public ReadOnlyReactiveCollection<SaveDataOwnerIdVm> SaveDataOwnerIds { get; set; }

        public ReactiveCommand AddCommand { get; }
        public ReactiveCommand<SaveDataOwnerIdVm> RemoveCommand { get; }

        public SaveDataOwnerIdParamVm(string captionTag, string commentTag, ObservableCollection<SaveDataOwnerId> model, ReactiveProperty<bool> isReadOnly)
            : base(captionTag, commentTag, null)
        {
            Debug.Assert(model != null);

            IsReadOnly = isReadOnly;

            AddCommand = Observable
                .Merge(model.CollectionChangedAsObservable().ToUnit())
                .Merge(IsReadOnly.ToUnit())
                .Select(_ => IsReadOnly.Value == false)
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);
            AddCommand
                .Subscribe(_ =>
                {
                    model.Add(new SaveDataOwnerId());
                })
                .AddTo(CompositeDisposable);

            RemoveCommand = IsReadOnly.Select(x => x == false)
                .ToReactiveCommand<SaveDataOwnerIdVm>()
                .AddTo(CompositeDisposable);
            RemoveCommand.Subscribe(id => model.Remove(id.Model))
                .AddTo(CompositeDisposable);

            SaveDataOwnerIds =
                model.ToReadOnlyReactiveCollection(ownerId => new SaveDataOwnerIdVm(ownerId, RemoveCommand, IsReadOnly, UpdateValidation))
                    .AddTo(CompositeDisposable);

            SaveDataOwnerIds.CollectionChangedAsObservable().Subscribe(_ =>
            {
                UpdateValidation();
            }).AddTo(CompositeDisposable);

            UpdateValidation();
        }

        private void UpdateValidation()
        {
            if (SaveDataOwnerIds == null)
                return;

            foreach (var id in SaveDataOwnerIds)
            {
                id.ApplicationIdValidationError.Value =
                    SaveDataOwnerIds.Count(x => x.Model.ApplicationId == id.Model.ApplicationId) == 1
                        ? null
                        : Resources.SaveDataOwnerId_Id_Validate_Same;
            }
        }
    }

    public class SaveDataOwnerIdVm : ViewModelBase
    {
        public object[] Accessibilities { get; }

        public SaveDataOwnerId Model { get; }

        public ReactiveProperty<AccessibilityType> Accessibility { get; }

        public ReactiveProperty<ulong> Id { get; }

        public ReactiveProperty<bool> IsReadOnly { get; }

        public ReactiveCommand<SaveDataOwnerIdVm> RemoveCommand { get; }

        public ReactiveProperty<string> ApplicationIdValidationError { get; }

        public SaveDataOwnerIdVm(SaveDataOwnerId model, ReactiveCommand<SaveDataOwnerIdVm>  removeCommand,
            ReactiveProperty<bool> isReadOnly, Action onIdChanged)
        {
            Debug.Assert(model != null);

            Model = model;
            RemoveCommand = removeCommand;
            IsReadOnly = isReadOnly;

            Accessibilities = Enum.GetValues(typeof(AccessibilityType))
                .Cast<object>()
                .Where(x =>
                {
                    // nmeta に記載された値か、または SDK 利用者向けに許可されている Read 権限のみを表示
                    var t = (AccessibilityType)x;
                    return t == model.Accessibility || t == AccessibilityType.Read;
                } )
                .ToArray();
            Accessibility = model.ToReactivePropertyAsSynchronized(x => x.Accessibility).AddTo(CompositeDisposable);
            Id = model.ToReactivePropertyAsSynchronized(x => x.ApplicationId).AddTo(CompositeDisposable);
            Id.Subscribe(_ => onIdChanged()).AddTo(CompositeDisposable);

            ApplicationIdValidationError = new ReactiveProperty<string>().AddTo(CompositeDisposable);
        }
    }
}
