﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Reactive.Linq;
using BezelEditor.Foundation.Collections;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class CategoryVm : ViewModelBase
    {
        private const string CategorySeparator = "／";

        public ReadOnlyReactiveProperty<string> Name { get; }
        public AggregatedRatingCategoryData Model { get; }
        public bool IsRawValue { get; }

        public CategoryVm(AggregatedRatingCategoryData model, bool isRawValue = false)
        {
            Model = model;
            IsRawValue = isRawValue;
            Name = CultureService.Instance.ObserveProperty(x => x.Resources)
                .Select(x =>
                {
                    if (model == null)
                        return null;

                    List<string> names;
                    if (!model.Names.TryGetValue(CultureService.Instance.CultureName, out names))
                    {
                        names = model.Names.Values.FirstOrDefault() ?? new List<string>();
                    }

                    return string.Join(CategorySeparator, names);
                })
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);
        }
    }

    public class TargetSalesRegionVm : ViewModelBase
    {
        public TargetSalesRegion Region { get; set; }

        public string RegionName => Resources.ResourceManager.GetString($"TargetSalesRegion_{Region}", Resources.Culture) ?? string.Empty;

        public ReactiveProperty<bool> IsUse { get; }

        public TargetSalesRegionVm(SelectedTargetSalesRegion selectedRegion)
        {
            Region = selectedRegion.Region;
            IsUse = selectedRegion.ToReactivePropertyAsSynchronized(x => x.IsUse).AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                // ReSharper disable once ExplicitCallerInfoArgument
                .Subscribe(_ => RaisePropertyChanged(nameof(RegionName)))
                .AddTo(CompositeDisposable);
        }
    }

    public class AgeRatingSelectionVm : ViewModelBase
    {
        public ReactiveProperty<System.Windows.Visibility> Visibility { get; }

        public string Organization { get; }

        public ObservableCollection<CategoryVm> Categories { get; }

        public ReactiveProperty<CategoryVm> SelectedCategory { get; }

        public ReactiveProperty<string> WarningMessage { get; }

        public SelectedTargetSalesRegion SelectedRegion { get; }

        private static readonly CategoryVm NullCategoryVm = new CategoryVm(null);

        public AgeRatingSelectionVm(
            Rating model,
            RatingData ratingData,
            SelectedTargetSalesRegion selectedRegion,
            ReactiveProperty<string> warningMessageRp)
        {
            Organization = ratingData.RatingOrganization;
            SelectedRegion = selectedRegion;

            var reservedCategoryVm = new[] {NullCategoryVm}
                .Concat(ratingData.AgeAggregatedCategories.Select(x => new CategoryVm(x)))
                .ToArray();
            Categories = new ObservableCollection<CategoryVm>(reservedCategoryVm);

            SelectedCategory = model.ToReactivePropertyAsSynchronized(x => x.Age,
                r => GetSelectedRatingCategory(model),
                vm => vm.Model?.AgeValue).AddTo(CompositeDisposable);

            Visibility = new ReactiveProperty<System.Windows.Visibility>(System.Windows.Visibility.Collapsed)
                .AddTo(CompositeDisposable);

            model.ObserveProperty(x => x.IsUse).Where(x => x)
                .Subscribe(_ => selectedRegion.IsUse = true)
                .AddTo(CompositeDisposable);

            SelectedRegion.ObserveProperty(x => x.IsUse)
                .Subscribe(isUse =>
                {
                    model.IsUse = isUse;

                    Visibility.Value = isUse
                        ? System.Windows.Visibility.Visible
                        : System.Windows.Visibility.Collapsed;
                })
                .AddTo(CompositeDisposable);

            WarningMessage = warningMessageRp.AddTo(CompositeDisposable);
        }

        private CategoryVm GetSelectedRatingCategory(Rating r)
        {
            if (!r.Age.HasValue)
                return NullCategoryVm;

            // Age に値が反映されている時点で使用するフラグを立てる
            r.IsUse = true;

            // 年齢で集約されたレーティング区分のうち選択されているものがあれば、それを元に CategoryVm を得る
            var categoryData = r.RatingData.AgeAggregatedCategories.FirstOrDefault(y => y.AgeValue == r.Age);
            if (categoryData != null)
            {
                var selectedCategory = Categories.FirstOrDefault(y => y.Model == categoryData);
                return selectedCategory;
            }
            // レーティング区分から CategoryVm が得られなければ、CategoryVm のリストから年齢値の一致するものを得る
            var category = Categories.FirstOrDefault(x => x.Model?.AgeValue == r.Age && r.IsUse);
            if (category != null)
            {
                return category;
            }
            // 年齢値が一致するカテゴリがなければ、CategoryVm のリストにその年齢値の項目を新設し、選択済みの値として返す
            var rawAgeCategoryVm = new CategoryVm(new AggregatedRatingCategoryData
            {
                Names = new Dictionary<string, List<string>>
                {
                    { "en", new List<string>{r.Age.ToString()} },
                    { "ja", new List<string>{r.Age.ToString()} }
                },
                RatingCategories = new[]
                {
                    r.Age.ToString()
                },
                AgeValue = r.Age ?? 0
            }, true);
            Categories.Where(x => x.IsRawValue).ToArray().RemoveRangeTo(Categories);
            Categories.Add(rawAgeCategoryVm);
            return Categories.Last();
        }
    }

    public class RatingParamVm : ParamVm
    {
        public AgeRating AgeRating { get; }

        public ObservableCollection<TargetSalesRegionVm> TargetSalesRegions { get; }

        public ObservableCollection<AgeRatingSelectionVm> AgeRatingSelections { get; }

        public RatingParamVm(string captionTag, string commentTag, Application application, ReactiveProperty<bool> isReadOnly)
            : base(captionTag, commentTag, null)
        {
            var model = application.Ratings;
            Debug.Assert(model != null);

            IsReadOnly = isReadOnly;

            AgeRating = new AgeRating();
            AgeRatingSelections = Constants.AllRatingData
                .Select(x => new AgeRatingSelectionVm(
                    model.FirstOrDefault(y => y.Organization == x.Value.RatingOrganization),
                    x.Value,
                    AgeRating.SelectedTargetSalesRegions.FirstOrDefault(y => y.Region == x.Value.TargetSalesRegion),
                    new ReactiveProperty<string>()))
                .ToObservableCollection();
            TargetSalesRegions = new ObservableCollection<TargetSalesRegionVm>(
                AgeRating.SelectedTargetSalesRegions.Select(x => new TargetSalesRegionVm(x)));
        }

        protected override bool IsVisibled(string keyword)
        {
            if (base.IsVisibled(keyword))
                return true;

            var texts = new[]
            {
                Resources.Rating_Organization_Caption,
                Resources.Rating_Category_Caption
            };

            foreach (var text in texts)
                if (text.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            return false;
        }
    }
}
