﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class ParamVm : ViewModelBase
    {
        public virtual string Caption
            => CaptionTag == null ? null : Resources.ResourceManager.GetString(CaptionTag, Resources.Culture);

        public virtual string Comment
            => CommentTag == null ? null : Resources.ResourceManager.GetString(CommentTag, Resources.Culture);

        public virtual string CaptionIsUse
            => CaptionIsUseTag == null ? null : Resources.ResourceManager.GetString(CaptionIsUseTag, Resources.Culture);

        public virtual string ReadOnlyComment
            =>
            IsReadOnly.Value == false || ReadOnlyCommentTag == null
                ? null
                : Resources.ResourceManager.GetString(ReadOnlyCommentTag, Resources.Culture);

        #region CaptionIsUseTag

        private string _CaptionIsUseTag;

        public string CaptionIsUseTag
        {
            get { return _CaptionIsUseTag; }
            set
            {
                if (SetProperty(ref _CaptionIsUseTag, value))
                {
                    // ReSharper disable once ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(CaptionIsUse));
                }
            }
        }

        #endregion

        public IReactiveProperty Property { get; set; }
        public ReactiveProperty<Visibility> Visibility { get; }

        // 以下、省略可
        public double Width { get; set; } = GuiConstants.LargeWidth;
        public ReactiveProperty<bool> IsReadOnly { get; set; }
        public string ReadOnlyCommentTag { get; set; }


        private ReactiveProperty<bool> _IsUse;

        // 「項目を使用するかどうか」のチェックボックスの状態を表す
        public ReactiveProperty<bool> IsUse
        {
            get { return _IsUse; }
            set
            {
                _IsUse = value;
                IsInputEnabled = value;
            }
        }

        // 「項目が有効かどうか」の状態を表す
        // デフォルトでは IsUse と同期する（IsUse.Value == true なら同じく true）が、
        // 項目によっては逆の意味を持たせたい場合があるので、その場合はプロパティを手動設定する
        // （例）アプリケーションIDを使用する => 使用するときは項目を無効化する
        public ReactiveProperty<bool> IsInputEnabled { get; set; }

        public int MaxLength { get; set; }

        #region Additional

        private ViewModelBase _Additional;

        public ViewModelBase Additional
        {
            get { return _Additional; }
            set { SetProperty(ref _Additional, value); }
        }

        #endregion

        public ReadOnlyReactiveProperty<bool> IsHidden { get; set; }

        public string CaptionTag { get; }
        public  string CommentTag { get; }

        public ParamVm(string captionTag, string commentTag, IReactiveProperty property)
        {
            CaptionTag = captionTag;
            CommentTag = commentTag;

            Property = property;

            Visibility = new ReactiveProperty<Visibility>(System.Windows.Visibility.Visible).AddTo(CompositeDisposable);

            IsReadOnly = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            if (Property != null)
                CompositeDisposable.Add((IDisposable) Property);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    // ReSharper disable ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(Caption));
                    RaisePropertyChanged(nameof(Comment));
                    RaisePropertyChanged(nameof(CaptionIsUse));
                    // ReSharper restore ExplicitCallerInfoArgument

                    if (property != null)
                        ((dynamic) property).ForceValidate();
                })
                .AddTo(CompositeDisposable);
        }

        public void UpdateVisibility(string keyword)
        {
            if (IsHidden != null && IsHidden.Value)
                return;

            Visibility.Value = IsVisibled(keyword)
                ? System.Windows.Visibility.Visible
                : System.Windows.Visibility.Collapsed;
        }

        protected virtual bool IsVisibled(string keyword)
        {
            if (string.IsNullOrEmpty(Caption) == false)
                if (Caption.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            if (string.IsNullOrEmpty(Comment) == false)
                if (Comment.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            if (string.IsNullOrEmpty(CaptionIsUse) == false)
                if (CaptionIsUse.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            return false;
        }
    }
}
