﻿using System;
using System.Collections.Generic;
using System.Linq;
using BezelEditor.Mvvm;
using Livet;
using Nintendo.Authoring.AuthoringEditor.Convertes;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class EnumParamVm : ParamVm
    {
        public EnumChoiceVm[] Choices { get; set; }

        public static EnumParamVm Factory<T>(string captionTag, string commentTag, ReactiveProperty<T> property, IEnumerable<object> choices = null)
        {
            var vm = choices == null
                ? new EnumParamVm(captionTag, commentTag, property)
                : new EnumParamVm(captionTag, commentTag, property, choices);
            property.Subscribe(_ => vm.UpdateChoices()).AddTo(vm.CompositeDisposable);
            return vm;
        }

        private EnumParamVm(string captionTag, string commentTag, IReactiveProperty property, IEnumerable<object> choices)
            : base(captionTag, commentTag, property)
        {
            Choices = choices.Select(o =>
                {
                    var c = new EnumChoiceVm(o)
                    {
                        IsChecked = { Value = o == Property.Value },
                        IsReadOnly = IsReadOnly,
                    };

                    c.IsChecked.Subscribe(i =>
                    {
                        if (_isInUpdateChoices)
                            return;

                        if (i)
                            Property.Value = o;

                        UpdateChoices();
                    }).AddTo(CompositeDisposable);

                    return c;
                }
                ).ToArray();

            CompositeDisposable.Add(() => Choices.ForEach(c => c.Dispose()));

            UpdateChoices();
        }

        private EnumParamVm(string captionTag, string commentTag, IReactiveProperty property)
            : this(captionTag, commentTag, property, Enum.GetValues(property.Value.GetType()).Cast<object>())
        {
        }

        protected override bool IsVisibled(string keyword)
        {
            if (base.IsVisibled(keyword))
                return true;

            var enumType = Property.Value.GetType();

            var toCaption = new EnumToCaptionConverter();
            var toComment = new EnumToCommentConverter();

            foreach (var member in Enum.GetValues(enumType))
            {
                var caption = toCaption.Convert(member, null, null, null).ToString();
                if (caption.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

                var comment = toComment.Convert(member, null, null, null).ToString();
                if (comment.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;
            }

            return false;
        }

        private bool _isInUpdateChoices;

        private void UpdateChoices()
        {
            if (Choices == null)
                return;

            using (new AnonymousDisposable(() => _isInUpdateChoices = false))
            {
                _isInUpdateChoices = true;

                var value = Property.Value;
                if (value == null)
                {
                    foreach (var c in Choices)
                        c.IsChecked.Value = false;
                    return;
                }

                foreach (var c in Choices)
                    c.IsChecked.Value = (int)c.Value == (int)value;
            }
        }
    }

    public class EnumChoiceVm : ViewModelBase
    {
        public object Value { get; }
        public ReactiveProperty<bool> IsChecked { get; }
        public ReactiveProperty<bool> IsReadOnly { get; set; }

        public EnumChoiceVm(object value)
        {
            Value = value;
            IsChecked = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    // ReSharper disable ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(Value));
                    // ReSharper restore ExplicitCallerInfoArgument
                })
                .AddTo(CompositeDisposable);
        }
    }

}
