﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Reactive.Linq;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class ApplicationIdParamVm : ParamVm
    {
        public ReadOnlyReactiveCollection<ApplicationIdVm> ApplicationIds { get; set; }

        public int MaxApplicationIdCount { get; }

        public ReactiveCommand AddCommand { get; }
        public ReactiveCommand<ApplicationIdVm> RemoveCommand { get; }

        public string AddIdTag { get; }
        public string AddId =>
            AddIdTag == null ? null : Resources.ResourceManager.GetString(AddIdTag, Resources.Culture);

        public ApplicationIdParamVm(string captionTag, string commentTag, string addIdTag,
            ObservableCollection<NintendoApplicationId> model,
            int maxIdCount,
            ReactiveProperty<bool> isReadOnly)
            : base(captionTag, commentTag, null)
        {
            AddIdTag = addIdTag;
            MaxApplicationIdCount = maxIdCount;
            IsReadOnly = isReadOnly;

            Debug.Assert(model != null);

            RemoveCommand = IsReadOnly.Select(x => x == false)
                    .ToReactiveCommand<ApplicationIdVm>()
                    .AddTo(CompositeDisposable);
            RemoveCommand.Subscribe(id => model.Remove(id.Model))
                .AddTo(CompositeDisposable);

            AddCommand = Observable
                .Merge(model.CollectionChangedAsObservable().ToUnit())
                .Merge(IsReadOnly.ToUnit())
                .Select(_ => IsReadOnly.Value == false && model.Count < maxIdCount)
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);
            AddCommand.Subscribe(_ =>
                {
                    model.Add(new NintendoApplicationId());
                }).AddTo(CompositeDisposable);

            ApplicationIds =
                model.ToReadOnlyReactiveCollection(comm => new ApplicationIdVm(comm, RemoveCommand, IsReadOnly))
                    .AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    RaisePropertyChanged(nameof(AddIdTag));
                })
                .AddTo(CompositeDisposable);
        }
    }

    public class ApplicationIdVm : ViewModelBase
    {
        public ReactiveProperty<ulong> Id { get; }
        public ReactiveProperty<bool> IsReadOnly { get; }

        public ReactiveCommand<ApplicationIdVm> RemoveCommand { get; }
        public NintendoApplicationId Model { get; }

        public ApplicationIdVm(NintendoApplicationId model,
            ReactiveCommand<ApplicationIdVm> removeCommand,
            ReactiveProperty<bool> isReadOnly)
        {
            Debug.Assert(model != null);

            Model = model;
            RemoveCommand = removeCommand;

            Id = model.ToReactivePropertyAsSynchronized(m => m.Id).AddTo(CompositeDisposable);
            IsReadOnly = isReadOnly;
        }
    }
}
